package fc.fp.util.xas;

import java.util.Vector;

/**
 * A factory for instantiating XML serializers and parsers.  The actual
 * encoding and decoding of XML data is decoupled from any application
 * operations so as to better switch among alternate encodings on the
 * network or file system.  It is sensible to implement a
 * <code>CodecFactory</code> as a Singleton object.
 *
 * <p>Most often a <code>CodecFactory</code> object is registered to
 * handle a certain MIME type in the {@link CodecIndustry}, but this
 * is not necessary for usage.  Reasons for doing the registration are
 * given in the {@link CodecIndustry} documentation.
 */
public interface CodecFactory {

    /**
     * Return a new XML parser.  The source of the input will need to
     * be set by the application by calling a <code>setInput</code>
     * method of the returned object.  The token is provided to allow
     * data to be communicated between different parsers; the actual
     * functionality and requirements for the token are defined by
     * each implementing class.
     *
     * @param token a token to enable relations between created
     * parsers
     * @return an XML parser object
     */
    TypedXmlParser getNewDecoder (Object token);

    /**
     * Return a new XML serializer.  Any output target will need to be
     * set by the application with a <code>setOutput</code> method of
     * the returned serializer.  The token is provided to allow data to
     * be communicated between different serializers; the actual
     * functionality and requirements for the token are defined by
     * each implementing class.
     *
     * @param token a token to enable relations between created
     * serializers
     * @return an XML serializer object
     */
    TypedXmlSerializer getNewEncoder (Object token);

    /**
     * Clear any serializer state related to a token.  This method
     * will forget any accumulated state related to a token regarding
     * the effect of {@link #getNewEncoder}.
     *
     * <p>The guarantee is that in the code
     * <pre>
     * CodecFactory factory = new ...
     * TypedXmlSerializer ser = factory.getNewEncoder(token);
     * // Use ser for serialization
     * factory.resetOutState(token);
     * ser = factory.getNewEncoder(token);
     * // Use ser for serialization
     * </pre>
     * the two serialization stages will produce the same output if
     * given the same input.
     *
     * @param token a token identifying the state to reset
     */
    void resetOutState (Object token);

    /**
     * Clear any parser state related to a token.  This method
     * will forget any accumulated state related to a token regarding
     * the effect of {@link #getNewDecoder}.
     *
     * <p>The guarantee is that in the code
     * <pre>
     * CodecFactory factory = new ...
     * XmlPullParser parser = factory.getNewDecoder(token);
     * // Use parser for parsing
     * factory.resetInState(token);
     * parser = factory.getNewDecoder(token);
     * // Use parser for parsing
     * </pre>
     * the two parsing stages will produce the same output if given
     * the same input.
     *
     * @param token a token identifying the state to reset
     */
    void resetInState (Object token);

    /**
     * Register a list of typed content codec factories.  This method
     * gives a list of {@link ContentCodecFactory} objects.  For each
     * new {@link TypedXmlSerializer} or {@link TypedXmlParser}
     * created, the installed property {@link
     * XasUtil#PROPERTY_CONTENT_CODEC} needs to be set to a chained
     * codec that is generated by applying each of the {@link
     * ContentCodecFactory} objects in the supplied list to the
     * default value of this property.
     *
     * <p>Note that a reference to the list is passed instead of the
     * actual objects to permit modifications to happen to the list
     * while it is registered.  Therefore a <code>CodecFactory</code>
     * implementation should only save the passed-in reference and
     * apply the chaining at generation time.
     *
     * @param list the list of {@link ContentCodecFactory} objects
     */
    void installContentFactories (Vector list);

}
// arch-tag: 600cb654090a18e290ba4b319223080f *-
