<?xml version="1.0"?>
<!-- Copyright 1998-2003 W3C (MIT, ERCIM, Keio), All Rights Reserved. See http://www.w3.org/Consortium/Legal/. -->
<!-- Brought to you by... matto@tellme.com on 10/21/2002 -->
<xsl:stylesheet version="1.0" 
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
<xsl:output method="text"/>

<xsl:template match="/" >
  <xsl:apply-templates />
</xsl:template>

<!-- root document element -->
<xsl:template match="ircgi">
  <xsl:call-template name="header" />
  <xsl:apply-templates />
  <xsl:call-template name="footer" />
</xsl:template>

<!-- handle CGI parameter checks -->
<xsl:template match="if-parameter" >
  $val = param("<xsl:value-of select="@name"/>");
  <xsl:call-template name="check-value">
    <xsl:with-param name="value" select="@value"/>
    <xsl:with-param name="starts-with" select="@starts-with"/>
    <xsl:with-param name="ignore-case" select="@ignore-case"/>
  </xsl:call-template>
</xsl:template>

<!-- handle HTTP Request header checks -->
<xsl:template match="if-header">
  $val = $ENV{<xsl:call-template name="map-header"><xsl:with-param name="header" select="@name"/></xsl:call-template>};
  <xsl:call-template name="check-value">
    <xsl:with-param name="value" select="@value"/>
    <xsl:with-param name="starts-with" select="@starts-with"/>
    <xsl:with-param name="ignore-case" select="@ignore-case"/>
  </xsl:call-template>
</xsl:template>

<!-- in Perl, the Request-Method is just another HTTP Request header -->
<xsl:template match="if-method">
 $val = $ENV{REQUEST_METHOD};
  <xsl:call-template name="check-value">
    <xsl:with-param name="value">
      <xsl:choose>
        <xsl:when test="@type"><xsl:value-of select="@type"/></xsl:when>
        <xsl:otherwise>get</xsl:otherwise> <!-- default http request method -->
      </xsl:choose>
    </xsl:with-param>
    <xsl:with-param name="ignore-case" select="'true'"/>
  </xsl:call-template>
</xsl:template>

<!-- strip comment elements -->
<xsl:template match="comment">
  push @comments, qq {<xsl:value-of select="."/>};
</xsl:template>

<!-- handle next elements -->
<xsl:template match="next">
    return {
  <xsl:choose>
    <xsl:when test="not(@code) and not(@dest)">
      status => "200",
    </xsl:when>
  <xsl:otherwise>
    <xsl:if test="@dest">
      next => "<xsl:value-of select="@dest" />", 
      <xsl:if test="@include = 'true'">
        include => 1,
      </xsl:if>
    </xsl:if>
    <xsl:if test="@code">
      status => "<xsl:value-of select="@code" />",
    </xsl:if>    
  </xsl:otherwise>
  </xsl:choose>
  <xsl:if test="@sleep">
    sleep => "<xsl:value-of select="@sleep"/>",    
  </xsl:if>
  <xsl:if test="@expires">
    expires => int(<xsl:value-of select="@expires"/>),
  </xsl:if>
    comments => \@comments};
</xsl:template>

<!-- check the value, if any, and continue processing child elements -->
<xsl:template name="check-value">
<xsl:param name="value"/>
<xsl:param name="starts-with"/>
<xsl:param name="ignore-case"/>
<xsl:choose>
<xsl:when test="$starts-with">
  my $starts = '<xsl:value-of select="$starts-with"/>';
  if (defined($val) &amp;&amp; ($val =~ /^$starts/<xsl:if test="$ignore-case='true'">i</xsl:if>)) {
    <xsl:apply-templates/>
  }
</xsl:when>
<xsl:when test="$value=''">
  if (defined($val) &amp;&amp; ($val =~ /^\s*$/)) {
    <xsl:apply-templates/>
  }
</xsl:when>
<xsl:when test="$value">
  my $match = '<xsl:value-of select="$value"/>';
  if (defined($val) &amp;&amp; ($val =~ /^$match$/<xsl:if test="$ignore-case='true'">i</xsl:if>)) {
    <xsl:apply-templates/>
  }
</xsl:when>
<xsl:otherwise>
  if (defined($val)) {
    <xsl:apply-templates/>
  }
</xsl:otherwise>
</xsl:choose>
</xsl:template>

<xsl:template name="header">#!/usr/local/bin/perl -w
use strict;
use CGI qw(param);
use CGI::Util qw(expires);

# limit sleep time to 1 minute to prevent DOS attack
use constant SLEEP_LIMIT => 60;

# forward decls
sub GetStatusText;
sub Run;
sub JumpTo;
sub GetContentType;

my $rhRetval = Run();
my $next = $rhRetval->{next}; # where to Mr. Magoo?
my $statusCode = $rhRetval->{status};
my $statusText = "unknown status";
my $ctype = GetContentType($next);
my $raComments = $rhRetval->{comments};
my $bInclude = $rhRetval->{include};
my $expires_delta = $rhRetval->{expires};
my $epoch = time;

if (defined($next) &amp;&amp; defined($bInclude) &amp;&amp; 1 == $bInclude)
{
  # restrict paths when allowing source inclusion
  if (($next =~ /^\//) || ($next =~ /\/\.\./))
  {
    $statusCode = 403;
  }
}

my $sleep = $rhRetval->{sleep};
if (defined($sleep))
{
  if (($sleep =~ /^\d+$/) &amp;&amp; ($sleep &lt;= SLEEP_LIMIT))
  {
    sleep $sleep;
  }
  else
  {
    push @$raComments, "Bad sleep interval $sleep";
  }
}

print "Content-Type: $ctype\n";
if (defined($expires_delta))
{
  print ExpiresFromDelta($expires_delta) . "\n";
}
if(defined($statusCode))
{
  $statusText = GetStatusText($statusCode);
  print "Status: $statusCode $statusText\n\n";
}
else
{
  print "\n";
}

if (!defined($next))
{
  print "$statusText\n";
}
else
{
  my $content;
  if ($bInclude)
  {
    $! = 0; # clear i/o errs
    open HINCLUDE, $next;
    if ($! != 0)
    {
      push @$raComments, "Unable to open $next";
      $content = JumpTo($next, $raComments);
      print STDERR "Unable to open $next\n";
    }
	else
	{
	  my $eor = $/;
	  undef $/;
	  $content = &lt;HINCLUDE&gt;;
	  # allow caching tests to be performed by interpolating __EPOCH__
	  $content =~ s/__EPOCH__/$epoch/g;
	  close HINCLUDE;
	  $/ = $eor;  
	}
  }
  else
  {
    $content = JumpTo($next, $raComments);
  }

  print $content;
}

# Return a simple VoiceXML document that navigates 
#   to the URI specified by $next
# Dump the comments in the array $raComments to the call log
sub JumpTo
{
  my($next, $raComments) = @_;

<![CDATA[
  my $content = <<EOF;
<?xml version="1.0"?>
<vxml version="2.0"
  xmlns="http://www.w3.org/2001/vxml"
>
<form>
  <block>
EOF
]]>
foreach my $comment (@$raComments)
{
  $content .= qq{&lt;log>$comment &lt;/log>\n};
}
<![CDATA[
$content .= <<EOF;
    <goto next="$next"/>
  </block>
</form>
</vxml>
EOF
]]>

  $content;
}


# Determine what to do next
# Return a hash containing one or more of the following keys:
#   next - the next document to navigate to 
#   code - the HTTP response code
#   comments - a reference to an array of comments to aid in debugging
sub Run
{
  my $val; # temp var to stash param or header value
  my @comments = (); # array of comments obtained while processing
</xsl:template>

<xsl:template name="footer" >
}

# Map a status code to an informative string
# http://www.w3.org/Protocols/rfc2616/rfc2616-sec6.html#sec6.1.1
sub GetStatusText
{
  my($code) = @_;

  my $rhCodes = {100 => "Continue",
  101 => "Switching Protocols",
  200 => "OK",
  201 => "Created",
  202 => "Accepted",
  203 => "Non-Authoritative Information",
  204 => "No Content",
  205 => "Reset Content",
  206 => "Partial Content",
  300 => "Multiple Choices",
  301 => "Moved Permanently",
  302 => "Found",
  303 => "See Other",
  304 => "Not Modified",
  305 => "Use Proxy",
  307 => "Temporary Redirect",
  400 => "Bad Request",
  401 => "Unauthorized",
  402 => "Payment Required",
  403 => "Forbidden",
  404 => "Not Found",
  405 => "Method Not Allowed",
  406 => "Not Acceptable",
  407 => "Proxy Authentication Required",
  408 => "Request Time-out",
  409 => "Conflict",
  410 => "Gone",
  411 => "Length Required",
  412 => "Precondition Failed",
  413 => "Request Entity Too Large",
  414 => "Request-URI Too Large",
  415 => "Unsupported Media Type",
  416 => "Requested range not satisfiable",
  417 => "Expectation Failed",
  500 => "Internal Server Error",
  501 => "Not Implemented",
  502 => "Bad Gateway",
  503 => "Service Unavailable",
  504 => "Gateway Time-out",
  505 => "HTTP Version not supported extension-code"};

  return (exists($rhCodes->{$code}) ? $rhCodes->{$code} : "invalid status code");
}

sub GetContentType
{
  my($next) = @_;

  my $ctype = "text/plain";
  if (defined($next))
  {
    my $rhTypes = {'txml' => 'text/xml', 'vxml' => 'text/xml', 
      'xml' => 'text/xml', 'srgs' => 'text/xml'};
    my @parts = split /\./, $next;    
    my $ext = $parts[0];
    if (exists($rhTypes->{$ext}))
    {
      $ctype = $rhTypes->{$ext};
    }    
  }
  
  $ctype;
}

# return an expires header given seconds since epoch
sub ExpiresFromDelta
{
  my($delta) = @_;
  $delta = (($delta >= 0 &amp;&amp; $delta !~ /^\+/) ? "+" : "") . $delta . "s";
  "Expires: " . expires($delta);
}
</xsl:template>

<!-- 
  the headers we're willing to expose.
  allowing arbitrary header requests is a security risk
-->
<xsl:template name="map-header">
<xsl:param name="header"/>
<xsl:choose>
  <xsl:when test="$header = 'User-Agent'">HTTP_USER_AGENT</xsl:when>
  <xsl:when test="$header = 'Request-Method'">REQUEST_METHOD</xsl:when>
  <xsl:when test="$header = 'Content-Type'">CONTENT_TYPE</xsl:when>
  <xsl:otherwise>__UNKNOWN__<xsl:value-of select="$header"/></xsl:otherwise>
</xsl:choose>
</xsl:template>

</xsl:stylesheet>
