/*
 * Copyright (c) 2002 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 */

package org.w3c.dom.validation;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Attr;

/**
 * This interface extends the <code>Element</code> interface with additional 
 * methods for guided document editing. An object implementing this 
 * interface must also implement NodeEditVAL interface.
 * <p>See also the <a href='http://www.w3.org/TR/2002/WD-DOM-Level-3-Val-20021008'>Document Object Model (DOM) Level 3 Validation Specification</a>.
 */
public interface ElementEditVAL extends NodeEditVAL {
    /**
     * The list of all element nodes defined in non-namespace aware grammar or 
     * list of all element nodes belonging to the particular namespace. 
     * These are not nodes from the instance document, but rather are new 
     * nodes that could be inserted in the document. 
     */
    public NodeList getDefinedElementTypes();

    /**
     * Determines element content type.
     * @return Constant for one of <code>EMPTY_CONTENTTYPE</code>, 
     *   <code>ANY_CONTENTTYPE</code>, <code>MIXED_CONTENTTYPE</code>, 
     *   <code>ELEMENTS_CONTENTTYPE</code>.
     */
    public short contentType();

    /**
     * Determines if the value for specified attribute can be set.
     * @param attrname Name of attribute.
     * @param attrval Value to be assigned to the attribute.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canSetAttribute(String attrname, 
                                   String attrval);

    /**
     * Determines if an attribute node can be added with respect to the 
     * validity check level.
     * @param attrNode <code>Node</code> in which the attribute can possibly 
     *   be set.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canSetAttributeNode(Attr attrNode);

    /**
     * Determines if the attribute with given namespace and qualified name can 
     * be created if not already present in the attribute list of the 
     * element. If the attribute with the same qualified name and 
     * namespaceURI is already present in the element's attribute list, it 
     * tests whether the value of the attribute and its prefix can be set to 
     * the new value. See DOM core <code>setAttributeNS</code>.
     * @param namespaceURI <code>namespaceURI</code> of namespace.
     * @param qualifiedName Qualified name of attribute.
     * @param value Value to be assigned to the attribute.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canSetAttributeNS(String namespaceURI, 
                                     String qualifiedName, 
                                     String value);

    /**
     * Verifies if an attribute by the given name can be removed.
     * @param attrname Name of attribute.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canRemoveAttribute(String attrname);

    /**
     * Verifies if an attribute by the given local name and namespace can be 
     * removed.
     * @param namespaceURI The namespace URI of the attribute to remove.
     * @param localName Local name of the attribute to be removed.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canRemoveAttributeNS(String namespaceURI, 
                                        String localName);

    /**
     * Determines if an attribute node can be removed.
     * @param attrNode The <code>Attr</code> node to remove from the 
     *   attribute list.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canRemoveAttributeNode(Node attrNode);

    /**
     * Returns a <code>NodeList</code> containing the possible 
     * <code>Element</code> nodes that can appear as children of this type 
     * of element, with certain conditions as specified below. These are not 
     * nodes from the instance document, but rather are new nodes that could 
     * be inserted in the document. 
     * @return List of possible children element types of this element. Note 
     *   that if no context of this element exists, then a <code>NULL</code> 
     *   is returned; an empty list is returned if the element is not in the 
     *   document tree.
     */
    public NodeList getChildElements();

    /**
     * Returns a <code>NodeList</code> containing the possible 
     * <code>Element</code> nodes that can appear as a parent of this type 
     * of element, with certain conditions as specified below. These are not 
     * nodes from the instance document, but rather are new nodes that could 
     * be inserted in the document.
     * @return List of possible parent element types of this element. Note 
     *   that if no context of this element exists, for example, the parent 
     *   element of this element, then a <code>NULL</code> is returned; an 
     *   empty list is returned if the element is not in the document tree.
     */
    public NodeList getParentElements();

    /**
     * Returns a <code>NodeList</code> containing all the possible 
     * <code>Attr</code>s that can appear with this type of element. These 
     * are not nodes from the instance document, but rather are new nodes 
     * that could be inserted in the document. 
     * @return List of possible attributes of this element.
     */
    public NodeList getAttributeList();

    /**
     * Determines if <code>name</code> is defined in the currently active 
     * grammar.
     * @param name Name of element.
     * @return A boolean that is <code>true</code> if the element is defined, 
     *   false otherwise.
     */
    public boolean isElementDefined(String name);

    /**
     * Determines if <code>name</code> in this namespace is defined in the 
     * currently active grammar.
     * @param name Name of element.
     * @param namespaceURI <code>namespaceURI</code> of namespace.
     * @return A boolean that is <code>true</code> if the element is defined, 
     *   false otherwise.
     */
    public boolean isElementDefinedNS(String name, 
                                      String namespaceURI);

}
