/**
 * Copyright  Sergey Melnik (Stanford University, Database Group) 
 *
 * Distribution policies are governed by the W3C software license.
 * http://www.w3.org/Consortium/Legal/copyright-software   
 * 
 * All Rights Reserved.
 * 
 * @author      Sergey Melnik <melnik@db.stanford.edu>
 */

package org.w3c.rdf.implementation.model;

import java.security.*;
import java.util.*;

import org.w3c.rdf.model.*;
import org.w3c.rdf.tools.crypt.*;
import org.w3c.rdf.vocabulary.rdf_syntax_19990222.RDF;

public class NodeFactoryImpl implements NodeFactory {

  private static Random rnd;

  private static final int MAX_ORD = 30; // maximal number of cached ordinals
  private static Resource[] ords;
  static {
    ords = new Resource[MAX_ORD + 1]; // 0 is wasted
    for(int i=1; i <= MAX_ORD; i++)
      ords[i] = new ResourceImpl(RDF._Namespace + "_" + i);
  }

  /**
   * Creates a resource out of a string
   */
  public Resource createResource(String str) {

    return str != null ? new ResourceImpl(str) : null;
  }

  /**
   * Creates a literal out of a string
   */
  public Literal createLiteral(String str) {

    return str != null ? new LiteralImpl(str) : null;
  }

  /**
   * Creates a triple
   */
  public Statement createStatement(Resource subject, Resource predicate, RDFNode object) {
		
    return (subject != null && predicate != null && object != null) ?
			new StatementImpl(subject, predicate, object) : null;
  }

	/**
	 * Creates a resource with a unique ID
	 */
	public Resource createUniqueResource() {

    if(rnd == null) {
      rnd = new SecureRandom();
    }
    byte[] b = new byte[16];
    rnd.nextBytes(b);
    return new ResourceImpl("urn:rdf:" + DigestUtil.toHexString(b));
  }

  /**
   * Creates an ordinal property (rdf:li, rdf:_N)
   */
  public Resource createOrdinal(int i) {
    
    if(i < 1)
      throw new IllegalArgumentException("Attempt to construct invalid ordinal resource");
    if(i <= MAX_ORD)
      return ords[i];
    else
      return new ResourceImpl(RDF._Namespace + "_" + i);
  }

}
