/*
 * Basically, a multi-tier data structure to support finds.
 *
 * This is memory inefficient currently because most of the functionality
 *   is aggregated to standard container classes.
 *
 * Copyright  William Grosso (Stanford University, Database Group)
 *
 * Distribution policies are governed by the W3C software license.
 * http://www.w3.org/Consortium/Legal/copyright-software   
 *
 * All Rights Reserved.
 *
 * Written by William Grosso <grosso@acm.org>
*/

package org.w3c.rdf.implementation.model;
import java.util.*;
import org.w3c.rdf.model.*;
import org.w3c.rdf.util.RDFDigestUtil;
import org.w3c.rdf.tools.crypt.*;

public class FindIndex
{
 private HashMap _indexesToTreemapsOfStatements;
 private StatementComparator _statementComparator;
 private int _totalSize = 0;
 public FindIndex()
 {
  _indexesToTreemapsOfStatements = new HashMap();
  _statementComparator = new StatementComparator();
 }

 public int size()
 {
  return _totalSize;
 }

 public void addLookup(Statement t) throws ModelException
 {
  Resource s = t.subject();
  Resource p = t.predicate();
  RDFNode o = t.object();
    
  put(getLookupValue(null, null, o),    t);
  put(getLookupValue(null, p,    null), t);
  put(getLookupValue(null, p,    o),    t);
  put(getLookupValue(s,    null, null), t);
  put(getLookupValue(s,    null, o),    t);
  put(getLookupValue(s,    p,    null), t);
  put(getLookupValue(s,    p,    o),    t);
 }

 public void removeLookup(Statement t) throws ModelException
 {
  Resource s = t.subject();
  Resource p = t.predicate();
  RDFNode o = t.object();

  remove(getLookupValue(null, null, o),    t);
  remove(getLookupValue(null, p,    null), t);
  remove(getLookupValue(null, p,    o),    t);
  remove(getLookupValue(s,    null, null), t);
  remove(getLookupValue(s,    null, o),    t);
  remove(getLookupValue(s,    p,    null), t);
  remove(getLookupValue(s,    p,    o),    t);
 }

 public Enumeration multiget(Resource subject, Resource property, RDFNode object) throws ModelException
 {
  Integer key = getLookupValue(subject, property, object);
  return new FindIndexEnumeration(key);
 }

 private void put(Integer index, Statement value) throws ModelException
 {
  TreeMap underlyingTreeMap = (TreeMap) _indexesToTreemapsOfStatements.get(index);
  if (null==underlyingTreeMap)
  {
   underlyingTreeMap = new TreeMap(_statementComparator);
   _indexesToTreemapsOfStatements.put(index, underlyingTreeMap);
  }
  StatementWrapper wrapper = new StatementWrapper(index, value);
  underlyingTreeMap.put(wrapper, wrapper);
  _totalSize++;
 }

 private void remove(Integer index, Statement value) throws ModelException
 {
  TreeMap underlyingTreeMap = (TreeMap) _indexesToTreemapsOfStatements.get(index);
  if (null==underlyingTreeMap)
  {
   return;
  }
  StatementWrapper wrapper = new StatementWrapper(index, value);
  underlyingTreeMap.remove(wrapper);
  _totalSize--;
 }

 private class FindIndexEnumeration implements Enumeration
 {
  private Statement _currentStatement;
  private Iterator _treemapIterator;
  private Integer _enumerationIndex;

  public FindIndexEnumeration(Integer indexValue)
  {
   _enumerationIndex = indexValue;
   TreeMap underlyingTreeMap = (TreeMap) _indexesToTreemapsOfStatements.get(_enumerationIndex);
   if (null==underlyingTreeMap)
   {
    return;
   }
   _treemapIterator = (underlyingTreeMap.values()).iterator();
   findNext();
  }
  
  public boolean hasMoreElements()
  {
   return (null!=_currentStatement);
  }

  public Object nextElement()
  {
   Statement returnValue = _currentStatement;
   if (null!=_currentStatement)
   {
    findNext();
   }
   return returnValue;
  }

  private void findNext()
  {
   while(_treemapIterator.hasNext())
   {
    StatementWrapper nextObject = (StatementWrapper) _treemapIterator.next();
    if (_enumerationIndex.equals(nextObject.index))
    {
     _currentStatement = nextObject.value;
     return;
    }
   }
   _currentStatement = null;
  }
 }

 private Integer getLookupValue(Resource subject, Resource predicate, RDFNode object)
 {
  int s1 = subject != null ? subject.hashCode() : 0;
  int s2 = predicate != null ? predicate.hashCode() : 0;
  int s3 = object != null ? object.hashCode() : 0;
  int l = (s1 * 37 + s2) * 37 + s3;
  return new Integer(l);
 }

 private class StatementComparator implements Comparator
 {
  public int compare(Object o1, Object o2)
  {
   int firstHash = o1.hashCode();
   int secondHash = o2.hashCode();
   if (firstHash < secondHash)
   {
    return -1;
   }
   if (firstHash > secondHash)
   {
    return 1;
   }
   return 0;   
  }

  public boolean equals(Object object)
  {
   if (object instanceof StatementComparator)
   {
    return true;
   }
   return false;
  }
 }

 private class StatementWrapper
 {
  public Integer index;
  public Statement value;

  public StatementWrapper(Integer index, Statement value)
  {
   this.index = index;
   this.value = value;
  }

  public boolean equals(Object object)
  {
   if (false==(object instanceof StatementWrapper))
   {
    return false;
   }
   StatementWrapper statementWrapper = (StatementWrapper) object;
   if (false == index.equals(statementWrapper.index))
   {
    return false;
   }
   return value.equals(statementWrapper.value); 
  }

  public int hashCode()
  {
   return value.hashCode();
  }
 }
}
