/**
 * DumpConsumer for the SiRPACServlet
 *
 * Copyright  World Wide Web Consortium, (Massachusetts Institute of
 * Technology, Institut National de Recherche en Informatique et en
 * Automatique, Keio University).
 *
 * All Rights Reserved.
 *
 * Please see the full Copyright clause at
 * <http://www.w3.org/Consortium/Legal/copyright-software.html>
 *
 * @author Arthur Barstow <barstow@w3.org>
 */
package org.w3c.rdf.examples;

import java.io.*;
import org.w3c.rdf.model.*;
import org.w3c.rdf.syntax.*;
import org.w3c.rdf.util.xml.DumpConsumer;
import org.w3c.rdf.implementation.model.NodeFactoryImpl;
import javax.servlet.ServletOutputStream;

public class SiRPACServletDumpConsumer extends DumpConsumer implements RDFConsumer {
    private NodeFactory         m_f = new NodeFactoryImpl();
    private ServletOutputStream m_out = null;
    private int                 m_numStatements = 0;
    private boolean             m_useNTriples = false;

    public void startModel () {}
    public void endModel () {}

    /*
     * Set the [Sevelet]OutputStream
     *
     *@param out - the stream
     */
    public void setOutputStream (ServletOutputStream out) {
        m_out = out;
    }

    /*
     * Set the output format to use the n-triples syntax
     *
     *@param flag - the switch
     */
    public void setNTriplesOutput (boolean flag) {
        m_useNTriples = flag;
    }

    public NodeFactory getNodeFactory() {
        return m_f;
    }

    public int getNumStatements() {
        return m_numStatements;
    }

    /* 
     * @@@@ - if string begins with http://, https://  or
     *   mailto:,  make it s a link
     */
    private String addAnchor(String s) {
      if (s.startsWith("http://")  || 
          s.startsWith("https://") ||
          s.startsWith("mailto:"))
        return "<a href=\"" + s + "\">" + s + "</a>";

      return s;
    }

    /*
     * Generate the NTriple syntax for the given statement
     *
     *@param s - the statement
     */
    private String generateNTripleStatment(Statement s) throws Exception
    {
      try {
        // Handle the subject first
        String sub = s.subject().toString();

        // The subject may be annon node
        sub = NTriplesDumpConsumer.convertGenId(sub, false);
        if (sub.equals(s.subject().toString()))
            // Not a genid/annon node
            sub = "&lt;" + sub + "&gt;";

        // Now handle the object
        String obj = s.object().toString();
        String tmpObjectName;
        if (s.object() instanceof Literal) 
            tmpObjectName = "\"" + obj + "\"";
        else {
            // Assume it is a Resource and check it for a genid
            tmpObjectName = NTriplesDumpConsumer.convertGenId(obj, false);
            if (tmpObjectName.equals(s.object().toString()))
                // Not a genid/annon node
                tmpObjectName = "&lt;" + tmpObjectName + "&gt;";
        }
        
        return new String(sub + " &lt;" + s.predicate().toString() + "&gt; " + tmpObjectName + " . <br />");

     } catch (Exception e) {
         e.printStackTrace ();
         return "An error occured during processing ...";
     }
    }

    public void addStatement (Statement s) {

        try {
            m_numStatements++;

            /* Each statement will be a row in a table */
            String st;

            if (m_useNTriples) {
                st = generateNTripleStatment(s);
            } else {
                /*
                 * If the object contains XML markup characters, must convert 
                 * them to entity chars.
                 */
                String tmpObjectName = new String(s.object().toString());

                if (tmpObjectName != null) {
                    if (s.object() instanceof Literal) {
                        tmpObjectName = SiRPACServlet.replaceString(tmpObjectName, "<", "&lt;");
                        tmpObjectName = SiRPACServlet.replaceString(tmpObjectName, ">", "&gt;");
                    } else {
                        tmpObjectName = addAnchor(s.object().toString());
                    }
                } else {
                    tmpObjectName = " ";
                }

                st = "<tr>" + 
                     "<td>" + m_numStatements + "</td>" +
                     "<td>" + addAnchor(s.subject().toString())   + "</td>" +
                     "<td>" + addAnchor(s.predicate().toString()) + "</td>" +
                     "<td>" + tmpObjectName + "</td>" +
                     "</tr>";
            }

            if (m_out == null)
                System.out.print (st);
            else
                m_out.print (st);

        } catch (Exception e) {
            e.printStackTrace ();
            return;
        }
    }
}
