/**
 * Copyright  World Wide Web Consortium, (Massachusetts Institute of
 * Technology, Institut National de Recherche en Informatique et en
 * Automatique, Keio University).
 *
 * All Rights Reserved.
 *
 * Please see the full Copyright clause at
 * <http://www.w3.org/Consortium/Legal/copyright-software.html>
 *
 * This class is a RDFConsumer that outputs RDF triples in the
 * n-triples format.  This format is documented (loosely) at:
 *
 *  http://lists.w3.org/Archives/Public/w3c-rdfcore-wg/2001Jun/0095.html
 *
 * Usage from command line:
 *
 *   java org.w3c.rdf.examples.NTriplesDumpConsumer input_file.rdf > output_file.nt
 * 
 * @author Art Barstow (barstow@w3.org)
 * @version 1.0
 */

package org.w3c.rdf.examples;

import java.util.*;
import java.io.PrintWriter;
import org.w3c.rdf.model.*;
import org.w3c.rdf.syntax.*;
import org.w3c.rdf.util.xml.DumpConsumer;
import org.w3c.rdf.implementation.model.StatementImpl;
import org.w3c.rdf.implementation.model.NodeFactoryImpl;

// The following are needed by main() for testing
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.w3c.rdf.util.xml.GenericParser;
import org.w3c.rdf.util.xml.ErrorStore;
import org.w3c.rdf.implementation.syntax.sirpac.SiRPAC;

public class NTriplesDumpConsumer extends DumpConsumer implements RDFConsumer {
      
  NodeFactory mFactory = new NodeFactoryImpl();

  /**
   * Default constructor
   */
  public NTriplesDumpConsumer () {
  }

  /**
   * This method handles the NTriples header.  Put any special NTriples 
   * commands here.
   */
  public void startModel () {
  }

  /**
   * This method handles the NTriples footer.
   */
  public void endModel () {
  }

  public NodeFactory getNodeFactory() {
    return mFactory;
  }

  /**
   * If the given string looks like a generated node id (aka
   * a annonymous node):
   *
   *   http://foo.com/#genid<n>
   *
   * repace it with a string of the following form:
   * 
   *   _:genid<n>
   *
   * @param s the input string
   */
  public static String convertGenId(String s, boolean addbrackets) {
    int i = s.indexOf("#genid");
    if (i < 0) 
        i = s.indexOf(":genid");
    String tmp = s;
    if (i < 0) {
        if (addbrackets)
            tmp = new String ("<" + s + ">");
    } else {
        tmp = new String("_:" + s.substring(i+1));
    }

    return tmp;
  }

  /**
   * This method outputs text to stdout that is
   * suitable for use in NTriples.
   * 
   * @param s
   */
  public void addStatement (Statement s) {
    try
    {
        String subjectName = s.subject().getLabel();   
        String predicateName = s.predicate().getLabel();
        String objectName = s.object().getLabel();

        /*
         * If the object is a literal, embed it in quotes
         */
        String tmpObjectName = objectName;
        if (s.object() instanceof Literal) 
            tmpObjectName = "\"" + tmpObjectName + "\"";
        else
            // Assume it is a Resource and check it for a genid
            tmpObjectName = convertGenId(objectName, true);

        subjectName = convertGenId(subjectName, true);

        System.out.println(subjectName + " " + 
                           "<" + predicateName + ">" + " " +
                           tmpObjectName + " .");

    } catch( ModelException me ) {
        System.err.println( "An attempt to create the NTriples failed. " + me );
        System.exit(0);
    }    
  }

  /**
   * main method for debugging the NTriples output generator
   *
   * Input is a URI
   *
   * Output (to stdout) is the input for the DOT program
   */
  public static void main (String args[]) throws Exception {

    SiRPAC parser = new SiRPAC ();

    if (args == null) {
        System.err.println("USAGE: must enter a URI");
        System.exit(1);
    }

    NTriplesDumpConsumer consumer = new NTriplesDumpConsumer();
    InputSource source  = GenericParser.getInputSource(args[0]);
    ErrorStore handler = new ErrorStore();

    parser.setErrorHandler(handler);

    try {
      parser.parse(source, consumer);
    } catch (SAXException se) {
      System.err.println("Error during parsing: " + se.toString());
      String sErrors = handler.errors ();
      if (sErrors != null && sErrors.length() > 0)
          System.err.println ("Parsing errors:\n"+sErrors);
      se.getException().printStackTrace(System.err);
    }
  }
}
