/**
 * Copyright  Sergey Melnik (Stanford University, Database Group) 
 *
 * Distribution policies are governed by the W3C software license.
 * http://www.w3.org/Consortium/Legal/copyright-software   
 * 
 * All Rights Reserved.
 * 
 * @author      Sergey Melnik <melnik@db.stanford.edu>
 */

package org.w3c.rdf.util.xml;

import org.w3c.rdf.model.*;
import org.w3c.rdf.implementation.model.StatementImpl;
import org.w3c.rdf.syntax.*;
import org.xml.sax.*;
import org.xml.sax.helpers.*;

import java.util.*;
import java.io.*;
import java.net.*;

public class GenericParser implements EntityResolver, DTDHandler, DocumentHandler, RDFParser {

  static final public String REVISION = "Generic RDF/XML parser v0.1 1999-12-15";
  static final String XMLNS = "xmlns";
  static final String XMLNS_COLON = XMLNS + ":";

  protected ErrorHandler errorHandler = null;
  protected Locator locator = new FakeLocator();
  protected Stack namespaceStack;
  protected InputSource source;
  protected Element current;

  protected NodeFactory nodeFactory;
  protected RDFConsumer consumer;

  public GenericParser () {

    this(true);
  }

  public GenericParser (boolean warn) {

    initXMLParser(warn);
  }

  protected void initXMLParser(boolean warn) {

    if(System.getProperty("org.xml.sax.parser") == null) {
      if(warn)
	System.err.println("Warning: using the default XML parser AElfred (com.microstar.xml.SAXDriver).\n" +
			   "Override for IBM xml4j is: java -Dorg.xml.sax.parser=com.ibm.xml.parser.SAXDriver org.w3c.rdf.SiRPAC");
      try {
	System.getProperties().put("org.xml.sax.parser", "com.microstar.xml.SAXDriver");
      } catch (Exception any) {
	if(warn)
	  System.err.println("Warning: could not set default parser");
      }
    }
  }

  /**
   * Generate a warning message as a string
   */
  public void addWarning (String sMsg) throws SAXException {
    errorHandler.warning(new SAXParseException(sMsg, locator));
  }
  
  /**
   * Generate an error message as a string
   */
  public void addError (String sMsg) throws SAXException {
    errorHandler.error(new SAXParseException(sMsg, locator));
  }
  
  public void setDocumentLocator (Locator locator) {

    this.locator = locator;
  }
  
  public void startDocument () {
    
    namespaceStack = new Stack();
    Hashtable topNS = new Hashtable();
    topNS.put("xml", "xml:");
    /*
    String uri = getSourceURI();
    if(uri != null) {
      if(!uri.endsWith("#") && !uri.endsWith("/"))
	uri += "#";
      topNS.put(XMLNS, uri);
    }
    */
    namespaceStack.push(topNS);
  }
  
  public void endDocument () throws SAXException {
  }
  
  public void doctype (String name, String publicID, String systemID) {
  }
  
  public void ignorableWhitespace (char ch[], int start, int length) {
  }
  
  public void processingInstruction (String target, String data) {
  }
  
  public InputSource resolveEntity (String publicId, String systemId) {
    return null;
  }

  public void notationDecl (String name, String publicId, String systemId) {
  }

  public void unparsedEntityDecl (String name,
				  String publicId,
				  String systemId,
				  String notationName) {
  }

  public void setErrorHandler (ErrorHandler handler) {
    this.errorHandler = handler;
  }

  void addNamespace(String ns) {
  }

  protected Hashtable getNamespaces() {

    return (Hashtable)namespaceStack.peek();
  }

  /*
  String namespace(String abb) {

    return (String)getNamespaces().get(abb);
  }
  */

  protected String getQualifiedName(String nsAbb, String name) throws SAXException {

    Hashtable namespaces = getNamespaces();
    String ns = (String)namespaces.get(nsAbb);
    if(ns == null)
      if(nsAbb == XMLNS)
	return name;
      else
	throw new SAXException("Undeclared namespace qualifier: " + nsAbb);
    else
      return ns + name;
  }

  protected String getQualifiedName(String colonName) throws SAXException {

    // create element
    int cp = colonName.indexOf(':');
    if(cp < 0)
      return getQualifiedName(XMLNS, colonName);
    else
      return getQualifiedName(colonName.substring(0, cp), colonName.substring(cp + 1));
  }

  /**
   * creates a new namespace declaration if needed
   */
  protected Hashtable updateNamespaceStack(Hashtable namespaces, String aName, String aValue) {

    if(namespaces == getNamespaces())
      namespaces = (Hashtable)getNamespaces().clone();

    namespaces.put(aName, aValue);
    addNamespace(aValue);

    return namespaces;
  }

  protected String getSourceURI() {

    return source.getSystemId();
  }

  public void startElement (String name, AttributeList al) throws SAXException {

    _startElement(name, al);
    System.out.println("Start element: " + current.getName());
  }

  protected void _startElement (String name, AttributeList al) throws SAXException {

    Hashtable namespaces = getNamespaces();

    for (int x = 0; x < al.getLength(); x++) {

      String aName = al.getName (x);
      String aValue = al.getValue (aName);
      if (aValue != null && aValue.length() == 0 && getSourceURI() != null)
	aValue = getSourceURI();

      if (aName.equals (XMLNS)) {

	namespaces = updateNamespaceStack(namespaces, XMLNS, aValue);

      } else if (aName.startsWith (XMLNS_COLON)) {

	aName = aName.substring (XMLNS_COLON.length());
	namespaces = updateNamespaceStack(namespaces, aName, aValue);
      }
    }

    // push namespaces on stack
    namespaceStack.push (namespaces);

    // determine tag namespace prefix
    int cp = name.indexOf(':');
    String tagNS = cp > 0 ? name.substring(0, cp + 1) : null;

    name = getQualifiedName(name);

    Element el = createElement();
    el.setName(name);

    // add attributes
    for (int x = 0; x < al.getLength(); x++) {
      String aName = al.getName (x);
      if(!aName.startsWith(XMLNS)) {
	String aValue = al.getValue (aName);
	if(tagNS != null && aName.indexOf(':') < 0)
	  aName = tagNS + aName;
	el.setAttribute(getQualifiedName(aName), aValue);
      }
    }

    if(current == null)
      current = el;
    else {
      current.setChild(el);
      el.setParent(current);
      current = el;
    }

  }

  protected Element createElement() {
    return new Element();
  }

  public void endElement (String name) throws SAXException {

    System.out.println("End element: " + current.getName());
    _endElement(name);
  }

  protected void _endElement (String name) throws SAXException {

    //    name = getQualifiedName(name);
    namespaceStack.pop();
    current = current.getParent();
  }

  protected boolean preserveWhiteSpace() {
    return false;
  }

  public void characters (char ch[], int start, int length)
    throws SAXException {
  
      String s = new String (ch, start, length);
      String sTrimmed = s.trim();
      if (sTrimmed.length() > 0 || preserveWhiteSpace()) {
	System.out.println("Adding characters: \"" + s + "\"");
      }
  }

  protected void createStatement(Resource subject, Resource predicate, RDFNode object) throws ModelException {
    consumer.addStatement(nodeFactory.createStatement(subject, predicate, object));
  }

  public static Parser createParser (String className) {

    try {
      // Get the named class.
      Class c = Class.forName(className);
      // Instantiate the parser.
      return (Parser)(c.newInstance());
    } catch (Exception any) {
      throw new RuntimeException("Could not instantiate XML parser: " + any);
    }
  }    

  public void parse(InputSource source, RDFConsumer consumer) throws SAXException {

    this.source = source;
    this.consumer = consumer;

    try {
      this.nodeFactory = consumer.getNodeFactory();
      consumer.startModel();

      Parser p = ParserFactory.makeParser();

      // Register the handlers
      p.setEntityResolver(this);
      p.setDTDHandler (this);
      p.setDocumentHandler(this);
      p.setErrorHandler (errorHandler);

      p.parse ( source );

      consumer.endModel();

    } catch (Exception any) {
      if(any instanceof SAXException)
      	throw (SAXException)any;
      else
      	throw new SAXException("Fatal error", any);
    }
  }

  public static InputSource getInputSource(String urlStr) throws MalformedURLException, IOException {

    InputStream in = null;
    URL url = null;

    if(urlStr != null) {

      try {
	url = new URL (urlStr);
      } catch (Exception e) {
	url = new URL ("file", null, urlStr);
      }
      in = url.openStream();
      
    } else {
      // read from stdin
      in = System.in;
    }
    
    Reader r = new BufferedReader(new InputStreamReader(in));
    InputSource source = new InputSource(r);
    if(urlStr != null)
      source.setSystemId(url.toString());

    return source;
  }

  protected static void _main(String url, RDFParser parser) throws IOException, MalformedURLException {

    InputSource source = getInputSource(url);
    RDFConsumer consumer = new DumpConsumer();
    ErrorStore handler = new ErrorStore();

    parser.setErrorHandler(handler);

    try {
      parser.parse(source, consumer);
    } catch (SAXException se) {
      System.err.println("Error during parsing: " + se);
      se.getException().printStackTrace(System.err);
    }
    
    String sErrors = handler.errors ();
    if (sErrors != null && sErrors.length() > 0)
      System.err.println ("Errors during parsing:\n"+sErrors);
  }

  public static void main (String args[]) throws Exception {

    if(args.length != 1) {
      System.err.println("Usage: java -Dorg.xml.sax.parser=<classname> org.w3c.rdf.syntax.xml.GenericParser " +
			 //			 "[-fetch_schemas | -f] [-stream | -s] [-robust | -r] [ URI | filename ]");
			 "[ URI | filename ]");
      System.err.println ("This is revision " + REVISION);
      System.exit(1);
    }

    _main(args[0], new GenericParser());
  }

}
