/**
 * Copyright  Sergey Melnik (Stanford University, Database Group) 
 *
 * Distribution policies are governed by the W3C software license.
 * http://www.w3.org/Consortium/Legal/copyright-software   
 * 
 * All Rights Reserved.
 * 
 * @author      Sergey Melnik <melnik@db.stanford.edu>
 */

package org.w3c.rdf.util;

import org.w3c.rdf.model.*;
import org.w3c.rdf.syntax.*;
import org.w3c.rdf.vocabulary.rdf_syntax_19990222.*;
import org.w3c.rdf.vocabulary.rdf_schema_19990303.*;

import org.w3c.rdf.implementation.model.ResourceImpl;
import org.w3c.rdf.util.SetOperations;
import org.w3c.rdf.tools.crypt.*;
import org.xml.sax.*;

import java.util.*;
import java.io.*;
import java.net.*;

/**
 * Useful utility methods.
 *
 * @author Sergey Melnik <melnik@db.stanford.edu>
 */

public class RDFUtil {

  /**
   * Creates a new unique unnamed resource.
   */
  public static Resource noname(Model m) throws ModelException {

    return m.getNodeFactory().createUniqueResource();
  }

  /**
   * Converts an ordinal property to an integer.
   */
  public static int getOrd(Resource r)  throws ModelException {

    if(r == null)
      return -1;

    String uri = r.toString();
    if(!isRDF(uri))
      return -1;

    int pos = getNamespaceEnd(uri);
    if(pos > 0 && pos + 1 < uri.length()) {
      try {
        int n = Integer.parseInt(uri.substring(pos + 1));
        if(n >= 1)
          return n;
      } catch (Exception any) {}
    }
    return -1;
  }

  /**
   * Tests if the URI is qualified, i.e. has a namespace prefix.
   */
  public static boolean isQualified(String s) {
    
    int l = s.length()-1;
    do {
      char c = s.charAt(l);
      if(c == '#' || c == ':')
        return true;
      l--;
    } while (l >= 0);
    return false;
  }

  /**
   * Extracts the namespace prefix out of a URI.
   */
  public static String guessNamespace(String uri) {

    int l = getNamespaceEnd(uri);
    return l > 1 ? uri.substring(0, l /*-1*/) : "";
  }

  /**
   * Delivers the name out of the URI (without the namespace prefix).
   */
  public static String guessName(String uri) {

    return uri.substring(getNamespaceEnd(uri));
  }

  /**
   * Tests if the URI belongs to the RDF syntax/model namespace.
   */
  public static boolean isRDF(String uri) {
    return uri != null && uri.startsWith(RDF._Namespace);
  }

  /**
   * Tests if the resource belongs to the RDF syntax/model namespace.
   */
  public static boolean isRDF(Resource r) throws ModelException {
    return isRDF(r.getURI());
  }

  /**
   * Position of the namespace end
   */
  static int getNamespaceEnd(String uri) {

    int l = uri.length()-1;
    do {
      char c = uri.charAt(l);
      if(c == '#' || c == ':' || c == '/')
        break;
      l--;
    } while (l >= 0);
    l++;
    return l;
  }

  /**
   * Returns the first triple of the model
   */
  public static Statement get1(Model m) throws ModelException {
    if(m == null || m.isEmpty())
      return null;

    return (Statement)m.elements().nextElement();
  }

  public static void add(Model m, Resource subject, Resource predicate, RDFNode object) throws ModelException {

    m.add(m.getNodeFactory().createStatement(subject, predicate, object));
  }

  /**
   * returns true if old triples from r were removed
   */
  public static boolean setUniqueObject(Model r, Resource subject, Resource predicate, RDFNode object) throws ModelException {

    Model old = r.find(subject, predicate, null);
    SetOperations.subtract(r, old);
    r.add(r.getNodeFactory().createStatement(subject, predicate, object));
    return !old.isEmpty();
  }

  /**
   * returns the literal value of the node reachable from subject via predicate
   */
  public static String getObjectLiteral(Model r, Resource subject, Resource predicate) throws ModelException {

    RDFNode obj = getObject(r, subject, predicate);
    if(obj instanceof Literal)
      return obj.toString();
    else
      return null;
  }

  public static Resource getObjectResource(Model r, Resource subject, Resource predicate) throws ModelException {

    RDFNode obj = getObject(r, subject, predicate);
    if(obj instanceof Resource)
      return (Resource)obj;
    else
      return null;
  }

  public static boolean isInstanceOf(Model r, Resource i, Resource cls) throws ModelException {
    return !r.find(i, RDF.type, cls).isEmpty();
  }

  public static RDFNode getObject(Model r, Resource subject, Resource predicate) throws ModelException {

    Model m = r.find(subject, predicate, null);
    if(m == null || m.size() == 0)
      return null;
    //    if(m.size() > 1)
    //      throw new RuntimeException("Model contains more than one triple");
    // FIXME: we do not check whether it is a resource or literal
    return ((Statement)m.elements().nextElement()).object();
  }

  public static Resource getSubject(Model r, Resource predicate, RDFNode object) throws ModelException {

    Model m = r.find(null, predicate, object);
    if(m == null || m.size() == 0)
      return null;
    //    if(m.size() > 1)
    //      throw new RuntimeException("Model contains more than one triple");
    return ((Statement)m.elements().nextElement()).subject();
  }

  /**
   * Prints the triples of a model to the given PrintStream.
   */
  public static void printStatements(Model m, PrintStream ps) throws ModelException {

    for(Enumeration en = m.elements(); en.hasMoreElements();) {

      Statement t = (Statement)en.nextElement();
      ps.println (t); // "triple(\""+t.subject()+"\",\""+t.predicate()+"\",\""+t.object()+"\").");
    }
  }

  /**
   * Dumps the model in a serialized form.
   */
  public static void dumpModel(Model m, PrintStream ps, RDFSerializer s) throws ModelException, IOException, SerializationException {
    s.serialize( m, new OutputStreamWriter(ps) );
  }

  /**
   * Dumps the model in a serialized form in a string
   */
  public static String dumpModel(Model m, RDFSerializer s) throws ModelException, IOException, SerializationException {

    StringWriter w = new StringWriter();
    s.serialize( m, w );
    return w.toString();
  }

  /**
   * Collects the triples of a model in a vector.
   */
  public static Vector getStatementVector(Model m) throws ModelException {

    Vector v = new Vector();
    for(Enumeration en = m.elements(); en.hasMoreElements();) {

      Statement t = (Statement)en.nextElement();
      v.addElement(t);
    }
    return v;
  }

  /**
   * Removes all triples which have something to do with the given namespace
   */
  public static Model removeNamespace( String ns, Model m ) throws ModelException {

    Model res = m.duplicate();
    for(Enumeration en = m.duplicate().elements(); en.hasMoreElements();) {

      Statement t = (Statement)en.nextElement();
      if(t.subject().toString().startsWith( ns ) ||
         t.predicate().toString().startsWith( ns ) ||
         t.object().toString().startsWith( ns )) {
        //      System.err.println("REMOVING TRIPLE: " + t);
        res.remove(t);
      }
    }
    return res;
  }

  /**
   * returns a subgraph of "m" containing
   * "r" and all nodes reachable from "r" via directed edges.
   * These edges are also included in the resulting model.
   */
  public static Model getReachable(Resource r, Model m) throws ModelException {

    Model result = m.create();
    getReachable(r, m, result);
    return result;
  }

  // FIXME: use digest instead of size
  static void getReachable(Resource r, Model m, Model result) throws ModelException {

    int oldSize = result.size();
    Model directlyReachable = m.find(r, null, null);
    SetOperations.unite(result, directlyReachable);
    if(result.size() == oldSize)
      return;
    for(Enumeration en = directlyReachable.elements(); en.hasMoreElements();) {
      Statement t = (Statement)en.nextElement();
      if(t.object() instanceof Resource)
        getReachable((Resource)t.object(), m, result);
    }
  }

  public static void parse(String fileNameOrURL, RDFParser parser, Model model) throws IOException, SAXException, MalformedURLException, ModelException {

    URL url = new URL (normalizeURI(fileNameOrURL));

    // maybe this model is loaded as schema...
    //    Model model = factory.registry().get(url.toString());
    //    if(model != null)
    //      return model;

    // Prepare input source
    InputSource source = new InputSource( url.openStream() );
    source.setSystemId( url.toString() );
    model.setSourceURI( url.toString() );

    parser.parse( source, new ModelConsumer(model) );
  }

  public static String normalizeURI(String uri) {

    // normalise uri
    URL url = null;
    try {
      url = new URL (uri);
    } catch (Exception e) {
      try {
        if(uri.indexOf(':') == -1)
          url = new URL ("file", null, uri);
      } catch (Exception e2) {}
    }
    return url != null ? url.toString() : uri;
  }

  public static Hashtable getResources(Model m) throws ModelException {

    Hashtable t = new Hashtable();
    for(Enumeration en = m.elements(); en.hasMoreElements();) {
      Statement s = (Statement)en.nextElement();
      t.put(s.subject(), s.subject());
      if(s.object() instanceof Resource)
        t.put(s.object(), s.object());
    }
    return t;
  }
}
