/**
 * Copyright  Sergey Melnik (Stanford University, Database Group) 
 *
 * Distribution policies are governed by the W3C software license.
 * http://www.w3.org/Consortium/Legal/copyright-software   
 * 
 * All Rights Reserved.
 * 
 * @author      Sergey Melnik <melnik@db.stanford.edu>
 */

package org.w3c.rdf.util;

import org.w3c.rdf.model.*;
import org.w3c.rdf.tools.crypt.DigestException;
import org.w3c.rdf.tools.crypt.*;
import java.security.*;
import java.util.Enumeration;

/**
 * An implementation of the algorithms for computing digests of triples and models.
 */

public class RDFDigestUtil {

  private static String algorithm = Digest.SHA1;

  public static String getDigestAlgorithm() {
    return algorithm;
  }
  
  public static void setDigestAlgorithm(String alg) {
    algorithm = alg;
  }

  public static String modelDigestToURI(Digest d) throws DigestException {

    return "uuid:rdf:" + algorithm + "-" + DigestUtil.toHexString(d);
  }

  public static String statementDigestToURI(Digest d) throws DigestException {

    return "uuid:rdf:" + algorithm + "-" + DigestUtil.toHexString(d);
  }

  /**
   * @returns the digest of an RDF node
   */
  public static Digest computeNodeDigest(RDFNode n) throws ModelException {
    
    try {
      return DigestUtil.computeDigest(algorithm, n.getLabel());
    } catch (NoSuchAlgorithmException ex) {
      throw new RuntimeException("RDFDigestUtil: no implementation for " + algorithm + " on your Java plattform");
    }
  }

  /**
   * @returns the standard digest of an RDF triple
   */
  public static Digest computeStatementDigest(Statement t)  throws DigestException, ModelException {

    byte[] s = getNodeDigest(t.subject()).getDigestBytes();
    byte[] p = getNodeDigest(t.predicate()).getDigestBytes();
    byte[] o = getNodeDigest(t.object()).getDigestBytes();

    int l = s.length;
    byte[] b = new byte[l * 3];

    System.arraycopy(s, 0, b, 0, l);
    System.arraycopy(p, 0, b, l, l);

    if(t.object() instanceof Resource)
      System.arraycopy(o, 0, b, l*2, l);
    else { // rotate by one byte
      for(int i=0; i < l; i++)
      b[l * 2 + ( (i+1) % l )] = o[i];
    }

    try {
      return DigestUtil.computeDigest(algorithm, b);
    } catch (NoSuchAlgorithmException ex) {
      throw new RuntimeException("RDFDigestUtil: no implementation for " + algorithm + " on your Java plattform");
    }

  }

  /**
   * @returns the standard digest of an model
   */
  public static Digest computeModelDigest(Model m) throws DigestException, ModelException {

    byte[] digest = null;
    Enumeration en = m.elements();

    if(!en.hasMoreElements())
      return null;

    while(en.hasMoreElements()) {

      Statement t = (Statement)en.nextElement();
      Digest d = getStatementDigest(t);
      if(digest == null)
	digest = d.getDigestBytes();
      else
	DigestUtil.xor(digest, d.getDigestBytes());

    }

    try {
      return DigestUtil.createFromBytes(algorithm, digest);
    } catch (DigestException de) { // cannot happen
      throw new InternalError(de.toString());
    }
  }

  public static Digest getNodeDigest(RDFNode n) throws DigestException, ModelException {

    return n instanceof Digestable ? ((Digestable)n).getDigest() : computeNodeDigest(n);
  }

  public static Digest getStatementDigest(Statement t)  throws DigestException, ModelException {

    return t instanceof Digestable ? ((Digestable)t).getDigest() : computeStatementDigest(t);
  }

  public static Digest getModelDigest(Model m) throws DigestException, ModelException {

    return m instanceof Digestable ? ((Digestable)m).getDigest() : computeModelDigest(m);
  }
}
