/**
 * Copyright  Sergey Melnik (Stanford University, Database Group) 
 *
 * Distribution policies are governed by the W3C software license.
 * http://www.w3.org/Consortium/Legal/copyright-software   
 * 
 * All Rights Reserved.
 * 
 * @author      Sergey Melnik <melnik@db.stanford.edu>
 */

package org.w3c.rdf.tools.crypt;

import java.security.*;
import java.io.*;
import java.math.BigInteger;

/**
 * This class is a container of static functions for manipulating cryptographic digests.
 *
 * @see org.w3c.rdf.tools.crypt.Digestable
 * @see org.w3c.rdf.tools.crypt.Digest
 */

public class DigestUtil {

  public static Digest computeDigest(String alg, String str) throws NoSuchAlgorithmException {

    try {
      byte[] b = str.getBytes("UTF8");
      return computeDigest(alg, b);
    } catch (UnsupportedEncodingException exc) {
      throw new RuntimeException("DigestImpl: weird internal error: UTF-8 is not supported");
    }
  }

  public static Digest computeDigest(String alg, byte[] b) throws NoSuchAlgorithmException {

    MessageDigest md = MessageDigest.getInstance(alg);
    byte[] digest = md.digest(b);
    try {
      return createFromBytes(alg, digest);
    } catch (DigestException de) {
      // cannot happen
      throw new RuntimeException("Bogus implementation of digest algorithm " + alg + ": " + de.getMessage());
    }
  }

  public static Digest createFromBytes(String alg, byte[] digest) throws DigestException {

    if(Digest.MD5.equals(alg)) {
      if(digest.length != 16)
				throw new DigestException("MD5 digest must be 16 bytes long");
      return new MD5Digest(digest);
    }

    if(Digest.SHA1.equals(alg)) {
      if(digest.length != 20)
				throw new DigestException("SHA-1 digest must be 20 bytes long");
      return new SHA1Digest(digest);
    }

    return new GenericDigest(alg, digest);
  }

  /*
   *@return -1 if a DigestException is caught ...
   */
  public static int getHashCode(Digest d) {

    try {
      return digestBytes2HashCode(d.getDigestBytes());
    } catch (DigestException de) {
      // XXXXX - need to do something here ...
      return -1;
    }
  }

  public static int digestBytes2HashCode(byte[] digest) {
    
    return
			((int) digest[0] & 0xff) |
      (((int) digest[1] & 0xff) << 8) |
      (((int) digest[2] & 0xff) << 16) |
      (((int) digest[3] & 0xff) << 24);
  }

  public static void xor(byte[] d1, byte[] d2) {
    
    int l = d1.length;
    for(int i = 0; i < l; i++)
      d1[i] ^= d2[i];
  }

  public static void xor(byte[] d1, byte[] d2, int shift) {
    
    int l = d1.length;
    for(int i = 0; i < l; i++)
      d1[(i + shift) % l] ^= d2[i];
  }
	
  public static boolean equal(Digest d1, Digest d2) {

    if(d1 == d2)
      return true;

    try {
      if(d1 instanceof AbstractDigest && d2 instanceof AbstractDigest)
      	return equal(((AbstractDigest)d1).digest, ((AbstractDigest)d2).digest);
/*tdh*/      else{ // general case
      if(d1.getDigestAlgorithm().equals(d2.getDigestAlgorithm()))
        return equal(d1.getDigestBytes(), d2.getDigestBytes());
      else // undecidable. FIXME: throw an exception?
        return false; }
    } catch (DigestException de) {
        // XXXXX - throw an exception ???
        return false;
    }
  }

  /**
   * Tests the equality of digests
   */
  public static boolean equal(byte[] d1, byte[] d2) {

    // just to be sure
    if(d1.length != d2.length)
      return false;

    for(int i = 0; i < d1.length; i++)
      if(d1[i] != d2[i])
				return false;

    return true;
  }

  /**
   * @returns hexadecimal representation of the Digest or null if a DigestException is caught
   */  
  public static String toHexString (Digest d) {
    try {
      return toHexString(d.getDigestBytes());
    } catch (DigestException de) {
      // XXXXX - throw an exception ???
      return null;
    }
  }

  /**
   * @returns hexadecimal representation of the byte array
   */  
  public static String toHexString (byte buf[]) {

    StringBuffer sb = new StringBuffer(2*buf.length) ;
    for (int i = 0 ; i < buf.length; i++) {
      int h = (buf[i] & 0xf0) >> 4 ;
      int l = (buf[i] & 0x0f) ;
      sb.append ((char)((h>9) ? 'a'+h-10 : '0'+h)) ;
      sb.append ((char)((l>9) ? 'a'+l-10 : '0'+l)) ;
    }

    return sb.toString() ;
  }

}

// AUXILIARY CLASSES

abstract class AbstractDigest implements Digest {

	byte[] digest;

	protected AbstractDigest(byte[] b) {

		digest = new byte[b.length];
		System.arraycopy(b, 0, digest, 0, b.length);
	}

	public byte[] getDigestBytes() {

		byte[] result = new byte[digest.length];
		System.arraycopy(digest, 0, result, 0, digest.length);
		return result;
	}

	public int hashCode() {

		return DigestUtil.digestBytes2HashCode(digest);
	}

	public boolean equals (Object that) {

		if(!(that instanceof Digest))
			return false;

		return DigestUtil.equal(this, (Digest)that);
	}

	public String toString() {
		return DigestUtil.toHexString(digest);
	}
}

class GenericDigest extends AbstractDigest {

	String algorithm;

	protected GenericDigest(String alg, byte[] b) {
		super(b);
		algorithm = alg;
	}

	public String getDigestAlgorithm() {
		return algorithm;
	}
}

// Specific digests: save bytes for algorithm type

final class SHA1Digest extends AbstractDigest {

	SHA1Digest(byte[] b) {
		super(b);
	}

	public String getDigestAlgorithm() {
		return SHA1;
	}
}

final class MD5Digest extends AbstractDigest {

	protected MD5Digest(byte[] b) {
		super(b);
	}

	public String getDigestAlgorithm() {
		return MD5;
	}
}

