/**
 * Copyright  Sergey Melnik (Stanford University, Database Group) 
 *
 * Distribution policies are governed by the W3C software license.
 * http://www.w3.org/Consortium/Legal/copyright-software   
 * 
 * All Rights Reserved.
 */

package org.w3c.rdf.implementation.model;

import org.w3c.rdf.model.*;
import org.w3c.rdf.util.RDFDigestUtil;
import org.w3c.rdf.tools.crypt.*;

import java.util.*;

/**
 * @author Sergey Melnik <melnik@db.stanford.edu>
 *
 * modified by William Grosso <grosso@acm.org>.
 */

public class ModelImpl implements Model, Digestable, Digest {

  static NodeFactory DEFAULT_NODE_FACTORY = new NodeFactoryImpl();
  NodeFactory nodeFactory = DEFAULT_NODE_FACTORY;

  byte[] digest;

  String uri;
  Hashtable triples;

  // first find operation creates lookup table
  private FindIndex _findIndex;

  public ModelImpl() {
    triples = new Hashtable();
    _findIndex = new FindIndex();
  }

  public ModelImpl(NodeFactory f) {
		this();
		nodeFactory = f;
  }

  protected ModelImpl(String uri, Hashtable triples, FindIndex findIndex) {
    this.uri = uri;
    this.triples = triples;
    _findIndex = findIndex;
  }

  /**
   * Set a base URI for the message.
   * Affects creating of new resources and serialization syntax.
   * <code>null</code> is just alright, meaning that
   * the serializer will use always rdf:about and never rdf:ID
   */
  public void setSourceURI(String uri) {
    this.uri = uri;
  }

  /**
   * Returns current base URI setting.
   */
  public String getSourceURI() {
    return uri;
  }

  // Model access

  /**
   * Number of triples in the model
   *
   * @return  number of triples
   */
  public int size() {
    return triples.size();
  }

	public boolean isEmpty() {
		return triples.isEmpty();
	}

  /**
   * Enumerate triples
   */
  public Enumeration elements() {
    return triples.elements();
  }

  /**
   * Tests if the model contains the given triple.
   *
   * @return  <code>true</code> if the triple belongs to the model;
   *          <code>false</code> otherwise.
   * @see     org.w3c.rdf.model.Statement
   */
  public synchronized boolean contains(Statement t) {
    return triples.contains(t);
  }

  // Model manipulation: add, remove, find

  /**
   * Adds a new triple to the model. A literal is created out of the string parameter <code>object</code>.
   * This method is just a shortcut.
   *
   * @see     org.w3c.rdf.model.Resource
   */
  public synchronized void add(Resource subject, Resource predicate, String object) throws ModelException {

    add(nodeFactory.createStatement(subject, predicate, nodeFactory.createLiteral(object)));
  }

  /**
   * Adds a new triple to the model. The object of the triple is an RDFNode.
   *
   * @see     org.w3c.rdf.model.Resource
   * @see     org.w3c.rdf.model.RDFNode
   */
  public synchronized void add(Resource subject, Resource predicate, RDFNode object) throws ModelException {

    add(nodeFactory.createStatement(subject, predicate, object));
  }


  /**
   * Adds a new triple to the model.
   *
   * @see     org.w3c.rdf.model.Statement
   */
  public synchronized void add( Statement t ) throws ModelException {

    triples.put(t, t);

    if(validLookup())
         _findIndex.addLookup(t);

    updateDigest(t);
  }

  synchronized void updateDigest(Statement t)  throws ModelException {

    if(digest == null)
      return;

    try {
      Digest d = RDFDigestUtil.getStatementDigest(t);
      DigestUtil.xor(digest, d.getDigestBytes());
    } catch (DigestException exc) {

      throw new ModelException(exc.toString());
    }
  }

  boolean validLookup() {

    return _findIndex.size() > 0;
  }

  /**
   * Removes the triple from the model.
   *
   * @see     org.w3c.rdf.model.Statement
   */
  public synchronized void remove(Statement t) throws ModelException {

    triples.remove(t);

    if(validLookup())
     _findIndex.removeLookup(t);

    updateDigest(t);
  }


  /**
   * General method to search for triples.
   * <code>null</code> input for any parameter will match anything.
   * <p>Example: <code>Model result = m.find( null, RDF.type, new Resource("http://...#MyClass") )</code>
   * <p>finds all instances of the class <code>MyClass</code>
   *
   * @see     org.w3c.rdf.model.Resource
   * @see     org.w3c.rdf.model.RDFNode
   */
  public synchronized Model find( Resource subject, Resource predicate, RDFNode object ) throws ModelException {

	// TODO: make everything synchronized by triples

	Model res = new ModelImpl();// EMPTY_MODEL;

	if(triples.size() == 0)
		return res;

	if(subject == null && predicate == null && object == null)
		return this.duplicate();

	boolean createLookup = !validLookup();

		Enumeration en = createLookup ? triples.elements() : _findIndex.multiget(subject, predicate, object);

	int numFound = 0;

	while(en.hasMoreElements()) {

		Statement t = (Statement)en.nextElement();

		if(createLookup)
       _findIndex.addLookup(t);

		if(matchStatement(t, subject, predicate, object)) {
			res.add(t);
		}
	}
	return res;
  }


  boolean matchStatement(Statement triple, Resource subject, Resource predicate, RDFNode object) throws ModelException {

    if( subject != null && !triple.subject().equals(subject))
      return false;
      
    if(predicate != null && !triple.predicate().equals(predicate))
      return false;
      
    if(object != null && !triple.object().equals(object))
      return false;

    return true;
  }

  /**
   * Clone the model.
   */
  public Model duplicate() {
    // creates a model that shares triples and lookup with this model
    return new ModelImpl(uri, triples, _findIndex);
  }

  public Object clone() {
    return duplicate();
  }

  /**
   * Creates empty model of the same Class
   */
  public Model create() {
    return new ModelImpl();
  }

	public NodeFactory getNodeFactory() {

		return nodeFactory;
	}
  // Resource implementation

  public String getLabel() throws ModelException {

    return getURI();
  }

  public Digest getDigest() {

    return this;
  }

  public boolean isMutable() {

    return true;
  }

  public String getURI() throws ModelException {

    if(isEmpty())
      return nodeFactory.createUniqueResource().toString();
    else
      try {
      return RDFDigestUtil.modelDigestToURI(getDigest());
      } catch (DigestException de) {
	throw new ModelException("Could not obtain model digest: " + de);
      }
  }

  // Digest implementation
  public String getDigestAlgorithm() {

    return RDFDigestUtil.getDigestAlgorithm();
  }

  public synchronized byte[] getDigestBytes() throws DigestException {

    try {
      if(digest == null)
      digest = RDFDigestUtil.computeModelDigest(this).getDigestBytes();
      
      byte[] result = new byte[digest.length];
      System.arraycopy(digest, 0, result, 0, digest.length);
      return result;
    } catch (ModelException exc) {
      throw new DigestException(exc.toString());
    }
  }
}
