/**
 * Copyright  World Wide Web Consortium, (Massachusetts Institute of
 * Technology, Institut National de Recherche en Informatique et en
 * Automatique, Keio University).
 *
 * All Rights Reserved.
 *
 * Please see the full Copyright clause at
 * <http://www.w3.org/Consortium/Legal/copyright-software.html>
 *
 * This class is a RDFConsumer that outputs RDF triples in a format suitable
 * for GraphViz (http://www.research.att.com/sw/tools/graphviz/)input.  
 * All output is written to stdout by default but may be over-written
 * by calling setOutputStream.
 *
 * Sample usage from command line:
 * java GraphVizDumpConsumer input_file.rdf > output_file.dot
 * 
 * @author Thomas Hubbard (hubbard@w3.org)
 * @version 1.0
 */

package org.w3c.rdf.examples;

import java.util.*;
import java.io.PrintWriter;
import org.w3c.rdf.model.*;
import org.w3c.rdf.syntax.*;
import org.w3c.rdf.util.xml.DumpConsumer;
import org.w3c.rdf.implementation.model.StatementImpl;
import org.w3c.rdf.implementation.model.NodeFactoryImpl;
import javax.servlet.ServletOutputStream;

// The following are needed by main() for testing
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.w3c.rdf.util.xml.GenericParser;
import org.w3c.rdf.util.xml.ErrorStore;
import org.w3c.rdf.implementation.syntax.sirpac.SiRPAC;

public class GraphVizDumpConsumer extends DumpConsumer implements RDFConsumer {
      
  NodeFactory mFactory = new NodeFactoryImpl();
  Hashtable mHashtable = null;
  private PrintWriter mPrintWriter = null;
  private Vector graphAttrs = null;

  /**
   * Default constructor
   */
  public GraphVizDumpConsumer () {
  }

  /**
   * Default constructor
   */
  public GraphVizDumpConsumer (PrintWriter pw) {
      mPrintWriter = pw;
  }

  /**
   * This is the title to be used for the generated VizGraph
   * graph.  The default is "foo".
   */
  public String title = "foo";

  /**
   * Interface to accept graph-level attributes that should be
   * added to the graph's header information.
   *
   * @param attr a graph-level attribute string (e.g. size="4,4") to
   *   be added to the output
   */
  public void addGraphAttribute(String attr)
  {
      if (graphAttrs == null)
          graphAttrs = new Vector(3);

      graphAttrs.add(attr);
  }

  /**
   * This method adds a namespace prefix and the associated namespace URI 
   * to the consumer.  Instances of the fully qualified namespace URIs in 
   * all nodes are replaced with the prefix.
   * 
   * Note:
   * empty prefixes are supported, for example,
   * foo.addNamespace( "", fullURI );
   * Will cause all instances of fullURI to be replaced by
   * the empty string, useful for omitting the doc-source
   * node that SiRPAC produces.
   * 
   * @param namespacePrefix The prefix to associate with the 
   * fully qualified namespace URI.
   * @param namespaceName The fully qualified namespace name.
   */
  public void addNamespace(String namespacePrefix, String namespaceName )
  {
    if( mHashtable == null ){
        mHashtable = new Hashtable();
    }
    mHashtable.put(namespaceName, namespacePrefix );
  }

  /**
   * This method dumps the Graph Viz header.  Put any 
   * special GraphViz commands here.
   */
  public void startModel () 
  {
    if (mPrintWriter == null)
        System.out.println( "digraph " + title + "{ " );
    else
        mPrintWriter.println( "digraph " + title + "{ " );

    if (graphAttrs != null)
    {
        Enumeration e = graphAttrs.elements();
        while (e.hasMoreElements()) {   
            if (mPrintWriter == null)
                System.out.println( "  " + e.nextElement());
            else
                mPrintWriter.println( "  " + e.nextElement());
        }
    }
  }

  /**
   * This method dumps out the Graph Viz footer.
   */
  public void endModel () {
    if (mPrintWriter == null)
        System.out.println( " }");
    else
        mPrintWriter.println( " }");
  }

  public NodeFactory getNodeFactory() {
    return mFactory;
  }

  /**
   * This method outputs text to stdout that is
   * suitable for use in GraphViz.
   * 
   * @param s
   */
  public void addStatement (Statement s) {
    try
    {
        String subjectName = s.subject().getLabel();   
        String predicateName = s.predicate().getLabel();
        String objectName = s.object().getLabel();

        if( mHashtable != null )
        {
            Enumeration e = mHashtable.keys();
            while( e.hasMoreElements() )
            {
                String node = (String)e.nextElement();
                
                String prefix = null;
                if( subjectName.startsWith( node ) )
                {
                    prefix = (String)mHashtable.get( node ); 
                    subjectName =  (prefix == "" ? "" : ":") + subjectName.substring( node.length() );
                }
                if( predicateName.startsWith( node ) )
                {
                    prefix = (String)mHashtable.get( node ); 
                    predicateName = mHashtable.get( node ) + (prefix == "" ? "" : ":") + predicateName.substring( node.length() );
                }
                if( objectName.startsWith( node ) )
                {
                    prefix = (String)mHashtable.get( node ); 
                    objectName = mHashtable.get( node ) + (prefix == "" ? "" : ":") + objectName.substring( node.length() );
                }
            }
        }

        /*
         * Before outputing the object, must do the following if it
         * is a Literal:
         *
         *   Remove leading whitespace
         *
         *   GraphViz/DOT cannot handle embedded line terminators characters
         *   so they must be replaced with spaces
         *   
         *   Limit the number of chars to make the graph legible
         *
         *   Embedd the literal in quotes
         *
         *   Escape double quotes
         */
        String tmpObjectName = objectName;
        if (s.object() instanceof Literal) 
        {
            String s1 = objectName.trim();
            s1 = s1.replace('\n', ' ');
            s1 = s1.replace('\f', ' ');
            s1 = s1.replace('\r', ' ');
            if (s1.indexOf('"') != -1)
                s1 = SiRPACServlet.replaceString(s1, "\"", "\\\"");

            // Anything beyond 80 chars makes the graph too large
            if (s1.length() >= 80)
                tmpObjectName = s1.substring(0, 80) + " ...";
            else
                tmpObjectName = s1.substring(0, s1.length());

            tmpObjectName = "\\\"" + tmpObjectName + "\\\"";
        }

        if (mPrintWriter == null)
            System.out.println( "\"" + subjectName + "\" -> \"" + tmpObjectName + "\" [label=\"" +  predicateName + "\"];" );
        else
             mPrintWriter.println( "\"" + subjectName + "\" -> \"" + tmpObjectName + "\" [label=\"" +  predicateName + "\"];" );
    }
    catch( ModelException me )
    {
        // Let's see the broken graph.
        if (mPrintWriter == null)
        {
            System.out.println("}");
            System.err.println( "An attempt to create the GraphViz DOT file failed. " + me );
            System.exit(0);
        } else {
            mPrintWriter.println("}");
            System.err.println( "An attempt to create the GraphViz DOT file failed. " + me );
        }
    }    
  }

  /**
   * main method for debugging the DOT file generator
   *
   * Input is a URI
   * Output (to stdout) is the input for the DOT program
   */
  public static void main (String args[]) throws Exception {

    SiRPAC parser = new SiRPAC ();

    boolean robust = false;

    if (args == null) {
        System.err.println("USAGE: must enter a URI");
        System.exit(1);
    }

    String urlStr = args[0];

    GraphVizDumpConsumer consumer = new GraphVizDumpConsumer();
    InputSource source  = GenericParser.getInputSource(urlStr);
    ErrorStore handler = new ErrorStore();

    parser.setErrorHandler(handler);

    try {
      parser.parse(source, consumer);
    } catch (SAXException se) {
      System.err.println("Error during parsing: " + se);
      se.getException().printStackTrace(System.err);
    }

    String sErrors = handler.errors ();
    if (sErrors != null && sErrors.length() > 0)
      System.err.println ("Errors during parsing:\n"+sErrors);
    
  }
}
