// MPStream.java
// $Id: MPStream.java,v 1.2 1996/08/09 15:15:46 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.mux ;

import java.io.* ;
import java.net.* ;

public class MPStream implements MUX {
    public final static int DEFSIZE = 16 ;
    public final static int DEFINCR = 8 ;

    /**
     * Am I the server end of this connection ?
     * This dictate if we should use odd or even session identifiers.
     */
    boolean server = false ;
    /**
     * Our reader object.
     */
    MPReader reader = null ;
    /**
     * Our writer object.
     */
    MPWriter writer = null ;
    /**
     * The session we have created.
     */
    Session sessions[] = null ;
    /**
     * Pending session count.
     */
    int pending_sessions = 0 ;
    /**
     * A fake session to read a session while accepting connections.
     */
    Session fake = null ;
    
    /**
     * Get this MPStream fake session (used for accepting new connections)
     */

    protected Session getFakeSession() {
	return fake ;
    }

    /**
     * Get this MPStream reader object.
     */

    protected MPReader getReader() {
	return reader ;
    }

    /**
     * Get this MPStream writer object.
     */

    protected MPWriter getWriter() {
	return writer ;
    }

    /**
     * Allocate a new session.
     * If the provided id is positive, we try to allocate this one, otherwise
     * we allocate the first available one.
     * @param id The identifier of the session to allocate.
     * @return A Session instance.
     * @exception IOException If the provided session identifier is invalid.
     */

    protected synchronized Session allocateSession (int id, int protid) 
	throws IOException
    {
	if ( id < 0 ) {
	    for (id = (server ? 2 : 3) ; id < sessions.length ; id += 2) 
		if (sessions[id] == null)
		    break ;
	    if ( id >= sessions.length ) {
		// Resize the session array:
		Session ns[] = new Session[id+DEFINCR] ;
 		System.arraycopy(sessions, 0, ns, 0, sessions.length) ;
		sessions = ns ;
	    }
	} else {
	    if ( server & ((id & 1) == 1) ) 
		throw new IOException ("invalid odd session identifier.") ;
	    if ( id > MAX_SESSION )
		throw new IOException ("session identifier too big.") ;
	    if ( id >= sessions.length ) {
		// Resize session array:
		Session ns[] = new Session[id+DEFINCR] ;
		System.arraycopy (sessions, 0, ns, 0, sessions.length) ;
		sessions = ns ;
	    }
	}
	// We now have an appropriate identifier, allocate the session:
	Session s = new Session (this, id, protid) ;
	sessions[id] = s ;
	return s ;
    }

    /**
     * Get the session associated with the given session id.
     * If such a session doesn't exist, try creating it.
     * @param id The session identifier.
     */

    protected synchronized Session getSession (byte flags, int id, int protid) 
	throws IOException
    {
	// Check identifier:
	if ( id >= MAX_SESSION ) {
	    throw new IOException ("session identifier too large.");
	} else if ( id >= sessions.length ) {
	    // Resize session array:
	    Session ns[] = new Session[id+DEFINCR] ;
	    System.arraycopy (sessions, 0, ns, 0, sessions.length) ;
	    sessions = ns ;
	}
	// Create, or get the session:
	Session s = sessions[id] ;
	if ( s == null ) {
	    if ( server & ((id & 1) == 0) ) {
		throw new IOException ("invalid even session identifier.");
	    } else if ( (flags & MUX_SYN) != MUX_SYN ) {
		throw new IOException ("unable to create non SYN session.");
	    } else {
		s = new Session (this, id, protid, true) ;
		sessions[id] = s ;
		pending_sessions++ ;
		notify() ;
	    } 
	}
	return s ;
    }

    protected Session getSession (byte flags, int id)
	throws IOException
    {
	return getSession (flags, id, -1) ;
    }

    protected Session getSession (int id) 
	throws IOException
    {
	return getSession ((byte) 0, id, -1) ;
    }

    protected synchronized Session lookupSession (int id) {
	if ( id < sessions.length )
	    return sessions[id] ;
	return null ;
    }

    /**
     * Unregister the given session.
     * The session should have been closed properly (both its input and its
     * output). This method will than unregister any knowledge we have
     * of it, in particular, its session identifier will be available for
     * reuse.
     * @param session The session to unregister.
     */

    protected synchronized void unregisterSession (Session session) {
	int id = session.getIdentifier() ;
	if ( (id >= 0) && (id < sessions.length) )
	    sessions[id] = null ;
    }

    protected synchronized void syn (byte flags, int id, int protid) {
//	System.out.println("syn: not implemented.") ;
	return ;
    }

    /**
     * Close the corresponding input stream of the session.
     * This method is invoked by the reader, when it receives a message with
     * the FIN bit set. The corresponding input session stream is to be closed
     * and any reader need to be notified of the close operation.
     * @param flags Flags of the received message.
     * @param id The session identitifer of the message.
     * @param protid The protocol running on this session (or -1).
     * @exception RuntimeException If the FIN message is for an unexsiting
     *     session.
     */

    protected synchronized void fin (byte flags, int id, int protid)
	throws IOException
    {
	Session session = lookupSession (id) ;
	if ( session == null ) 
	    throw new RuntimeException (this.getClass().getName()
					+" fin on unexisting session "+id);
	session.closeInputStream() ;
    }

    protected synchronized void rst (byte flags, int id, int protid) {
	System.out.println ("rst: not implemented.") ;
    }

    protected synchronized void push (byte flags, int id, int protid) {
	System.out.println ("push: not implemented.") ;
    }

    /**
     * Connect to the other end, using the given protocol id.
     * @param session The session to use.
     * @param protocol The protocol number to use on this session.
     * @param buf Associated data (can be <strong>null</strong>).
     * @param off Offset in the provided buffer.
     * @param len Length in the provided buffer.
     */

    public Session connect (int id
			    , int protocol
			    , byte buf[], int off, int len)
	throws IOException
    {
	Session session = allocateSession (id, protocol) ;
	if ( protocol < 0 ) {
	    writer.writeMessage (session.getIdentifier()
				 , (byte) MUX_SYN
				 , protocol
				 , buf, off, len) ;
	} else {
	    writer.writeMessage(session.getIdentifier()
				, (byte)(MUX_SYN|MUX_LONG_LENGTH)
				, protocol
				, buf, off, len) ;
	}
	return session ;
    }

    /**
     * Connect to target, to speak the given protocol.
     * @param protocol The protocol identfier to speak on the new session.
     */

    public Session connect (int protocol)
	throws IOException
    {
	return connect (-1, protocol, null, 0, 0) ;
    }

    /**
     * Connect to target, to speak the given protocol, send data.
     * @param protocol The protocol to speak on the new session.
     * @param buf The data to send with the connection packet.
     * @param buf The buffer containing the data.
     * @param off Offset in the buffer.
     * @param len Length of data to send.
     */

    public Session connect (int protocol, byte buf[], int off, int len)
	throws IOException
    {
	return connect (-1, protocol, buf, off, len) ;
    }
    /**
     * Connect to target, using any session identifier, and without protocol.
     */

    public Session connect () 
	throws IOException
    {
	return connect (-1, -1, null, 0, 0) ;
    }

    /**
     * Check for session waiting for accept:
     */

    protected synchronized Session checkAccept(int protid)
	throws IOException
    {
	if ( pending_sessions > 0 ) {
	    for (int id=(server ? 3 : 2) ; id < sessions.length ; id+=2) {
		Session s = sessions[id] ;
		if ( s == null )
		    continue ;
		if ( s.accept(protid) != null ) {
		    pending_sessions-- ;
		    return s ;
		}
	    }
	}
	return null ;
    }

    /**
     * Accept any incoming connection on this stream.
     */

    public Session accept (int protid) 
	throws IOException 
    {
	while ( true ) {
	    // Any pending session ?
	    Session s = checkAccept(protid) ;
	    if ( s != null )
		return s;
	    // Should I become a reader for the stream ?
	    switch (reader.tryRead (fake, (byte[]) null, 0, 0)) {
	      case -3:
		  break ;
	      case -2:
		  synchronized (fake) {
		      try {
			  fake.wait() ;
		      } catch (InterruptedException ex) {
		      }
		  }
		  break ;
	      default:
		  System.out.println ("*** accept (error !)") ;
	    }
	}
    }

    public Session accept ()
	throws IOException
    {
	return accept(-1) ;
    }

    /**
     * Build a multiplexed stream out of the given streams.
     * @param server Am I server on this connection ?
     * @param input The input stream to multiplex.
     * @param output The output stream to multiplex.
     */

    public MPStream (boolean server, InputStream input, OutputStream output)
	throws IOException
    {
	this.server   = server ;
	this.fake     = new Session (this, -1) ;
	this.reader   = new MPReader (this, input) ;
	this.writer   = new MPWriter (this, output) ;
	this.sessions = new Session[DEFSIZE] ;
    }

    /**
     * Build a multiplexed stream out of the given socket.
     * @param server Should this end of the multiplexer act as the server or
     *    as the client.
     * @param socket The socket to multiplex.
     */

    public MPStream (boolean server, Socket socket) 
	throws IOException
    {
	this (server, socket.getInputStream(), socket.getOutputStream()) ;
    }
	
    
}
