// ClientPool.java
// $Id: ClientPool.java,v 1.19 1996/10/02 19:52:37 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.http ;

import java.io.* ;
import java.net.* ;

import w3c.jigsaw.daemon.*;
import w3c.util.*;

class ClientState implements LRUAble {
    static final int C_IDLE = 0;	// Zombie
    static final int C_BUSY = 1;	// Is in busy list
    static final int C_FREE = 2;	// Is in free list
    static final int C_KILL = 3;	// Being killed
    static final int C_FIN  = 4;
    LRUAble next   = null;
    LRUAble prev   = null;
    Client  client = null;
    int     id     = 0;
    int     status = C_IDLE;
    boolean bound  = false;

    ClientState csnext = null;
    ClientState csprev = null;

    static int nextid = 0;
    static final synchronized int nextId() {
	return nextid++;
    }

    public final LRUAble getNext() {
	return next;
    }

    public final LRUAble getPrev() {
	return prev;
    }

    public final void setNext(LRUAble next) {
	this.next = next;
    }

    public final void setPrev(LRUAble prev) {
	this.prev = prev;
    }

    ClientState(ClientState cshead) {
	this.status   = C_IDLE;
	this.id       = nextId();
	cshead.csprev = this;
	this.csnext   = cshead;
    }

    // Used to create the head of the list.
    ClientState() {
    }

}

class DebugThread extends Thread {
    ClientPool pool = null;

    public void run() {
	while ( true ) {
	    try {
		sleep(1000*10);

		/*
		ClientState cs = (ClientState) pool.freeList.getHead();
		while (cs != null) {
		    System.out.println("[free]:"+cs.client);
		    cs = (ClientState) pool.freeList.getNext((LRUAble) cs);
		    
		}
		*/
		System.out.println("freeCount ="+pool.freeCount);
		System.out.println("idleCount ="+pool.idleCount);
		System.out.println("totalCount="+pool.clientCount);
		System.out.println("estimCount="+pool.clientEstim);
		System.out.println("Average: "+pool.loadavg);
	    } catch (Exception ex) {
		ex.printStackTrace();
	    }
	}
    }

    DebugThread(ClientPool pool) {
	this.pool = pool;
	setPriority(Thread.MAX_PRIORITY);
    }

    
}


/**
 * The client pool is a kind of client factory.
 * Each time the server gets a new connection, it calls the client pool
 * to bound a client object (newly created or spared) to handle it.
 */

public class ClientPool implements PropertyMonitoring {
    private static final boolean debug = false;

    // FIXME doc
    public final static
    String MINSPARE_FREE_P = "w3c.jigsaw.http.ClientPool.minFree";
    // FIXME doc
    public final static
    String MAXSPARE_FREE_P = "w3c.jigsaw.http.ClientPool.maxFree";
    // FIXME doc
    public final static
    String MAXSPARE_IDLE_P = "w3c.jigsaw.http.ClientPool.maxIdle";
    // FIXME doc
    public final static
    String MAXTHREADS_P = "w3c.jigsaw.http.ClientPool.maxThreads";

    int minFree    = 0;
    int maxFree    = 0;
    int maxIdle    = 0;
    int maxThreads = 0;

    int              count     = 0 ;	// number of created clients.
    httpd            server    = null ;
    DaemonProperties props     = null ;
    int              busyCount = 0 ;	// number of busy clients.

    LRUList idleList = null;
    LRUList freeList = null;

    ClientState csList = null;

    int idleCount   = 0 ;
    int freeCount   = 0 ;
    int clientCount = 0 ;
    int clientEstim = 0 ;

    public static final int MINSPARE_FREE = 5;
    public static final int MAXSPARE_FREE = 10;
    public static final int MINSPARE_IDLE = 5;
    public static final int MAXSPARE_IDLE = 30;
    public static final int MAXTHREADS    = 50;

    public static final int AVG_LIGHT = 1;
    public static final int AVG_NORMAL = 2;
    public static final int AVG_HIGH = 3;
    public static final int AVG_DEAD = 4;
    
    int loadavg = AVG_LIGHT;

    boolean alive = true;

    /**
     * Some property have changed, update our setting.
     * @param name The name of the property that has changed.
     * @return A boolean, <strong>true</strong> if we updated ourself 
     *    successfully.
     */

    public boolean propertyChanged (String name) {
	httpd s = server;
	if ( name.equals(props.getPropertyName(s, MINSPARE_FREE_P)) ) {
	    minFree = props.getInteger(server, MINSPARE_FREE_P, minFree);
	} else if ( name.equals(props.getPropertyName(s, MAXSPARE_FREE_P)) ) {
	    maxFree = props.getInteger(server, MAXSPARE_FREE_P, maxFree);
	} else if ( name.equals(props.getPropertyName(s, MAXSPARE_IDLE_P)) ) {
	    maxIdle = props.getInteger(server, MAXSPARE_IDLE_P, maxIdle);
	} else if ( name.equals(props.getPropertyName(s, MAXTHREADS_P)) ) {
	    maxThreads = props.getInteger(server, MAXTHREADS_P, maxThreads);
	}	
	return true ;
    }

    /**
     * Remove this client state from the glohbal client list.
     * @param cs The client state to remove from the list.
     */

    protected synchronized void deleteClient(ClientState cs) {
	if ( cs.csprev == null ) {
	    csList = cs.csnext;
	} else if ( cs.csnext == null ) {
	    cs.csprev.csnext = null;
	} else {
	    cs.csprev.csnext = cs.csnext;
	    cs.csnext.csprev = cs.csprev;
	}
    }

    /**
     * Create a new client for this pool.
     * @param free A boolean, if <strong>true</strong> the client is inserted
     * straight into the free list, otherwise, it is not plugged into any
     * list.
     * @return A ClientState instance, if creation of a new client was allowed,
     * <strong>null</strong> if no more clients could be created.
     */

    protected synchronized ClientState addClient (boolean free) {
	// Create a new client. 
	csList         = new ClientState(csList);
	ClientState cs = csList;
	Client      client = new Client(server, this, cs);
	cs.client = client;
	clientCount++;
	clientEstim++;
	// Plug into free LRU if required:
	if ( free ) {
	    cs.status = ClientState.C_FREE;
	    freeList.toHead(cs);
	    freeCount++;
	}
	return cs ;
    }

    /**
     * Update our idea of the current load.
     * The one important invariant here, is that whenever the free list
     * becomes empty, then the load average should be equals to
     * <strong>AVG_DEAD</strong>. This ensures that the server will start 
     * dropping connections.
     *
     */

    private final void updateLoadAverage() {
	int oldavg = loadavg;
	if ( freeCount >= maxFree ) {
	    loadavg = AVG_LIGHT;
	} else if ((freeCount >= minFree) || (idleCount >= maxIdle)) {
	    if ((loadavg = AVG_NORMAL) < oldavg)
		server.thread.setPriority(Thread.MAX_PRIORITY);
	} else if ( freeCount > 0 ) {
	    /* idleCount < MINSPARE_IDLE */
	    if ((loadavg = AVG_HIGH) > oldavg)
		server.thread.setPriority(server.getClientThreadPriority()-2);
	} else {
	    loadavg = AVG_DEAD;
	}
    }
	    
    private final synchronized void incrClientCount() {
	++clientCount;
	++clientEstim;
	updateLoadAverage();
    }

    private final synchronized void decrClientCount() {
	--clientCount;
	updateLoadAverage();
    }

    private final synchronized boolean incrFreeCount() {
	if ( clientEstim >= maxThreads ) {
	    clientEstim--;
	    return false;
	}
	++freeCount;
	updateLoadAverage();
	return true;
    }

    private final synchronized boolean decrFreeCount() {
	if ( freeCount > 0 ) {
	    --freeCount;
	    updateLoadAverage();
	    return true;
	} else {
	    return false;
	}
    }

    private final synchronized boolean incrIdleCount() {
	if ((loadavg >= AVG_NORMAL) || (idleCount + 1 >= maxIdle))
	    return false;
	++idleCount;
	updateLoadAverage();
	return true;
    }
    
    private final synchronized boolean decrIdleCount() {
	if ( idleCount > 0 ) {
	    --idleCount;
	    updateLoadAverage();
	    return true;
	} else {
	    return false;
	}
    }

    /**
     * Notify that this client has finished with its connection.
     * If the pool wants the client to be freed (because it has too many of 
     * them), it makes the client kill itself (which will trigger a call to
     * the clientFinished method, were enventual cleanup is performed).
     * @param client The client that is done with its connection.
     */

    protected boolean clientConnectionFinished (Client client) {
	ClientState cs = client.state;
	switch(cs.status) {
	  case ClientState.C_IDLE:
	      decrIdleCount();
	      idleList.remove(cs);
	      break;
	  case ClientState.C_BUSY:
	  case ClientState.C_KILL:
	      break;
	  default:
	    System.out.println(client+": conn-fin "+cs.status);
	    break;
	}
	if ( incrFreeCount() ) {
	    freeList.toTail(cs);
	    return true;
	} else {
	    return false;
	}
    }

    /**
     * Notify that this client has been killed.
     * @param client The client that has terminate.
     */

    protected void clientFinished (Client client) {
	ClientState cs = client.state;
	if ( debug )
	    System.out.println(client+": finished "+cs.status);
	switch(cs.status) {
	  case ClientState.C_IDLE:
	      decrIdleCount();
	      idleList.remove(cs);
	      break;
	  case ClientState.C_BUSY:
	  case ClientState.C_FREE:
	      break;
	  default:
	      String msg = (client 
			    + ": finished with unknown status "
			    + cs.status);
	      server.errlog(msg);
	      break;
	}
	cs.status = ClientState.C_FIN;
	decrClientCount();
    }

    /**
     * The client notifies the pool that is has been activated.
     * The client state object is updated to unmark the client as idle.
     * <p>This method needs not be synchronized, as it affect only the client
     * state, <em>not</em> the client list.
     * @param client The activated client.
     */

    protected void notifyUse(Client client) {
	if ( debug )
	    System.out.println(client+": used.");
	ClientState cs = client.state;
	decrIdleCount();
	idleList.remove(cs);
	cs.status = ClientState.C_BUSY;
    }

    /**
     * The client notifies the pool that it enters idle state.
     * <p>This method needs not be synchronized, as it affect only the client
     * state, <em>not</em> the client list.
     * @param client The client that is going to be idle.
     */

    protected boolean notifyIdle(Client client) {
	if ( debug ) 
	    System.out.println(client+": idle.");
	ClientState cs = client.state;
	if ( alive && incrIdleCount() ) {
	    idleList.toHead(cs);
	    cs.status = ClientState.C_IDLE;
	    return true;
	} else {
	    return false;
	}
    }

    protected void killSomeClients() {
	int count = maxFree - freeCount;
	while ( --count >= 0 ) {
	    ClientState cs = (ClientState) idleList.removeTail();
	    if ( cs != null ) {
		if ( debug )
		    System.out.println(cs.client+": kill (some-client).");
		decrIdleCount();
		cs.status = ClientState.C_KILL;
		cs.client.unbind();
	    } else {
		break;
	    }
	}
    }

    /**
     * Handle the given connection.
     * Find a free client, bind it to the given socket, and run it. If we
     * have reached our maximum allowed number of clients, kill some
     * connections.
     * <p>A client enters the LRU list (and become a candidate for kill) only
     * after it has handle one request (or if some timeout expires). This is
     * performed by the first call to <code>notifyUse</code> which will
     * silently insert the client into the LRU if it was not there already.
     * <p>This client pool does a lot of nice thinigs, but could probably be
     * implemented in a much better way (while keeping the features it has).
     * Contention arond the pool is probably concern number 1 of performances.
     * @param socket The connection to handle.
     */

    public void handleConnection (Socket socket) {
	ClientState cs = null;
	switch(loadavg) {
	  case AVG_LIGHT:
	      // Free list is non empty, be fast:
	      if ( decrFreeCount() )
		  cs = (ClientState) freeList.removeTail();
	      break;
	  case AVG_NORMAL:
	  case AVG_HIGH:
	      // Free list is non empty, but we try killing a client:
	      killSomeClients();
	      if ( decrFreeCount() )
		  cs = (ClientState) freeList.removeTail();
	      break;
	  case AVG_DEAD:
	      System.out.println("Dropping connections: "+freeCount);
	      break;
	}
	// At this point, we do have a free client, bind it:
	if ( cs != null ) {
	    cs.client.bind(socket);
	    cs.status = ClientState.C_BUSY;
	} else {
	    try {
		socket.close();
	    } catch (IOException ex) {
	    }
	}
	return;
    }

    protected synchronized void killClients(boolean force) {
	alive = false;
	// Kill all clients (first shot):
	ClientState cs = csList;
	while ((cs != null) && (cs.client != null)) {
	    // Only if a client is idely read'ing its socket, do we close it
	    cs.client.kill(cs.status == ClientState.C_IDLE);
	    cs = cs.csnext;
	}
	// Kill all clients (second shot):
	// Some client may be in transition during first shot, second shot
	// really kills everything.
	try {
	    Thread.sleep(5000);
	} catch (Exception ex) {
	}
	cs = csList;
	while ((cs != null) && (cs.client != null)) {
	    // Only if a client is idely read'ing its socket, do we close it
	    cs.client.kill(true);
	    cs = cs.csnext;
	}
    }

    /**
     * Shutdown the client pool. 
     * If force is <strong>true</strong>, kill all running clients right
     * now, otherwise, wait for them to terminate gracefully, and return
     * when done.
     * @param force Should we interrupt running clients.
     */

    public void shutdown (boolean force) {
	// First stage: kill all clients (synchronized)
	killClients(force) ;
	// Second stage (unsynchronized), join all client threads
	ClientState cs = csList;
	while ((cs != null) && (cs.client != null)) {
	    if ( debug )
		System.out.println(cs.client+": join."); 
	    try {
		cs.client.getThread().join();
		cs = cs.csnext;
	    } catch (InterruptedException ex) {
	    }
	}
	// Some cleanup (helps the GC, and make sure everything is free)
	props.unregisterObserver(this);
	props    = null;
	csList   = null;
	freeList = null;
	idleList = null;
	server   = null;
    }

    ClientPool (httpd server) {
	this.server = server ;
	this.props  = server.getProperties() ;
	this.props.registerObserver (this) ;
	// Initialize parameters from properties:
	this.minFree = props.getInteger(server
					, MINSPARE_FREE_P
					, MINSPARE_FREE);
	this.maxFree = props.getInteger(server
					, MAXSPARE_FREE_P
					, MAXSPARE_FREE);
	this.maxIdle = props.getInteger(server
					, MAXSPARE_IDLE_P
					, MAXSPARE_IDLE);
	this.maxThreads = props.getInteger(server
					   , MAXTHREADS_P
					   , MAXTHREADS);
	// Create the LRU lists:
	idleList = new AsyncLRUList();
	freeList = new AsyncLRUList();
	// Create the full client list:
	csList = new ClientState();
	// Create all our clients:
	for (int i = 0 ; i < maxThreads ; i++) {
	    if ( addClient(true) == null )
		throw new RuntimeException (this.getClass().getName()
					    + "[construstructor]"
					    + ": unable to create clients.");
	}
	// new DebugThread(this).start();
    }

}
