// Client.java
// $Id: Client.java,v 1.21 1996/10/02 19:52:01 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.http ;

import java.io.* ;
import java.net.*;
import java.util.*;

import w3c.jigsaw.resources.* ;
import w3c.tools.timers.* ;
import w3c.www.mime.*;
import w3c.www.http.*;

class TimeoutEvent {
    Client client ;
    TimeoutEvent (Client client) {
	this.client = client ;
    }
}

class ConnectionTimeout extends TimeoutEvent {
    ConnectionTimeout (Client client) {
	super (client) ;
    }
}

class RequestTimeout extends TimeoutEvent {
    RequestTimeout (Client client) {
	super (client) ;
    }
}

class MimeClientFactory implements MimeParserFactory {
    Client client = null;

    public MimeHeaderHolder createHeaderHolder(MimeParser parser) {
	return new Request(client, parser);
    }

    MimeClientFactory(Client client) {
	this.client = client;
    }
}

/**
 * This class implements the object that handles client connections. 
 * One such object exists per open connections at any given time.
 * <p>The basic architecture is the following: the httpd instance accepts 
 * new connections on its port. When such a connection is accepted a Client 
 * object is requested to the client pool (which can implement what ever 
 * strategy is suitable). Each request is than managed by looking up an
 *  resource and invoking the resource methods corresponding to the request.
 * @see w3c.jigsaw.http
 * @see w3c.jigsaw.http.Request
 * @see w3c.jigsaw.http.Repy
 */

public class Client implements EventHandler, Runnable {

    private MimeClientFactory factory = null;
    private Reply contreply = null;

    /**
     * This client data shuffler, <strong>null</strong> if none.
     */
    private Shuffler shuffler = null ;
    /**
     * Should this client emit debugging traces ?
     */
    private boolean debug = false;

    private long       tstart   = -1 ;		// start time for this client
    private byte       buffer[] = null ;	// output buffer
    private Object     timer  	= null ;
    private boolean    alive    = true ;
    private ClientPool pool     = null ;
    private short      major    = -1;		// Major HTTP version here
    private short      minor    = -1;		// Major HTTP version here

    protected httpd                server = null ;
    protected MimeParser           parser = null ;
    protected Thread               thread = null ;	
    protected InputStream          input  = null ;
    protected DataOutputStream     output = null ;
    private   BufferedOutputStream bufout = null;
    protected Socket               socket = null ;

    ClientState state = null;

    /**
     * Get the client identifier.
     */

    public int getIdentifier () {
	return state.id;
    }

    /**
     * Get this client input stream.
     */

    public InputStream getInputStream() {
	return input ;
    }

    /**
     * get this client output stream.
     */

    public OutputStream getOutputStream() {
	return output ;
    }

    /**
     * Get the client server's
     */

    public final httpd getServer () {
	return server ;
    }

    /**
     * Get the client thread.
     */

    public Thread getThread () {
	return thread ;
    }

    /**
     * Get the IP address of this client.
     * @return InetAddress
     */

    public InetAddress getInetAddress () {
	return socket.getInetAddress() ;
    }

    /**
     * Get this client output file descriptor.
     * This is used by the shuffler, when possible.
     */
    
    public FileDescriptor getOutputFileDescriptor() 
	throws IOException
    {
	return ((SocketOutputStream) socket.getOutputStream()).getFD() ;
    }

    public void flushOutput () 
	throws IOException
    {
	output.flush() ;
    }

    /**
     * Print a client. 
     * This methods print the best identifier for clients. It
     * includes the client thread (if possible) and its identifier with regard
     * to the server context.
     * <p>If no thread is attached to the client (which means the client has
     *  been * interrupted but has not been killed), than a <b>z</b> for 
     * zombie is printed.
     */

    public String toString() {
	if ( thread == null ) 
	    return "<z" + getIdentifier() + ">" ;
	else
	    return "<c"+ getIdentifier() +";"+thread.getName()+">" ;
    }

    /**
     * Make this client emit a trace. 
     * This function will take care of preceeding any traces with a client 
     * uniq identifier. All traces are managed by the <strong>httpd</strong>
     * instances that created the client.
     * @param msg The message to emit.
     * @se w3c.jigsaw.core.httpd
     */

    public void trace (String msg) {
	server.trace (this, msg) ;
    }

    /**
     * Emit an error on behalf of this client.
     * Error reporting is managed by the instance of <strong>httpd</strong>
     * that created this client.
     * @param msg The error message.
     * @see w3c.jigsaw.core.httpd
     */

    public void error (String msg) {
	server.errlog (this, msg) ;
    }

    /**
     * Log the given request.
     * Logging is delegated to the <strong>httpd</strong> instance that craetd
     * this client.
     * @param request The request to log.
     * @param reply Its corresponding reply.
     * @param nbytes The number of bytes sent to the client.
     * @param duration The time it took to process the request.
     */
    
    public void log (Request request, Reply reply, int nbytes, long duration) {
	server.log (this, request, reply, nbytes, duration) ;
    }

    /**
     * Handle timer events. 
     * For the time being, timer events are only used
     * to detect an overrunning request, so this handler just kills the 
     * correponding client.
     * <p>The timers package was written by Jonathan Payne.
     * @param data The timer closure.
     * @param time The absoliute time at which the event was triggered.
     * @see w3c.timers.EventManager
     * @see w3c.timers.EventHandler
     */

    public synchronized void handleTimerEvent (Object data, long time) {
	timer = null ;
	if ( data instanceof ConnectionTimeout ) {
	    // Time to abort the connection. Let the client threads do this
	    // by posting an exception to it
	} else if ( data instanceof RequestTimeout ) {
	    // This request has taken to long to fullfill, abort it
	    Reply abort = new Reply (this) ;
	    abort.setStatus (HTTP.REQUEST_TIMEOUT) ;
	}
	terminateConnection() ;
    }

    /**
     * Sets this client next timer. 
     * A client can only handle a single pending timeout, so setting a 
     * new timeout to some client cancels any previous pending timeout.
     * @param ms The number of milliseconds after which the timer should
     *    expire.
     * @param data The call data for the tevent timer handler.
     */

    public synchronized void setTimeout (int ms, Object data) {
	if ( timer != null ) {
	    server.timer.recallTimer (timer) ;
	    timer = null ;
	}
	timer = server.timer.registerTimer (ms, this, data) ;
    }
	
    /**
     * Get the next avaiable request for this client.
     * This is the first stage in actually processing a client connection. This
     * method sets a client timeout to detect idle clients connection. When
     * trigered this timeout will break the client's connection, which will
     * make the client thread exits from this method with a ClientException.
     * @return A Request instance, or <strong>null</strong> if the connection
     * was closed prematurely.
     * @exception ClientException If the request couldn't be parsed, or
     *    if the input stream was faulty.
     */

    protected Request getNextRequest () 
	throws ClientException
    {
	Request request = null ;
	
	try {
	    request = (Request) parser.parse();
	} catch (HttpParserException ex) {
	    // The connection has probably been closed prematurely:
	    return null;
	} catch (IOException ex) {
	    if ( debug ) 
		ex.printStackTrace();
	    throw new ClientException(this, ex);
	} catch (MimeParserException ex) {
	    if ( debug )
		ex.printStackTrace() ;
	    throw new ClientException (this, ex);
	}
	if ( debug ) 
	    request.dump(System.out) ;
	return request ;
    }

    /**
     * Process a request.
     * This methods processs the request to the point that a reply is 
     * available. This methods sets a timeout, to limit the duration of this 
     * request processing. This timeout overrides the getRequest timeout.
     * @param request The request to process.
     * @exception ClientException If either the timeout expires or the entity
     *     was unable to handle the request.
     */

    protected Reply processRequest (Request request) 
	throws ClientException
    {
	Reply        reply    = null ;
	setTimeout(server.getRequestTimeOut(), new RequestTimeout(this)) ;
	try {
	    reply = server.perform(request);
	} catch (HTTPException ex) {
	    if ( debug )
		ex.printStackTrace() ;
	    if ( ex.hasReply () )
		return ex.getReply() ;
	    else
		throw new ClientException (this, ex) ;
	}
	if ( reply == null ) {
	    String errmsg = "target resource emited a null Reply.";
	    throw new ClientException (this, errmsg);
	}
	return reply ;
    }

    /**
     * Should we keep alive for the given coupl of request and reply.
     * Test wether we can keep the connection alive, after the given
     * reply has been emited. This method distinguishes wether we are
     * acting as a proxy, or as a simple client, and calls the appropriate
     * method to finish the job.
     * @param request The request to examine.
     * @param reply Its computed reply.
     */

    protected boolean tryKeepConnection (Request request, Reply reply) {
	boolean keep = server.getClientKeepConnection() ;
	// The server doesn't want any keep connection
	if ( ! keep )
	    return false ;
	return request.canKeepConnection() && reply.tryKeepConnection();
    }
	
    /**
     * Emit the reply stream, using chunk encoding.
     * The client buffer should be allocated at this point.
     * @param is The stream to emit until EOF
     */

    protected int chunkTransfer(InputStream is)
	throws IOException
    {
	byte   crlf[]    = { ((byte) 13), ((byte) 10) } ;
	byte   bheader[] = new byte[32] ;
	int    blen      = 0 ;
	int    written   = 0 ;
	int    got       = 0 ;
	String header    = null ;

	try {
	    // Emit the reply stream:
	    while ((got = is.read(buffer)) > 0) {
		// Emit a full chunk: header first, followed by the body
		header = Integer.toString(got, 16).toUpperCase() + "\r\n" ;
		blen   = header.length() ;
		header.getBytes(0, blen, bheader, 0) ;
		output.write(bheader, 0, blen) ;
		output.write(buffer, 0, got) ;
		output.write(crlf, 0, 2) ;
		output.flush() ;
		written += (blen+got) ;
	    }
	} catch (IOException ex) {
	    // To cope with Java's exec bug 
	    // Anyway, if this really fails, the output will fail below too
	}
	// Emit the 0 chunk:
	header = "0\r\n" ;
	blen   = header.length() ;
	header.getBytes(0, blen, bheader, 0) ;
	output.write(bheader, 0, blen) ;
	output.write(crlf, 0, 2) ;
	output.flush() ;
	return written + blen ;
    }

    /**
     * Emit the given reply to the client.
     * @param reply The reply to be emited.
     * @return The number of body bytes emited.
     */

    protected int emitReply (Reply reply) 
	throws IOException
    {
	boolean chunkable = false ;
	// Emit the reply if needed:
	InputStream is = reply.openStream() ;
	if ( is == null ) {
	    if ( debug )
		reply.dump(System.out);
	    reply.emit(output);
	    return 0;
	} else {
	    chunkable = reply.canChunkTransfer() ;
	    if ( debug )
		reply.dump(System.out);
	    if ( reply.getStatus() != HTTP.NOHEADER )
		reply.emit(output) ;
	}
	// Try delegating this task to our possible shuffler:
	if ( shuffler != null ) {
	    int written = shuffler.shuffle (this, reply) ;
	    if ( written >= 0 ) {
		is.close() ;
		return written ;
	    }
	}
	// No shuffler available, perform the job ourselves.
	if ( buffer == null )
	    buffer = new byte[server.getClientBufferSize()] ;
	// Check if we can chunk the reply back:
	int written = 0 ;
	if ( chunkable ) {
	    written = chunkTransfer(is) ;
	} else {
	    int got = 0 ;
	    while ((got = is.read(buffer)) > 0) {
		output.write (buffer, 0, got) ;
		written += got ;
	    }
	}
	is.close() ;
	return written ;
    }

    /**
     * The public way of emitting replies.
     * Normally, a resource shouldn't access any client data, <em>however</em>
     * support for status code in the <string>100</strong> range requires that
     * a resource be able to emit a <em>Continue</em> reply before 
     * successfully processing a two phase method (see section 8.2 of HTTP/1.1
     * specification).
     * <p>This method allows a resource to send a reply straight to the client.
     * If the Client protocol version doesn't allow for such asynchronous 
     * replies, this method will return without emitting anything.
     * @param reply The reply to emit.
     * @return The number of bytes of content emited, or <strong>-1</strong>
     * if the client protocol version doesn't allow for such nifty things.
     * @exception IOException If emitting the reply failed.
     */

    public int sendReply(Reply reply) 
	throws IOException
    {
	if ((major >= 1) && (minor >= 1)) {
	    int len = emitReply(reply);
	    output.flush();
	    return len;
	} 
	return -1;
    }

    public int sendContinue() 
	throws IOException
    {
	if ((major >= 1) && (minor >= 1)) {
	    if ( contreply == null )
		contreply = new Reply(this, major, minor, HTTP.CONTINUE);
	    int len = emitReply(contreply);
	    output.flush();
	    return len;
	}
	return -1;
    }

    /**
     * Request get processed in three stages:
     * a) Read and pre-parse the request: in this stage, the parser throws
     *    HTTPParserException only.
     * b) Got the request, now handle it. In this stage only HTTPException 
     *    can occur.
     * c) Send back output, in this stage IOException only can occur.
     *
     * After this, we log the request and see if the client supports keep 
     * connection.
     */

    protected void loop () 
	throws ClientException
    {
	boolean   keep      = true ;
	long      tstart    = 0 ;
	long      tend      = 0 ;
	Request   request   = null ;
	Reply     reply     = null ;
	int       sent      = 0 ;
	int       processed = 0;

	try {
	alive_loop:
	    while ( alive && keep ) {
		// Get the next available request, and  mark client as used:
		if ( processed == 0 ) {
		    // Always run for the first request, update client's infos
		    if ((request = getNextRequest()) == null )
			return;
		    major = request.getMajorVersion();
		    minor = request.getMinorVersion();
		} else {
		    if ( ! pool.notifyIdle(this) ) 
			return;
		    if ((request = getNextRequest()) == null) 
			return;
		    pool.notifyUse(this) ;
		}
		// Process request, and time it:
		tstart  = System.currentTimeMillis() ;
		reply   = processRequest (request) ;
		if ( keep )
		    keep = tryKeepConnection (request, reply) ;
		sent    = emitReply(reply) ;
		tend    = System.currentTimeMillis() ;
		// Can we keep alive ?
		if ( keep )
		    output.flush() ;
		log (request, reply, sent, tend-tstart) ;
		processed++;
	    }
	} catch (IOException ex) {
	    if ( debug )
		ex.printStackTrace();
	    // Close any stream pending
	    try {
		InputStream in = null;
		if ( reply != null ) {
		    if ((in = reply.openStream()) != null)
			in.close();
		}
	    } catch (IOException ioex) {
	    }
	    throw new ClientException(this, ex);
	} catch (ClientException ex) {
	    if ( debug )
		ex.printStackTrace();
	    try {
		InputStream in = null;
		if ( reply != null ) {
		    if ((in = reply.openStream()) != null)
			in.close();
		}
	    } catch (IOException ioex) {
	    }
	    throw ex;
	}
	// At that point, we are leaving the loop normally
	// However, if this is a request with body, we still want to make sure
	// we got the body, due to some RST packets 
	if ((request != null)
	    && (request.getContentLength() > 0)
	    && (reply != null)
	    && (reply.getStatus() / 100 != 2)) {
	    // The request may have failed...
	    try {
		InputStream in = request.getInputStream();
		if ( in != null ) {
		    while (in.available() > 0)
			in.read(buffer, 0, buffer.length);
		}
	    } catch (Exception ex) {
	    }
	}
    }

    /**
     * Terminate the connection that the client is currently handling.
     * We clear up all our socket related resources, and hope that the
     * client thread will get some exception, if it is blocked on some
     * input/outputs.
     */

    protected synchronized void terminateConnection () {
	if ( socket == null )
	    return ;
	// remove timer:
	if ( timer != null )
	    server.timer.recallTimer (timer) ;
	// close all streams
	try {
	    if ( output != null ) {
		output.flush() ;
		if ( bufout != null )
		    bufout.flush();
	    }
	} catch (IOException ex) {
	}
	try {
	    socket.close() ;
	} catch (IOException ex) {
	}
	socket = null ;
	input  = null ;
	output = null ;
	bufout = null;
	parser = null ;
	// Reset HTTP protocol version:
	major = -1;
	minor = -1;
    }

    /**
     * Starts this client.
     * Any running client waits until it is bound to a socket. Once this happen
     * it runs the connection and return to its waiting state.
     */

    public void run () {
	try {
	    while ( alive ) {
		loopForBinding () ;
		if ( alive && (socket != null) && ( ! runConnection()) ) 
		    break;
	    }
	    pool.clientFinished (this) ;
	} catch (Exception ex) {
	    if ( debug )
		ex.printStackTrace();
	} 
    }

    /**
     * Run for our newly attached connection.
     */

    protected boolean runConnection () {
	this.tstart = System.currentTimeMillis() ;
	try {
	    output   = (new DataOutputStream
			(bufout = new BufferedOutputStream 
			              (socket.getOutputStream()))) ;
	    input  = (new BufferedInputStream (socket.getInputStream())) ;
	    parser = new MimeParser(input, factory) ;
	    loop () ;
	} catch (ClientException ex) {
	    if ( debug )
		ex.printStackTrace();
	    // Emit some debugging traces:
	    if ( debug ) {
		if (ex.ex != null )
		    ex.ex.printStackTrace() ;
		else
		    ex.printStackTrace();
	    }
	    // If output is null, we have killed the connection...
	    if ( output != null ) {
		error ("caught ClientException: [" 
                       + ex.getClass().getName()
		       + "] " + ex.getMessage()) ;
	    }
	} catch (Exception e) {
	    if ( debug )
		e.printStackTrace() ;
	    error ("caught exception [" 
		   + e.getClass().getName()
		   + "] " + e.getMessage()) ;
	} 
	terminateConnection() ;
	return pool.clientConnectionFinished(this) ;
    }

    /**
     * Bind this awaiting client to the provided socket.
     * @param s The socket this client should now handle.
     */

    public synchronized void bind (Socket s) {
	this.socket = s ;
	notify () ;
    }

    /**
     * Is this client bound to some socket already ?
     */

    public synchronized boolean isBound () {
	return socket != null ;
    }

    /**
     * Wait for this client to be bind to some socket.
     */

    public synchronized void loopForBinding () {
	while ( alive ) {
	    // Wait until we get some socket:
	    while ( socket == null ) {
		try {
		    wait() ;
		} catch (InterruptedException ex) {
		}
		if ( ! alive )
		    return ;
	    }
	    return ;
	}
    }

    public synchronized void unbind() {
	terminateConnection();
	notify();
    }

    /**
     * Kill this client, freeing all its associated resources.
     * @param force If <strong>true</strong>, kill the client even if it
     *     is handling a connection, otherwise, notify the client to terminate.
     */

    public synchronized void kill (boolean force) {
	alive = false ;
	if ( force && (socket != null) ) 
	    terminateConnection() ;
	notify() ;
    }

    /**
     * Create an empty client, that will be ready to work.
     * The created client will run and wait for it to be <code>bind</code>
     * to some socket before proceeding.
     * @param server The server to which this client is attached.
     * @param id The client identifier.
     * @see w3c.jigsaw.core.ClientPool
     */

    Client (httpd server, ClientPool pool, ClientState state) {
	this.factory  = new MimeClientFactory(this);
	this.socket   = null ;
	this.pool     = pool ;
	this.debug    = server.getClientDebug() ;
	this.server   = server ;
	this.state    = state;
	this.shuffler = server.getShuffler(this) ;
	this.thread   = new Thread (this) ;
	this.thread.setPriority (server.getClientThreadPriority()) ;
	this.thread.setName ("client"+getIdentifier()) ;
	this.thread.start () ;
    }


}
