// ServerHandlerManager.java
// $Id: ServerHandlerManager.java,v 1.3 1996/09/23 13:57:26 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.daemon;

import java.util.*;
import java.io.*;

/**
 * A ServerHandlerManager instance manages a set of ServerHandler.
 */

public class ServerHandlerManager {
    /**
     * The property containing the servers to be launched at startup time.
     * This property is a <code>|</code> separated list of server identifiers.
     * Declaring a server to this list requires that either:
     * <ul>
     * <li>An appropriate
     * <code>w3c.jigsaw.daemon.</code><em>identifier</em>.<code>class</code>
     * is declared and specify the class of the server to be launched (this 
     * class should implement the ServerHandler interface.).
     * <li>An appropriate
     * <code>w3c.jigsaw.daemon.</code><em>identifier</em>.<code>clones</code>
     * is declared and its value specify an existing server to be cloned in 
     * order to create the new server.
     * </ul>
     * @see ServerHandler.
     */
    protected static final String HANDLERS_P = "w3c.jigsaw.daemon.handlers";
    /**
     * The server handler property <code>class</code> suffix.
     */
    protected final static String CLASS_P = "w3c.jigsaw.daemon.class";
    /**
     * The server handler property <code>clones</code> prefix.
     */
    protected final static String CLONES_P = "w3c.jigsaw.daemon.clones";

    /**
     * The Application-Wide server manager.
     */
    protected static ServerHandlerManager manager = null;
    /**
     * The list of running server handlers.
     */ 
    protected Hashtable handlers = null;
    /**
     * The server handler manager property list.
     */
    protected DaemonProperties props = null;

    /**
     * Emit a non-fatal error.
     * @param msg The message to emit.
     */

    protected void error(String msg) {
	System.err.println(msg);
    }

    /**
     * Emit a fatal error.
     * This will abort the whole process !
     * @param msg The fata error message.
     */

    protected void fatal(String msg) {
	System.out.println("*** FATAL Error:");
	System.out.println(msg);
	System.exit(1);
    }

    /**
     * Launch a new server handler.
     * This method tries to launch a new server handler. If launching
     * succeeds, it returns happily, otherwise, it emits an error message
     * to the standard error stream.
     * @param identifier The identifier of the server handler to be launched.
     * @param props The properties from which the server handler should
     * initialize itself.
     */

    protected void launchServerHandler(String id, DaemonProperties props) {
	ServerHandler server = null;
	String        cls    = props.getString(id, CLASS_P, null);
	if ( cls != null ) {
	    // This is a fresh new server handler:
	    try {
		Class c = Class.forName(cls);
		server  = (ServerHandler) c.newInstance();
		server.initialize(id, props);
	    } catch (ClassNotFoundException ex) {
		error("Unable to launch "+id+": class "+cls+" not found.");
		return;
	    } catch (InstantiationException ex) {
		error("Unable to launch "+id+": unable to instanciate "+cls);
		return;
	    } catch (IllegalAccessException ex) {
		error("Unable to launch "+id+": illegal access to "+cls);
		return;
	    } catch (ServerHandlerInitException ex) {
		error("Unable to launch "+id+": "+ex.getMessage());
		return;
	    }
	} else {
	    // It may be a clone of some already existing server handler:
	    String clones = props.getString(id, CLONES_P, null);
	    if ( clones == null ) {
		error("Unable to launch "+id+": no class or clones defined.");
		return;
	    }
	    // Lookup the server handler to be cloned, clone it 
	    server = lookupServerHandler(clones);
	    if ( server == null ) {
		error("Unable to launch "+id+": "+clones+" doesn exit.");
		return;
	    }
	    try {
		server = server.clone(id, props);
	    } catch (Exception ex) {
		error("Unable to launch "+id+": clone failed on "+server+"\r\n"
		      + ex.getMessage());
		return;
	    }
	}
	// Register the created server:
	handlers.put(id, server);
    }

    /**
     * Get the server manager for this Java process.
     * If not already created, the application-wide server handler manager
     * is created and initialized from the set of global properties.
     * @return The application-wide ServerHandlerManager instance.
     */

    public static synchronized ServerHandlerManager getServerHandlerManager() {
	if ( manager == null ) 
	    manager = new ServerHandlerManager(System.getProperties());
	return manager;
    }

    /**
     * Lookup the server handler having the given identifier.
     * @param id The identifier of the server handler to look for.
     * @return A ServerHandler instance, or <strong>null</strong> if
     * undefined.
     */

    public ServerHandler lookupServerHandler(String id) {
	return (ServerHandler) handlers.get(id);
    }

    /**
     * Create and initialize a fresh server handler manager.
     * Each server handler declared in the properties is launched in turn.
     * If no server handlers is declared, or if none of them is initializable
     * the server manager is not created and a RuntimeException is thrown,
     * otherwise, if at least one server handler was initialized, the server
     * manager emits appropriate error messages to the error stream for each
     * of the server handlers whose launch failed.
     * @param props The properties this manager should be initialized from.
     * @exception RuntimeException If no server handlers was declared through
     * the properties.
     */

    ServerHandlerManager (Properties p) {
	// Initialize instance variables:
	this.handlers = new Hashtable(7);
	this.props    = (DaemonProperties) p;
	// Basically launch all requested servers, emiting errors if needed:
	String hprop = props.getProperty(HANDLERS_P);
	if ((hprop == null) || hprop.equals(""))
	    fatal("The property [w3c.jigsaw.daemon.handlers] is undefined.");
	StringTokenizer st = new StringTokenizer(hprop, "|");
	while (st.hasMoreTokens()) {
	    // Initialize and register this new server:
	    String id = st.nextToken();
	    launchServerHandler(id, props);
	}
	// Check that we launched at least one server
	if ( handlers.size() <= 0 )
	    fatal("No servers launched !");
    }

    public static void main(String args[]) {
	Properties props = new DaemonProperties(System.getProperties());
	if ( args.length > 0 ) {
	    try {
		props.load(new BufferedInputStream
			   (new FileInputStream(args[0])));
	    } catch (Exception ex) {
		ex.printStackTrace();
	    }
	    System.setProperties(props);
	}
	ServerHandlerManager manager = getServerHandlerManager();
    }

}
