// ClientFactory.java
// $Id: ClientFactory.java,v 1.1 1996/07/20 00:03:50 abaird Exp $ 
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.daemon;

import java.net.*;
import java.io.*;

/**
 * The ClientFactory interface describes a server's client pool management.
 * Lying in between a Server instance and a set of Client instances, the
 * ClientFactory is responsible for creating and managing some mapping
 * between incomming connections and <em>clients</em>. Each client instance
 * can be thought of as a specific protocol handler.
 * <p>This interface describes the relationship between the client factory
 * and it's associated server, and between the client factory and the client
 * instances it has created.
 */

public interface ClientFactory {
    
    /**
     * Server side interface - Handle this incomming connection.
     * If some error occurs during the processing of this new connection,
     * it is up to the factory to decide if emiting some error is appropriate.
     * @param socket The connection that has been freshly accepted.
     */

    public void handleConnection(Socket s);

    /**
     * Client side interface - This client is now idle, and interruptible.
     * The client we created to handle some connection is now back to idle
     * and is willing to be reused, it's the right time to keep track of it.
     * @param client The client that gives us the notification.
     */

    public void notifyIdle(ServerClient client);

    /**
     * Client side interface - This client is now in use, don't interrupt it !
     * This client is now processing some requests, it don't want to be
     * interrupted (ie killed) until it is back to idle again.
     * <p>A buzy client can still be killed ungracefully by using its
     * <code>kill</code> method with <strong>true</strong> as the parameter.
     * @param client The client that don't want to be disturbed.
     */

    public void notifyUse(ServerClient client);

    /**
     * Client side interface - This client has been killed.
     * The client has either encountered an abnormal situation, or has been
     * kill by the factory, it is not reusable any more.
     * <p>Dead clients typically don't have an associated thread any more.
     * @param dead The client dying in front of the factory.
     */

    public void notifyDead(ServerClient client);
    
    
}
