// CvsDirectory.java
// $Id: CvsDirectory.java,v 1.4 1996/09/26 21:12:41 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.cvs ;

import java.io.* ;
import java.util.* ;

/**
 * This class filters directory listings for CvsDirectory instances.
 */

class CvsDirectoryFilter implements FilenameFilter {
    
    /**
     * Should we accept this file in the CvsDirectory listing ?
     * @param directory The directory being listed.
     * @param name The file being checked.
     * @return <strong>true</strong> if the file is to be accepted.
     */

    public boolean accept (File directory, String name) {
	if (name.equals ("CVS") || name.equals ("core") || name.endsWith("~"))
	    return false ;
	File file = new File (directory, name) ;
	return ! file.isDirectory() ;
    }
    
}

/**
 * Implements a per-directory CVS state object.
 */

public class CvsDirectory {
    /**
     * Property giving the path of the cvs binary.
     * This property should be set to the absolute path to the cvs command
     * in your local environment.
     * <p>This property defaults to <code>/usr/local/bin/cvs</code>.
     */
    public static final String CVSPATH_P = "w3c.cvs.path" ;
    /**
     * Property giving your CVS repository.
     * This property should be set to the absolute path of your repository.
     * <p>This property defaults to <code>/afs/w3.org/pub/WWW</code>.
     */
    public static final String CVSROOT_P = "w3c.cvs.root" ;
    /**
     * Property giving the path of the cvswrapper.
     * Because CVS can't run without being in the right directory, this
     * classes use a shell script wrapper to issue cvs commands, that will
     * change directory appropriately.
     * <p>You should have gotten this wrapper in the distribution 
     * <code>bin</code> directory.
     * <p>This property defaults to 
     * <code>/afs/w3.org/usr/abaird/Jigsaw/bin/cvs_wrapper</code>.
     */
    public final static String CVSWRAP_P = "w3c.cvs.wrapper" ;
     
    private static final String tmpdir = "/tmp" ;

    /**
     * The default CVS path.
     */
    public final static String cvspath_def 
        = "/usr/local/bin/cvs" ;
    /**
     * The default CVS root path.
     */
    public final static String cvsroot_def 
        = "/afs/w3.org/pub/WWW";
    /**
     * The default CVS wrapper path.
     */
    public final static String cvswrap_def 
        = "/afs/w3.org/usr/abaird/Jigsaw/bin/cvs_wrapper";

    String cvspath = cvspath_def ;
    String cvsroot = cvsroot_def ;
    String cvswrap = cvswrap_def ;

    Hashtable entries   = null ;
    File      directory = null ;

    /**
     * Dump the given string into a temporary file.
     * This is used for th <code>-f</code> argument of the cvs commit command.
     * This method should only be used from a synchronized method.
     * @param string The string to dump.
     */

    protected File temporaryFile (String string) 
	throws CvsException
    {
	File temp = new File (tmpdir, "cvs.msg") ;
	try {
	    PrintStream out  = new PrintStream (new FileOutputStream(temp)) ;
	    out.print(string) ;
	    out.close() ;
	    return temp ;
	} catch (IOException ex) {
	    error ("temporaryFile"
		   , "unable to create/use temporary file: " 
		     + temp.getAbsolutePath()) ;
	}
	return temp ;
    }

    /**
     * Log some transaction. 
     * This defaults to printing some messages to the std output.
     * @param msg The message to emit.
     */

    protected void log (String msg) {
	System.out.println (this.getClass().getName()
			    + "[" + directory + "]: "
			    + msg) ;
    }

    /**
     * Emit an error.
     * Some abnormal situation occured, emit an error message.
     * @param mth The method in which the error occured.
     * @param msg The message to emit.
     * @exception CvsException The exception that will be thrown as a 
     *     result of the error.
     */

    protected void error (String mth, String msg) 
	throws CvsException
    {
	String emsg = this.getClass().getName()+"["+mth+"]: "+msg ;
	log (emsg) ;
	throw new CvsException (emsg) ;
    }

    /**
     * Build a command vector, including the given set of targets.
     * @param cvscmd The CVS comand to prepare for.
     * @param vfiles The set of target files. If this is <strong>null</strong>
     *    include all possible files as a target.
     * @return A command vector suitable for use as an exec argument.
     */

    protected String[] getCommand (String cvscmd[], CvsEntry targets[]) {
	String cmd[] = new String[7+cvscmd.length+targets.length] ;
	int    ptr   = 0 ;
	cmd[ptr++] = cvswrap ;
	cmd[ptr++] = "-directory" ;
	cmd[ptr++] = directory.getAbsolutePath() ;
	cmd[ptr++] = cvspath ;
	cmd[ptr++] = "-q" ;
	cmd[ptr++] = "-d" ;
	cmd[ptr++] = cvsroot ;
	for (int i = 0 ; i < cvscmd.length ; i++)
	    cmd[ptr++] = cvscmd[i] ;
	for (int i = 0 ; i < targets.length ; i++)
	    cmd[ptr++] = targets[i].getName() ;
	for (int i = 0 ; i < cmd.length ; i++)
	    System.out.print (cmd[i]+" ") ;
	System.out.println("") ;
	return cmd ;
    }

    protected String[] getCommand (String cvscmd[], boolean all) {
	String cmd[] = new String[7+cvscmd.length+(all?entries.size():0)] ;
	int    ptr   = 0 ;
	cmd[ptr++] = cvswrap ;
	cmd[ptr++] = "-directory" ;
	cmd[ptr++] = directory.getAbsolutePath() ;
	cmd[ptr++] = cvspath ;
	cmd[ptr++] = "-q" ;
	cmd[ptr++] = "-d" ;
	cmd[ptr++] = cvsroot ;
	for (int i = 0 ; i < cvscmd.length ; i++)
	    cmd[ptr++] = cvscmd[i] ;
	if ( all ) {
	    Enumeration e = entries.elements() ;
	    while ( e.hasMoreElements() )
		cmd[ptr++] = ((CvsEntry) e.nextElement()).getName();
	}
	for (int i = 0 ; i < cmd.length ; i++)
	    System.out.print (cmd[i]+" ") ;
	System.out.println("") ;
	return cmd ;
    }

    /**
     * Read the given input stream as a text.
     * @param in The input stream to read from.
     * @param into The StringBuffer to fill.
     * @return The provided StringBuffer, filled with the stream content.
     */

    private StringBuffer readText(InputStream procin, StringBuffer into) 
	throws IOException 
    {
	DataInputStream in   = new DataInputStream(procin) ;
	String          line = null ;

	while ((line = in.readLine()) != null) 
	    into.append (line+"\n") ;
	return into ;
    }

    /**
     * Run a cvs command, return the process object.
     * @exception CvsException If the process couldn't be launched.
     */

    protected Process runCvsProcess (String args[]) 
	throws IOException
    {
	return Runtime.getRuntime().exec (args) ;
    }

    /**
     * Wait for the CVS process completion.
     * @param proc The underlying CVS process.
     * @param ccode Check the returned status of the command, and throw an
     *    exception if none zero.
     * @exception CvsException If the exit code is not 0.
     */

    private void waitForCompletion (Process proc, boolean ccode)
	throws CvsException 
    {
	while ( true ) {
	    try {
		// Try reading the error stream:
		StringBuffer sb = new StringBuffer() ;
		try {
		    sb = readText(proc.getErrorStream(), sb) ;
		} catch (Exception ex) {
		}
		// Check ecode if requested to do so:
		proc.waitFor() ;
		int ecode = proc.exitValue() ;
		if ( ecode != 0 ) {
		    String msg = "Process exited with error code: " + ecode ;
		    log (msg) ;
		    if ( ccode )
			throw new CvsException (msg) ;
		} 
		return ;
	    } catch (InterruptedException e) {
	    }
	}
    }

    /**
     * Add an entry into this directory for the given file.
     * @param name The file's name.
     */

    protected void addEntry (String name) {
	CvsEntry entry = new CvsEntry (this, name, CvsEntry.STATUS_Q) ;
	entries.put (name, entry) ;
    }

/*
 * Status handling
 * ===============
 */

    private int parseStatusStatus(String line) 
	throws CvsException
    {
	int ptr = line.lastIndexOf("Status: ") ;
	if ( ptr >= 0 ) {
	    ptr += "Status: ".length() ;
	    String strstatus = line.substring(ptr) ;
	    if (strstatus.equals("Unknown")) {
		return CvsEntry.STATUS_Q ;
	    } else if (strstatus.equals ("Up-to-date")) {
		return CvsEntry.STATUS_OK ;
	    } else if (strstatus.equals ("Locally Removed")) {
		return CvsEntry.STATUS_R ;
	    } else if (strstatus.equals ("Locally Modified")) {
		return CvsEntry.STATUS_M ;
	    } else if (strstatus.equals ("Locally Added")) {
		return CvsEntry.STATUS_A ;
	    } else if (strstatus.equals ("Needs Checkout")) {
		return CvsEntry.STATUS_U ;
	    } else if (strstatus.equals ("Needs Merge")) {
		return CvsEntry.STATUS_C ;
	    } else if (strstatus.equals("Needs Patch")) {
		return CvsEntry.STATUS_P;
	    }
	}
	error ("parseStatusStatus", "Unknown status in: [" + line + "]");
	return -1 ;	// not reached
    }

    private String parseStatusVersion (String line) 
	throws CvsException
    {
	if ( line.endsWith ("No revision control file") )
	    return null ;
	if ( ! line.endsWith (",v") )
	    error ("parseStatusVerion"
		   , "Line doesn't end properly: [" + line + "]") ;
	int len = line.length() - 2 ;
	int ptr = len ;
	while (--ptr >= 0) {
	    if ( line.charAt(ptr) == '/' ) 
		return line.substring(ptr+1, len) ;
	}
	error ("parseStatusVersion", "Unable to find filename in ["+line+"]") ;
	return null ;	// not reached
    }

    private void parseStatusOutput (InputStream procin
				    , String files[]
				    , int offset) 
	throws CvsException, IOException
    {
	DataInputStream in   = new DataInputStream (procin) ;
	String          line = null ;
	while ( (line = in.readLine()) != null ) {
	    // Synchronize to per entry output:
	    while ( ! line.startsWith ("==================================")) {
		// We are out of sync, skip lines until we sync again:
		if ((line = in.readLine()) == null)
		    return ;
	    }
	    // Get the entry status, and full file name:
	    int status = parseStatusStatus (in.readLine()) ; 
	    in.readLine() ;
	    in.readLine() ;
	    String filename = parseStatusVersion (in.readLine()) ;
	    // This happens when the file has been locally added.
	    if ( filename == null ) {
		filename = files[offset] ;
	    } else if ( ! filename.equals(files[offset]) ) {
		log ("inconsistent filenames: "+filename+","+files[offset]);
	    }
	    offset++ ;
	    // Update the target entry:
	    CvsEntry entry  = (CvsEntry) entries.get (filename) ;
	    if ( entry == null ) {
		// Some problem here !
		log (filename+": has no entry.") ;
	    } else {
		entry.status = status ;
	    }
	}
    }

    /**
     * Status each entry in this directory.
     * @exception CvsException If someting went wrong.
     */

    public synchronized void status () 
	throws CvsException
    {
	String cmd[]  = { "status" } ;
	String args[] = getCommand (cmd, true) ;
	// Run the process:
	Process cvs = null ;
	try {
	    cvs = runCvsProcess (args) ;
	} catch (IOException ex) {
	    error ("status", "unable to launch cvs command: "+ex.getMessage());
	}
	try {
	    parseStatusOutput (cvs.getInputStream(), args, 8) ;
	} catch (IOException ex) {
//	    error ("status", "unable to read process input:"+ex.getMessage()) ;
	}
	waitForCompletion (cvs, true) ;
    }

/*
 * Update handling:
 * ================
 */

    private void parseUpdateOutput (InputStream procin) 
	throws IOException, CvsException
    {
	DataInputStream in   = new DataInputStream (procin) ;
	String          line = null ;
	while ((line = in.readLine()) != null) {
	    int status = -1 ;
	    int ch     = line.charAt(0) ;
	    // Parse the status:
	    switch (ch) {
	      case 'U': status = CvsEntry.STATUS_U ; break ;
	      case 'A': status = CvsEntry.STATUS_A ; break ;
	      case 'R': status = CvsEntry.STATUS_R ; break ;
	      case 'M': status = CvsEntry.STATUS_M ; break ;
	      case 'C': status = CvsEntry.STATUS_C ; break ;
	      case '?': status = CvsEntry.STATUS_Q ; break ;
	      default:
		  // We just ignore this right now.
		  continue ;
	    }
	    // Update the entry:
	    String   filename = line.substring(2) ;
	    CvsEntry entry    = (CvsEntry) entries.get (filename) ;
	    if ( entry == null ) {
		error ("parseUpdateOutput", "No filename in ["+line+"]") ;
	    } else {
		entry.status = status ;
	    }
	}
    }
	
    /**
     * Update the given set of files.
     * @param vnames The set of files to update, as a Vector of String.
     * @exception CvsException If something failed.
     */

    public synchronized void update (CvsEntry entries[]) 
	throws CvsException
    {
	String cmd[]  = { "update" } ;
	String args[] = getCommand(cmd, entries) ;
	// Run the command:
	Process cvs = null ;
	try {
	    cvs = runCvsProcess (args) ;
	} catch (IOException ex) {
	    error ("update", "unable to run cvs process:"+ex.getMessage());
	}
	try {
	    parseUpdateOutput (cvs.getInputStream()) ;
	} catch (IOException ex) {
//	    error ("update", "unable to read process input") ;
	}
	waitForCompletion (cvs, true) ;
	return ;
    }

/*
 * Commit handling:
 * ================
 */

    private void parseCommitOutput (InputStream procin) 
	throws IOException, CvsException
    {
	DataInputStream in   = new DataInputStream (procin) ;
	String          line = null ;
	while ((line = in.readLine()) != null) {
	    if ( ! line.startsWith ("Checking in ") )
		continue ;
	    String filename = line.substring ("Checking in ".length()
					      , line.length()-1) ;
	    CvsEntry entry  = (CvsEntry) entries.get(filename) ;
	    if ( entry == null ) {
		error ("parseCommitOutput", "No filename in ["+line+"]") ;
	    } else {
		entry.status = CvsEntry.STATUS_OK ;
	    }
	}
	return ;
    }

    /**
     * Commit the given set of files, with the provided message.
     * @param msg The commit messsage, describing changes.
     * @param entries The entries to commit.
     * @exception CvsException If commiting failed.
     */

    public synchronized void commit (String msg, CvsEntry entries[]) 
	throws CvsException
    {
	File   fmsg   = temporaryFile (msg) ;
	String cmd[]  = { "commit", "-F", fmsg.getAbsolutePath() } ;
	String args[] = getCommand(cmd, entries) ;
	Process cvs = null ;
	try {
	    cvs = runCvsProcess (args) ;
	} catch (IOException ex) {
	    fmsg.delete() ;
	    error ("commit", "unable to launch CVS process:"+ex.getMessage());
	} 
	try {
	    parseCommitOutput (cvs.getInputStream()) ;
	} catch (IOException ex) {
	    fmsg.delete() ;
//	    error ("commit", "unable to read process input:"+ex.getMessage()) ;
	}
	waitForCompletion (cvs, true) ;
	fmsg.delete() ;
	return ;
    }

/*
 * Log handling.
 */

    private void parseLogOutput (InputStream in, StringBuffer into) 
	throws IOException 
    {
	readText(in, into) ;
    }

    /**
     * Get the given entry log, as a String.
     * @param entry The entry whose log is queried.
     * @exception CvsException If getting the log failed.
     */

    public synchronized String log (CvsEntry entry) 
	throws CvsException
    {
	CvsEntry entries[] = { entry } ;
	String cmd[]       = { "log" } ;
	String args[]      = getCommand(cmd, entries) ;
	Process cvs = null ;
	try {
	    cvs = runCvsProcess (args) ;
	} catch (IOException ex) {
	    error ("log", "unable to launch CVS process:"+ex.getMessage()) ;
	}
	StringBuffer sb = new StringBuffer() ;
	try {
	    parseLogOutput(cvs.getInputStream(), sb) ;
	} catch (IOException ex) {
//	    error ("log", "unable to read process input:"+ex.getMessage()) ;
	}
	waitForCompletion (cvs, false) ;
	return sb.toString() ;
    }
    
/*
 * Diff handling.
 */

    private void parseDiffOutput (InputStream in, StringBuffer into) 
	throws IOException 
    {
	readText(in, into) ;
    }

    /**
     * Get the given entry diff, as a String.
     * @param entry The entry whose diff is queried.
     * @exception CvsException If getting the diff failed.
     */

    public synchronized String diff (CvsEntry entry) 
	throws CvsException
    {
	CvsEntry entries[] = { entry } ;
	String cmd[]       = { "diff" } ;
	String args[]      = getCommand(cmd, entries) ;
	Process cvs = null ;
	try {
	    cvs = runCvsProcess (args) ;
	} catch (IOException ex) {
	    error ("diff", "unable to launch CVS process:"+ex.getMessage()) ;
	}
	StringBuffer sb = new StringBuffer() ;
	try {
	    parseLogOutput(cvs.getInputStream(), sb) ;
	} catch (IOException ex) {
//	    error ("diff", "unable to read process input:"+ex.getMessage()) ;
	}
	waitForCompletion (cvs, false) ;
	return sb.toString() ;
    }
    
/*
 * Add handling:
 * =============
 */
    
    private void parseAddOutput (InputStream procin) 
	throws IOException
    {
	DataInputStream in = new DataInputStream (procin) ;
	String line = null ;
	while ((line = in.readLine()) != null)
	    ;
	return ;
    }

    /**
     * Add new entries for this directory to CVS control.
     * @param msg A message describing the entries.
     * @param entries The entries to put under CVS control.
     */

    public synchronized void add (String msg, CvsEntry entries[])
	throws CvsException
    {
	String  cmd[]  = { "add" } ;
	String  args[] = getCommand(cmd, entries) ;
	Process cvs    = null ;
	try {
	    cvs = runCvsProcess (args) ;
	} catch (IOException ex) {
	    error ("diff", "unable to launch CVS process:"+ex.getMessage()) ;
	}
	try {
	    parseAddOutput(cvs.getInputStream()) ;
	} catch (IOException ex) {
//	    error ("diff", "unable to read process input:"+ex.getMessage()) ;
	}
	waitForCompletion (cvs, true) ;
	// We update(entries) here, to update ourself to signal errors.
	update (entries) ;
	return ;
    }

    /**
     * Recompute this directory in-memory image.
     * This method will recompute the whole CVS state of the directory.
     */

    public synchronized void refresh () 
	throws CvsException
    {
	initialize() ;
    }

    /**
     * Remove an entry from CVS control.
     * This is not implemented since this would mean that the program would 
     * have to delete the file first.
     * @param entries The entries to remove from CVS control.
     */

    public synchronized void remove (CvsEntry entries[]) {
	log ("remove: not implemented.") ;
    }

    /**
     * Lookup a CVS entry by name.
     * @param The name of the entry to look for.
     * @return An instance of CvsEntry, or <strong>null</strong>.
     */

    public CvsEntry lookup (String name) {
	return (CvsEntry) entries.get (name) ;
    }

    /**
     * Get this directory working directory.
     * @return An instance of File.
     */

    public File getDirectory() {
	return directory ;
    }

    /**
     * Get all entries contained by this directory.
     * @return An enumeration, containing one element per items in the 
     * directory.
     */

    public synchronized Enumeration getEntries () {
	return entries.elements() ;
    }

    /**
     * Initialize this directory. 
     * Get the list of files potentially under CVS control, build one entry
     * for each of them, and save them into a hashtable. Than update all their
     * status to get into initial state.
     */
    
    protected void initialize ()
	throws CvsException
    {
	if ( ! directory.isDirectory() ) 
	    throw new RuntimeException (this.getClass().getName()
					+ "[initialize]: "
					+ directory.getAbsolutePath()
					+ " not a directory.") ;
	this.entries = new Hashtable() ;
	CvsDirectoryFilter filter  = new CvsDirectoryFilter () ;
	String             files[] = directory.list(filter) ;
	for (int i = 0 ; i < files.length ; i++) 
	    addEntry (files[i]) ;
	status() ;
    }

    /**
     * Print this directory current state.
     */

    public void print (PrintStream out) {
	Enumeration e = entries.elements() ;
	while ( e.hasMoreElements() ) {
	    CvsEntry entry = (CvsEntry) e.nextElement() ;
	    out.println (entry.getName() + " = " + entry.getStatus()) ;
	}
    }

    /**
     * Create an in-memory image of the CVS state of the directory.
     * This constructor will get your CVS settings from the global properties.
     * @param directory The directory to examine.
     */

    public CvsDirectory (File directory)
	throws CvsException
    {
	this (System.getProperties(), directory) ;
    }

    /**
     * Create an in-memory image of the CVS state of the directory.
     * This constructo will get your CVS settings from the provided 
     * properties.
     * @param props Were to get your CVS settings from.
     * @param directory The directory to exmaine.
     */

    public CvsDirectory (Properties props, File directory)
	throws CvsException
    {
	this (props.getProperty (CVSPATH_P, cvspath_def)
	      , props.getProperty (CVSROOT_P, cvsroot_def)
	      , props.getProperty (CVSWRAP_P, cvswrap_def)
	      , directory) ;
    }

    /**
     * Create an in-memory image of the CVS state of a directory.
     * This constructor allows you to specify explicitly the CVS settings.
     * @param cvspath The absolute path of the cvs command.
     * @param cvsroot The absolute path of the CVS repository.
     * @param cvswrap The absolute path of the CVS wrapper script.
     * @param directory The directory you want to examine.
     */

    public CvsDirectory (String cvspath
			 , String cvsroot
			 , String cvswrap
			 , File directory)
	throws CvsException
    {
	this.directory = directory ;
	this.cvspath   = cvspath ;
	this.cvsroot   = cvsroot ;
	this.cvswrap   = cvswrap ;
	this.entries   = new Hashtable() ;
	initialize() ;
    }

    public static void main (String args[]) 
	throws CvsException
    {
	File         dir = new File (args[0]) ;
	CvsDirectory cvs = new CvsDirectory(dir) ;
	cvs.print(System.out) ;
    }

}
