// ResourceSpace.java
// $Id: ResourceSpace.java,v 1.8 1997/07/30 14:03:45 ylafon Exp $  
// (c) COPYRIGHT MIT and INRIA, 1997.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.tools.resources;

import w3c.tools.resources.*;
import w3c.tools.resources.event.StructureChangedListener;

/**
 * A resource space manages a set of hierarchical resources.
 * A resource space is reponsible for a <em>space</em> of resources, in 
 * particular it <strong>must</strong> provide:
 * <dl>
 * <dt>Resource lookup
 * <dd>A way to resolve a hierarchical resource names within that space.
 * <dt>Security
 * <dd>Resource access security checks
 * <dd>A way to ensure that the resource references it creates are 
 * appropriately protected against unauthorized access.
 * </dl>
 */

public interface ResourceSpace {

  /**
   * Resolve the given <em>state</em> into a resource.
   * This metho is allowed (and should) perform any security check before 
   * granting access to the resource. Resources themselves may be used
   * to ensure finer grain access control.
   * @param state The lookup state to resolve into a resource.
   * @return A resource <em>reference</em> which guarantees correct accesses
   * to resources even if these are cached versions of what really 
   * is in some store.
   */

  public ResourceReference resolve(LookupState state);
 
  /**
   * Save the given resource in the resource space
   */
  public void save (Resource r)
    throws ChildNotSupportedException;

  /**
   * load the persistent resource indentied by the PersistentReference.
   */
  public ResourceReference load(PersistentReference pr,
				ResourceReference container) 
    throws ClassNotFoundException;


  /**
   * delete the persistent resource relative to the given resource.
   */
  public void delete (Resource r);

  public void delete (String name, ResourceReference container);

  /**
   * Check point that resource space.
   * The caller wants to be sure that any changes to that space has been
   * persistently applied, whatever this means.
   */

  /**
   * Trigger a resource unloaded event.
   * This method is called by resources themselves to notify the resource
   * space manager of the event.
   * @param resource The resource being unloaded.
   */
  public void resourceUnloaded(Resource resource);

  /**
   * Trigger a resource deleted event.
   * This method is called by resources themselves to notify the resource
   * space manager of the event.
   * @param resource The resource being deleted.
   */
  public void resourceDeleted(Resource resource);
  
  /**
   * Trigger a resource inited event.
   * This method is called by resources themselves to notify the space
   * manager of the event.
   * @param resource The resource that has been initialized.
   */
  public void resourceInited(Resource resource);

  /**
   * Trigger a resource created event.
   * This method is called by resources themselves to notify the space
   * manager of the event.
   * @param resource The resource that has been created.
   */
  public void resourceCreated(Resource resource);


  public void checkpoint();

  /**
   * Get access (if granted) to that space configuration.
   * As for anything else, the configuration is made accessible through
   * a resource which describes (and export) the possible configuration 
   * options to the outside world.
   * @return A Resource instance, describing and providing access to the
   * various configuration options.
   * @exception SecurityException If access to the configuration of that
   * space is denied to the caller.
   */

  public Resource getConfigResource()
    throws SecurityException;

  public PropertyHolder getPropertyHolder(Runner run, Resource r);

  /**
   * Add a structure listener on that space.
   * Structure listeners are meant to track structural changes to a resource
   * space. This includes creation and deletion of resources, as well
   * as loading and unloading.
   * @param l The StructureChangeListener to plug in.
   */

  public void addStructureChangedListener(StructureChangedListener l);

  /**
   * Remove a structure listener on that space.
   * @param l The listener to remove.
   */

  public void removeStructureChangedListener(StructureChangedListener l);

  /**
   * Initialize that resource space with given properties.
   * This method gets called by the resource space manager, right after the 
   * instance of that space is created.
   * <p>ResourceSpace implementors are expected to understand at least
   * the following set of properties:
   * <dl>
   * <dt>w3c.tools.resources.space.class<dd>The class that the space manager
   * will instantiate to get an instance.
   * <dt>w3c.tools.resources.space.loader<dd>The class of the class loader
   * to be used when loading classes for that space.
   * <dt>w3c.tools.resources.space.owner<dd>The owner of the space
   * </dl>
   * <p>And any additional properties required to initialize the space, as
   * defined by its class.
   */

  public void init(java.util.Properties properties, ResourceContext context)
    throws ResourceSpaceInitException, ResourceSaverInitException;
}
