package w3c.jigsaw.admin;

import java.io.*;
import java.net.*;
import java.util.*;

import w3c.tools.store.Attribute;
import w3c.www.protocol.http.*;

public class PlainRemoteResource implements RemoteResource {
    /**
     * The client side admin context
     */
    protected AdminContext admin = null;
    /**
     * The remote resource class hierarchy.
     */
    protected String classes[] = null;
    /**
     * The remote resource set of attributes.
     */
    protected Attribute attributes[];
    /**
     * The remote resource attribute values.
     */
    protected Object values[] = null;
    /**
     * Is that resource a container resource ?
     */
    protected boolean iscontainer = false;
    /**
     * Is that resource a filtered resource ?
     */
    protected boolean isfiltered = false;
    /**
     * The name of that resource (ie it's identifier attribute).
     */
    protected String identifier = null;
    /**
     * The name of the parent of that resource, as an URL.
     */
    protected URL parent = null;
    /**
     * The admin URL for the wrapped resource.
     */
    protected URL url = null;

    protected Request createRequest() {
	Request request = admin.http.createRequest();
	request.setURL(url);
	return request;
    }

    protected int lookupAttribute(String attr) 
	throws RemoteAccessException
    {
	Attribute attrs[] = getAttributes();
	for (int i = 0  ; i < attrs.length ; i++) {
	    if ( attrs[i] == null )
		continue;
	    if ( attrs[i].getName().equals(attr) )
		return i;
	}
	return -1;
    }
	
    /**
     * Get the target resource class hierarchy.
     * This method will return the class hierarchy as an array of String. The
     * first string in the array is the name of the resource class itself, the
     * last string will always be <em>java.lang.Object</em>.
     * @return A String array givimg the target resource's class description.
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public String[] getClassHierarchy()
	throws RemoteAccessException
    {
	return classes;
    }

    /**
     * Delete that resource, and detach it from its container.
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public void delete()
	throws RemoteAccessException
    {
	throw new RuntimeException("not implemented");
    }

    /**
     * Get the target resource list of attributes.
     * This method returns the target resource attributes description. The
     * resulting array contains instances of the Attribute class, one item
     * per described attributes.
     * <p>Even though this returns all the attribute resources, only the
     * ones that are advertized as being editable can be set through this
     * interface.
     * @return An array of Attribute.
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public synchronized Attribute[] getAttributes()
	throws RemoteAccessException
    {
	if ( attributes == null ) {
	    // Load this class attributes:
	    try {
		attributes = admin.getAttributes(classes[0]);
	    } catch (InvalidResourceClass ex) {
		// Probably (?) an implementation bug...
		throw new RuntimeException("invalid class "+classes[0]);
	    }
	}
	return attributes;
    }

    /**
     * @param name The attribute whose value is to be fetched, encoded as
     * its name.
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public Object getValue(String attr)
	throws RemoteAccessException
    {
	String attrs[] = new String[1];
	attrs[0] = attr;
	return getValues(attrs)[0];
    }

    /**
     * @param attrs The (ordered) set of attributes whose value is to be
     * fetched.
     * @return An (ordered) set of values, one per queried attribute.
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public Object[] getValues(String attrs[])
	throws RemoteAccessException
    {
	// Before going out to the server, check the cache:
	// FIXME

	// Load the attribute values:
	ByteArrayOutputStream bout = new ByteArrayOutputStream();
	DataOutputStream      out  = new DataOutputStream(bout);
	try {
	    // Write out the query body:
	    for (int i = 0 ; i < attrs.length ; i++) {
		int idx = lookupAttribute(attrs[i]);
		if ( idx < 0 )
		    throw new RuntimeException("invalid attribute "+attrs[i]);
		out.writeInt(idx);
	    }
	    out.writeInt(-1);
	    out.close();
	    byte bits[] = bout.toByteArray();
	    // Write the tunnelling HTTP request:
	    Request req = createRequest();
	    req.setMethod("GET-VALUES");
	    req.setContentType(admin.conftype);
	    req.setContentLength(bits.length);
	    req.setOutputStream(new ByteArrayInputStream(bits));
	    // Run that request:
	    Reply rep = admin.runRequest(req);
	    // Parse the admin reply:
	    DataInputStream in    = new DataInputStream(rep.getInputStream());
	    int             idx   = -1;
	    Object          ret[] = new Object[attrs.length];
	    int             reti  = 0;
	    while ((idx = in.readInt()) != -1) {
		if ( idx >= 0 ) {
		    Attribute a = attributes[idx];
		    ret[reti] = a.unpickle(in);
		}
		reti++;
	    }
	    in.close();
	    return ret;
	} catch (Exception ex) {
	    ex.printStackTrace();
	    throw new RemoteAccessException("exception "+ex.getMessage());
	}
	
    }

    /**
     * @param attr The attribute to set, encoded as it's name.
     * @param value The new value for that attribute.
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public void setValue(String attr, Object value)
	throws RemoteAccessException
    {
	String attrs[] = new String[1];
	attrs[0] = attr;
	setValues(attrs, values);
    }

    /**
     * Set a set of attribute values in one shot.
     * This method guarantees that either all setting is done, or none of
     * them are.
     * @param attrs The (ordered) list of attribute to set, encoded as their
     * names.
     * @param values The (ordered) list of values, for each of the above
     * attributes.
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public void setValues(String names[], Object values[])
	throws RemoteAccessException
    {
	// Prepare the output:
	Attribute             attrs[] = getAttributes();
	ByteArrayOutputStream bout    = new ByteArrayOutputStream();
	DataOutputStream      out     = new DataOutputStream(bout);
	try {
	    // Write out the query body:
	    for (int i = 0 ; i < names.length ; i++) {
		int       idx = lookupAttribute(names[i]);
		Attribute a   = attrs[idx];
		out.writeInt(idx);
		a.pickle(out, values[i]);
	    }
	    out.writeInt(-1);
  	    out.close();
	    byte bits[] = bout.toByteArray();
	    // Write the tunnelling HTTP request:
	    Request req = createRequest();
	    req.setMethod("SET-VALUES");
	    req.setContentType(admin.conftype);
	    req.setContentLength(bits.length);
	    req.setOutputStream(new ByteArrayInputStream(bits));
	    // Run that request:
	    Reply rep = admin.runRequest(req);
	} catch (Exception ex) {
	    ex.printStackTrace();
	    throw new RemoteAccessException("exception "+ex.getMessage());
	}
	return;
    }

    /**
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public boolean isContainer()
	throws RemoteAccessException
    {
	return iscontainer;
    }

    /**
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public String[] enumerateResourceIdentifiers()
	throws RemoteAccessException
    {
	if ( ! iscontainer )
	    throw new RuntimeException("not a container");
	try {
	    // Send the request:
	    Request req = createRequest();
	    req.setMethod("ENUMERATE-IDENTIFIERS");
	    req.setContentType(admin.conftype);
	    // Get and check the reply:
	    Reply rep = admin.runRequest(req);
	    // Decode the result:
	    DataInputStream in   = new DataInputStream(rep.getInputStream());
	    Vector          vids = new Vector();
	    String          id   = null;
	    while (! (id = in.readUTF()).equals(""))
		vids.addElement(id);
	    in.close();
	    // Wrapup the result
	    String ids[] = new String[vids.size()];
	    vids.copyInto(ids);
	    return ids;
	} catch (Exception ex) {
	    ex.printStackTrace();
	    throw new RemoteAccessException("http "+ex.getMessage());
	}
    }

    /**
     * @exception RemoteAccessException If somenetwork failure occured.
     */

    public RemoteResource loadResource(String identifier)
	throws RemoteAccessException
    {
	try {
	    // Prepare the request:
	    Request req = createRequest();
	    req.setMethod("LOAD-RESOURCE");
	    req.setContentType(admin.conftype);
	    req.setURL(new URL(url, identifier));
	    // Run it:
	    Reply rep = admin.runRequest(req);
	    // Decode the reply:
	    DataInputStream in  = new DataInputStream(rep.getInputStream());
	    RemoteResource  ret = admin.reader.readResource(url,identifier,in);
	    in.close();
	    return ret;
	} catch (Exception ex) {
	    ex.printStackTrace();
	    throw new RemoteAccessException(ex.getMessage());
	}
    }

    /**
     * Register a new resource within this container.
     * @param id The identifier of the resource to be created.
     * @param classname The name of the class of the resource to be added.
     */

    public void registerResource(String id, String classname) 
	throws RemoteAccessException
    {
	throw new RuntimeException("not implemented");
    }


    /**
     * Is this resource a filtered resource ?
     * @return A boolean, <strong>true</strong> if the resource is filtered
     * and it currently has some filters attached, <strong>false</strong>
     * otherwise.
     */

    public boolean isFiltered() 
	throws RemoteAccessException
    {
	throw new RuntimeException("not implemented");
    }

    /**
     * Get the filters attached to that resource.
     * Each filter is itself a resource, so it is returned as an instance of
     * a remote resource.
     * @return A (posssibly <strong>null</strong>) array of filters attached
     * to that resource.
     */

    public RemoteResource[] getFilters()
	throws RemoteAccessException
    {
	throw new RuntimeException("not implemented");
    }

    /**
     * Unregister a given filter from that resource.
     * @param filter The filter to unregister.
     */

    public void unregisterFilter(RemoteResource filter)
	throws RemoteAccessException
    {
	throw new RuntimeException("not implemented");
    }

    /**
     * Attach a new filter to that resource.
     * @param identifier The name for this filter (if any).
     * @param clsname The name of the filter's class.
     * @return A remote handle to the (remotely) created filter instance.
     */

    public RemoteResource registerFilter(String identifier, String classname)
	throws RemoteAccessException
    {
	throw new RuntimeException("not implemented");
    }
   
    /**
     * Dump that resource to the given output stream.
     * @param prt A print stream to dump to.
     */

    public void dump(PrintStream prt)
	throws RemoteAccessException
    {
	// Dump the class hierarchy:
	String classes[] = getClassHierarchy();
	System.out.println(classes[0]);
	// Dump attribute values:
	Attribute attrs[]   = getAttributes();
	Vector    query     = new Vector();
	for (int i = 0 ; i < attrs.length ; i++) {
	    if ( attrs[i] != null )
		query.addElement(attrs[i].getName());
	}
	String attnames[] = new String[query.size()];
	query.copyInto(attnames);
	// Run the query, and display results:
	Object values[] = getValues(attnames);
	for (int i = 0 ; i < attnames.length ; i++) {
	    Attribute a = attrs[lookupAttribute(attnames[i])];
	    if ( values[i] != null )
		prt.println("\t"+a.getName()+"="+a.stringify(values[i]));
	}
    }
    
    PlainRemoteResource(AdminContext admin
			, URL parent
			, String identifier
			, String classes[]) {
	this.admin      = admin;
	this.identifier = identifier;
	this.parent     = parent;
	this.classes    = classes;
	this.attributes = null;
	this.values     = null;
	// That's how we do it (!) - could make better
	// [container implies filtered => break;]
	for (int i = 0 ; i < classes.length ; i++) {
	    if (classes[i].equals("w3c.jigsaw.resources.ContainerResource")) 
		iscontainer = true;
	    if (classes[i].equals("w3c.jigsaw.resources.FilteredResource"))
		isfiltered = true;
	}
	// Build up that resource URL:
	try {
	    this.url = ((identifier != null) 
			? new URL(parent
				  , (iscontainer ? identifier+"/":identifier))
			: parent);
	} catch (MalformedURLException ex) {
	    // Again, an implementation bug:
	    throw new RuntimeException("invalid server URL");
	}
    }
}

