// MPReader.java
// $Id: MPReader.java,v 1.1 1996/04/10 13:52:53 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.mux ;

import java.io.* ;

class MPReader implements MUX {
    private final boolean debug = false ;

    public static final int DEFSIZE = 16 ;
    public static final int DEFINC  = 8 ;
    public static final int BUFSIZE = 512 ;

    MPStream    stream    = null ;
    InputStream input     = null ;
    Thread      blocked[] = null ;
    int         blocked_count = 0 ;
    Thread      reader    = null ;

    Session     fake         = null ;
    boolean     fake_blocked = false ;

    byte        buffer[]  = null ;
    int         bufptr    = 0 ;
    int         buflen    = 0 ;

    byte    flags    = (byte) 0 ;
    int     sessid   = -1 ;
    Session session  = null ;
    int     protid   = -1 ;
    int     length   = -1 ;
    int     received = 0 ;
    int     nextpad  = 0 ;

    private final void trace (String msg) {
	if ( debug ) 
	    System.out.println (Thread.currentThread()+":"+msg) ;
    }

    private static final int min (int i, int j) {
	return (i < j) ? i : j ;
    }

    private static final int min (int i, int j, int k) {
	return ((i < j) ? ((i < k) ? i : k) : ((j < k) ? j : k));
    }

    private final void clearMessage () {
	flags    = (byte) 0 ;
	sessid   = -1 ;
	session  = null ;
	length   = -1 ;
	protid   = -1 ;
	received = 0 ;
    }

    private final void dumpMessage (String h) {
	if ( debug )
	    System.out.println(h+"("+Thread.currentThread()+")"
			       +":f="+flags
			       +" ,s="+sessid
			       +" ,p="+protid
			       +" ,l="+length) ;
    }

    /**
     * Fill in the internal buffer, as much as possible.
     */

    private void fillBuffer() 
	throws IOException
    {
	// Rotate the input buffer:
	if ( buflen == 0 ) {
	    bufptr = 0 ;
	} else if ( bufptr > (buffer.length >> 1) ) {
	    System.arraycopy (buffer, bufptr, buffer, 0, buffer.length-bufptr);
	    bufptr = 0 ;
	}
	// Get more input:
	int miss = buffer.length - (bufptr+buflen) ;
	int got  = input.read(buffer, bufptr+buflen, miss) ;
	if ( got < 0 )
	    throw new IOException ("read error.") ;
	buflen += got ;
    }

    private void getNextMessage () 
	throws IOException
    {
trace("getNextMessage: buflen="+buflen) ;
	// Can I rotate the buffer ?
	if ( buflen == 0 ) {
	    bufptr = 0 ;
	} else if (bufptr > (buffer.length >> 1) ) {
	    System.arraycopy (buffer, bufptr, buffer, 0, buffer.length-bufptr);
	    bufptr = 0 ;
	}
	int     rdptr = bufptr + buflen ;
	int     got   = 0 ;
	boolean isbig = false ;
	// Read input, parse header:
	while ( true ) {
	    // Any padding to skip ?
	    if ( nextpad != 0 ) {
		if ( buflen >= nextpad ) {
trace("getNextMessage: skip "+nextpad+" bytes.");
		    bufptr += nextpad ;
		    buflen -= nextpad ;
		    nextpad = 0 ;
		    rdptr   = bufptr + buflen ;
		}
	    } 
	    if ( buflen >= 4 ) {
		// Enough bytes for a small header ?
		flags   = (byte) (buffer[bufptr] & 0xfc) ;
		sessid  = (((int) buffer[bufptr] & 0x03)
			   | (((int) buffer[bufptr+1]) & 0xfc)) ;
		length  = (((int) buffer[bufptr+1] & 0x03)
			   | (((int) buffer[bufptr+2] & 0xff) << 2)
			   | (((int) buffer[bufptr+3] & 0xff) << 10)) ;
dumpMessage("GetNextMessage") ;
		if ( isbig = ((flags & MUX_LONG_LENGTH) == MUX_LONG_LENGTH) ) {
System.out.println("*** message *is* big !") ;
		    protid  = length ;
		    session = stream.getSession (flags, sessid, protid) ;
		} else {
		    nextpad = ((length & 0x3)!=0) ? (4 - (length & 0x3)) : 0 ;
		    buflen -= 4 ;
		    bufptr += 4 ;
		    session = stream.getSession (flags, sessid) ;
		    return ;
		}
	    }
	    if ( isbig ) {
		if ( buflen >= 8 ) {
		    length = (((int) buffer[bufptr+4] & 0xff)
			      | (((int) buffer[bufptr+5] & 0xff) << 8)
			      | (((int) buffer[bufptr+6] & 0xff) << 16)
			      | (((int) buffer[bufptr+7] & 0xff) << 24)) ;
		    buflen -= 8 ;
		    bufptr += 8 ;
		    nextpad = ((length & 0x3)!=0) ? (4 - (length & 0x3)) : 0 ;
dumpMessage ("getNextMessage(big)") ;
		    return ;
		}
	    }
	    if ((got = input.read(buffer, rdptr, buffer.length-rdptr)) < 0 ) {
		// Handle the error !
		throw new IOException ("IO read error") ;
	    } 
	    rdptr  += got ;
	    buflen += got ;
trace ("getNextMessage: read "+got+" bytes, buflen="+buflen) ;
	}
    }

    /**
     * Wake up the given session, if it is pending.
     * @return A boolean <strong>true</strong> if the session was waken up, 
     *    <strong>false</strong> otherwise.
     */

    protected boolean awakeSession (Session s)
	throws IOException
    {
	// Is it the fake session ?
	if ( s == fake ) {
	    if ( fake_blocked ) {
		fake_blocked = false ;
		s.notifyAll() ;
		return true ;
	    } else {
		return false ;
	    }
	}
	// Normal sesion, whose id is known:
	int id = s.getIdentifier() ;
	if ((id < blocked.length) && (blocked[id] != null)) {
	    blocked[id] = null ;
	    SessionInputStream in = s.getInputStream() ;
	    synchronized (in) {
		in.notifyAll() ;
	    }
	    return true ;
	}
	return false ;
    }

    /**
     * The given session has gopt its message, awake any pending session.
     */

    protected void awakeAnySession () 
	throws IOException
    {
	// Try for a normal session first:
	for (int i = 0 ; i < blocked.length ; i++) {
	    if ( blocked[i] != null ) {
trace ("awakeSession: awakes "+i) ;
		awakeSession (stream.getSession(i)) ;
		return ;
	    }
	}
	// Else wake up the accepting session:
	if ( fake_blocked ) {
trace("awakeAnySession: awakes accept.") ;
	    synchronized (fake) {
		fake.notifyAll() ;
	    }
	}
	return ;
    }

    /**
     * Mark the given session as blocked.
     */
    
    protected synchronized void blockSession (Session s) {
	// Is it the fake session ?
	if ( s == fake ) {
trace ("blockSession: accepting.") ;
	    fake_blocked = true ;
	    return ;
	}
	// Ok, we should know about this session identifier:
	int id = s.getIdentifier() ;
	if ( id >= blocked.length ) {
	    // Resize the blocked array:
	    if ( id > MAX_SESSION )
		throw new RuntimeException (this.getClass().getName()
					    + ": invalid session id.") ;
	    Thread nb[] = new Thread[id+DEFINC] ;
	    System.arraycopy (blocked, 0, nb, 0, blocked.length) ;
	    blocked = nb ;
	}
trace ("blockSession: " + id) ;
	blocked[id] = Thread.currentThread() ;
	blocked_count++ ;
    }

    /**
     * Unblock the session.
     */

    protected synchronized void unblockSession (Session s) {
	// Is it the fake session ?
	if ( s == fake ) {
trace ("unblockSession: accepting session.") ;
	    fake_blocked = false ;
	    return ;
	}
	// We should know about this session identifier:
trace ("unblockSession "+s.getIdentifier()) ;
	blocked[s.getIdentifier()] = null ;
	blocked_count-- ;
    }

    /**
     * See if we can be the reader.
     * If no reader is set for this stream, become the reader: set the reader
     * variable, and return -3. If some reader is here, than mark the thread
     * as blocked, and return -2.
     * @return The number of bytes read for the given session.
     */

    protected synchronized int becomeReader(Session s, byte b[], int o, int l)
	throws IOException
    {
trace ("becomeReader (reader="+reader+")");
	if ( reader == null ) {
	    reader = Thread.currentThread() ;
	    return -4 ;
	} else {
	    blockSession (s) ;
	    return -2 ;
	}
    }

    
    /**
     * Try to become the reader for this stream.
     * This methods will elect only one of the Session readers, to read the 
     * input stream (making best effort to elect the right one).
     * @param s The session for wich we should read.
     * @param b Were we should put the data.
     * @param o The offset in buffer.
     * @param l The length of the data to read.
     * @return Number of bytes read, or -1 if end of input, or -2 if the
     *    thread wasn't elected for reading, or -3 if some conditions might 
     *    have changed for the session input stream.
     *    In case of -2 return, the session input stream will be notified
     *    when some data is available, in case of -3, the caller should
     *    tests any of its condition before invoking again tryRead.
     * @exception IOException If something went wrong.
     */
     
    protected int tryRead (Session s, byte b[], int o, int l)
	throws IOException
    {
	int cnt = becomeReader (s, b, o, l) ;
	if ( cnt >= -3 ) {
	    return cnt ;
	} else {
	    return doRead (s, b, o, l) ;
	}
    }

    /**
     * Do the actual reading of the input stream.
     */

    protected int doRead(Session s, byte b[], int o, int l)
	throws IOException
    {
	SessionInputStream in = s.getInputStream() ;
trace ("doRead.") ;

        unblockSession(s) ;
	while ( ! in.closed ) {
	    if ( sessid == -1 ) {
		// Read in a new message
		getNextMessage() ;
		if ((flags & MUX_SYN) == MUX_SYN) 
		    stream.syn(flags, sessid, protid) ;
		if ((flags & MUX_FIN) == MUX_FIN)
		    stream.fin(flags, sessid, protid) ;
		if ((flags & MUX_RST) == MUX_RST)
		    stream.rst(flags, sessid, protid) ;
		if ( length == 0 ) {
		    trace ("doread: ctrl msg done.") ;
		    // No additional data with the message
		    clearMessage() ;
		    reader = null ;
		    awakeAnySession() ;
		    return -3 ;
		} else if ( session != s ) {
		    // Some data, re-elect a new reader for the appropriate
		    // session if any, or continue.
		    if ( awakeSession (session) ) {
			reader = null ;
			blockSession (s) ;
trace ("doRead: " +sessid+" awaken.") ;
			return -2 ;
		    }
		}
		
	    }
	    if ( session == s ) {
		// I am the reader for this session
		int sent = 0 ;
		int miss = length - received ;
trace("doRead: missing "+miss+" bytes."+length+"/"+received);
		if ( buflen > 0 ) {
trace("doRead: buflen="+buflen) ;
		    sent = min (l, buflen, miss) ;
		    System.arraycopy (buffer, bufptr, b, o, sent) ;
		    bufptr   += sent ;
		    buflen   -= sent ;  
		    received += sent ;
		    o += sent ;
		    l -= sent ;
		} else {
		    sent = min (miss, l) ;
		    sent = input.read(b, o, sent) ;
		    received += sent ;
		    o += sent ;
		    l -= sent ;
trace("doRead: sent="+sent) ;
		}
		if ( received >= length ) {
		    clearMessage() ;
		    reader   = null ;
		    awakeAnySession() ;
trace ("doRead: msg done (sent="+sent+") .");
		    return sent ;
		} else if (l <= 0) {
trace ("doRead: input buffer full.") ;		    
                    reader = null ;
                    awakeAnySession() ;
                    return sent ;
		}
	    }  else if ( session != null ) {
		if ( buflen == 0 ) 
		    fillBuffer() ;
		if ( queueMessage() ) {
trace("incomplete msg queued.") ;
		    // An incomplete message has been queued.
		    if ( awakeSession (session) ) {
			// But its session thread is ready (good)
			blockSession (s) ;
			reader = null ;
			return -2 ;
		    } 
		} else {
		    // A complete message has been read and queued, continue
trace ("full msg queued.") ;
		    clearMessage() ;
		    continue ;
		}
	    }
	}
	// We can exit here if the reader's session has been closed.
	reader = null ;
	return -1 ;
    }

    /**
     * Queue current message in its target session.
     */

    protected boolean queueMessage ()
	throws IOException
    {
	if ( buflen == 0 )
	    return true ;
	int miss = length - received ;
	int sent = min (buflen, miss) ;
trace ("queueing " + sent + " bytes of " + length +"/" +received) ;
	session.getInputStream().addInput (buffer, bufptr, sent) ;
	unblockSession (session) ;
	buflen   -= sent ;
	bufptr   += sent ;
	received += sent ;
	return (received < length) ;
    }

    MPReader (MPStream stream, InputStream input) 
	throws IOException
    {
	this.stream  = stream ;
	this.fake    = stream.getFakeSession() ;
	this.input   = input ;
	this.blocked = new Thread[DEFSIZE] ;
	this.buffer  = new byte[BUFSIZE] ;
    }

}
