// MIMEParser.java
// $Id: MIMEParser.java,v 1.1 1996/04/10 13:52:18 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.mime ;

import java.util.Vector ;
import java.util.Hashtable ;
import java.util.Date ;

import java.io.* ;

/**
 * The MIMEParser class parses an input MIME stream. 
 * <p>This parsing is done in two stages. In the first stage, only header
 * names and header values (as String) are parsed. These informations
 * are built into a MIMEHeader object which than parses header value
 * according to some specific type.</p>
 * <p>You can use this class either directly, or subclass MIMEParser
 * and MIMEHeader for example, to get HTTPParser and Request in the
 * HTTP context, or MAILParser and MAILHeaders in SMTP context.
 * @see w3c.mime.MIMEHeaders 
 */

public class MIMEParser {
    protected int         ch       = -1 ;
    protected InputStream input    = null ;
    protected byte        buffer[] = new byte[128] ;
    protected int         bsize    = 0 ;

    protected void expect (int car) 
	throws MIMEParserException, IOException
    {
	if ( car != ch ) {
	    String sc = (new Character((char) car)).toString() ;
	    String se = (new Character((char) ch)).toString() ;
	    throw new MIMEParserException ("expecting "
					   + sc + "("+car+")"
					   + " got "
					   + se + "("+ch+")\n"
					   + "context: " 
					   + new String (buffer, 0, 0, bsize)
					   + "\n") ;
	}
	ch = input.read() ;
    }

    protected void skipSpaces () 
	throws MIMEParserException, IOException
    {
	while ( (ch == ' ') || (ch == '\t') )
	    ch = input.read() ;
    }

    protected final void append (int c) {
	if ( bsize+1 >= buffer.length ) {
	    byte nb[] = new byte[buffer.length*2] ;
	    System.arraycopy (buffer, 0, nb, 0, buffer.length) ;
	    buffer = nb ;
	}
	buffer[bsize++] = (byte) c ;
    }

    /*
     * Get the header name:
     */

    protected String parse822HeaderName () 
	throws MIMEParserException, IOException
    {
	bsize = 0 ;
	while ( (ch >= 32) && (ch != ':') ) {
	    append (Character.toLowerCase((char) ch)) ;
	    ch = input.read() ;
	}
	expect (':') ;
	if ( bsize <= 0 )
	    throw new MIMEParserException ("expected a header name.") ;
	return new String (buffer, 0, 0, bsize) ;
    }
    
    /*
     * Get the header body, still trying to be 822 compliant *and* HTTP
     * robust, which is unfortunatelly a contrdiction.
     */

    protected String parse822HeaderBody () 
	throws MIMEParserException, IOException
    {
	bsize = 0 ;
	skipSpaces () ;
      loop:
	while ( true ) {
	    switch (ch) {
	      case -1:
		  break loop ;
	      case '\r':
		  if ( (ch = input.read()) != '\n' ) {
		      append ('\r') ; 
		      continue ;
		  }
		  // no break intentional
	      case '\n':
		  // do as if '\r' had been received. This defeats 822, but
		  // makes HTTP more "robust". I wish HTTP were a binary 
		  // protocol.
		  switch (ch = input.read()) {
		    case ' ': case '\t':
			do {
			    ch = input.read () ;
			} while ((ch == ' ') || (ch == '\t')) ;
			break ;
		    default:
			break loop ;
		  }
		  break ;
	      default:
		  append ((char) ch) ;
		  break ;
	    }
	    ch = input.read() ;
	}
	return new String (buffer, 0, 0, bsize) ;
    }
    
    /*
     * Parse the given input stream for an HTTP 1.1 token. 
     */

    protected String parseToken (boolean lower) 
	throws MIMEParserException, IOException
    {
	bsize = 0 ;
	while ( true ) {
	    switch ( ch ) {
	      // CTLs
	      case -1:
	      case 0: case 1: case 2: case 3: case 4: case 5: case 6: case 7:
	      case 8: case 9: case 10: case 11: case 12: case 13: case 14: 
	      case 15: case 16: case 17: case 18: case 19: case 20: case 21:
	      case 22: case 23: case 24: case 25: case 26: case 27: case 28:
	      case 29: case 30: case 31: 
	      // tspecials
	      case '(': case ')': case '<': case '>': case '@':
	      case ',': case ';': case ':': case '\\': case '\"':
	      case '/': case '[': case ']': case '?': case '=':
	      case '{': case '}': case ' ': 
		  return new String (buffer, 0, 0, bsize) ;
	      default:
		  append ((char) (lower 
				  ? Character.toLowerCase((char) ch)
				  : ch)) ;
	    }
	    ch = input.read() ;
	}
    }

    /*
     * Parse the current input stream for rfc822 headers. For each
     * found header, the <em>into</em> hashtable is looked up. If an
     * object is found, it should be a StringBuffer, to which we
     * concatenate the newly parsed value preceded by a
     * comma. Otherwise a new entry is created, associating the header
     * name to the StringBuffer holding its value.
     * @parameter	Hashtable to fill.
     * @exception	MIMEParserException, if HTTP stream cannot be
     * parsed.
     */
     
    protected void parse822Headers (Hashtable into) 
	throws MIMEParserException, IOException
    {
	while ( true ) {
	    if ( ch == '\r' ) {
		if ( (ch = input.read()) == '\n' )
		    return ;
	    } else if ( ch == '\n' ) {
		return ;
	    }
	    String name = parse822HeaderName () ; skipSpaces() ;
	    String body = parse822HeaderBody () ;
	    String old  = (String) into.get (name) ;
	    if ( old != null ) 
		body = old + "," + body ;
	    into.put (name, body) ;
	}
    }

    /**
     * Get the set of headers defined in this MIME stream
     * This method return a container for the headers encountered
     * while parsing the stream.
     * @exception MIMEException If the input stream couldn't be parsed.
     */
    
    public MIMEHeaders getMIMEHeaders ()
	throws MIMEException
    {
	Hashtable headers  = new Hashtable (13) ;
	Hashtable rheaders = new Hashtable (13) ;
	try {
	    ch = input.read() ;
	    parse822Headers (rheaders) ;
	} catch (IOException e) {
	    throw new MIMEException ("IO Error while reading input.") ;
	} catch (MIMEParserException e) {
	    throw new MIMEException ("Invalid MIME message.") ;
	}
	return new MIMEHeaders (headers, rheaders) ;
    }
    
    /**
     * Get the message body, as an input stream.
     * FIXME: I don't know if Request should implements IO interface (and hence
     * check that the body and only the body gets read), or ...
     */

    public BufferedInputStream getInputStream () {
	return (BufferedInputStream) input ;
    }

    /**
     * Create an instance of the MIMEParser class.
     * @param in The input stream to be parsed as a MIME stream.
     */

    public MIMEParser (InputStream input) {
	this.input = input ;
    }

    /**
     * Debug only. Parse the given file name as a MIME stream.
     */

    public static void main (String args[]) {
	if ( args.length != 1 ) {
	    System.out.println ("Usage: httpd.MIMEParser <file>") ;
	    System.exit (1) ;
	}
	try {
	    InputStream input = (new BufferedInputStream 
				 (new FileInputStream 
				  (new File (args[0])))) ;
	    MIMEParser  parser  = new MIMEParser (input) ;
	    MIMEHeaders headers = parser.getMIMEHeaders() ;
	    headers.dump() ;
	} catch (Exception e) {
	    e.printStackTrace() ;
	    System.out.println ("Caught exception.") ;
	}
	System.exit (0) ;
    }
}

