// ResourceStoreManager.java
// $Id: ResourceStoreManager.java,v 1.6 1996/05/31 14:38:36 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.resources ;

import java.io.* ;
import java.util.* ;

import w3c.jigsaw.http.*;

/**
 * The list of loaded store, kept in an LRU list.
 */

class StoreEntry {
    // LRU management infos:
    StoreEntry next = null ;
    StoreEntry prev = null ;
    // ResourceStore infos:
    ResourceStore       store      = null ;
    ResourceStoreHolder holders[]  = null ;
    File                repository = null ;

    /**
     * LRU management - remove this entry from the LRU list.
     */
    
    synchronized void LRUremove(StoreEntry head) {
	synchronized(head) {
	    prev.next = next;
	    next.prev = prev ;
	    next = null ;
	    prev = null ;
	}
    }

    /**
     * LRU management - bring to front.
     */
    
    synchronized void LRUfront(StoreEntry head) {
	synchronized(head) {
	    prev.next = next ;
	    next.prev = prev ;
	    head.next.prev = this ;
	    next           = head.next ;
	    prev           = head ;
	    head.next      = this ;
	}
    }

    /**
     * LRU management - Insert on front.
     */

    synchronized void LRUinsert(StoreEntry head) {
	synchronized (head) {
	    head.next.prev = this ;
	    next           = head.next ;
	    prev           = head ;
	    head.next      = this ;
	}
    }
     
    synchronized void LRUdisplay(StoreEntry head) {
	System.out.println("LRU list from "+head) ;
	synchronized (head) {
	    for (StoreEntry ptr = head ; ptr != null ; ptr = ptr.next) {
		System.out.println(ptr.repository
				   + " " 
				   + ((ptr.next == null) 
				      ? "*"
				      : ((ptr.next.prev == ptr) ? "+" : "-"))
				   + " "
				   + ((ptr.prev == null)
				      ? "*"
				      : ((ptr.prev.next == ptr) ? "+" : "-")));
	    }
	}
    }

    /**
     * Add a holder for this store.
     */

    synchronized void addHolder(ResourceStoreHolder holder) {
	// Check for a free entry 
	for (int i = 0 ; i < holders.length ; i++) {
	    if ( holders[i] == null ) {
		holders[i] = holder ;
		return ;
	    } else if ( holders[i] == holder ) {
		return ;
	    }
	}
	// Resize the holders array:
	ResourceStoreHolder nholders[] = null ;
	nholders = new ResourceStoreHolder[holders.length+1] ;
	nholders[holders.length] = holder ;
	holders = nholders ;
    }
	
    /**
     * Remove a holder for this store.
     * Don't panic: if no more holders hold the given store, it will quickly
     * be brought to the end of the LRU queue, and disappear by itself.
     */

    synchronized void removeHolder(ResourceStoreHolder holder) {
	for (int i = 0 ; i < holders.length ; i++) {
	    if ( holders[i] == holder ) {
		holders[i] = null ;
		return ;
	    } 
	}
    }

    /**
     * Load the store of this entry.
     */

    synchronized ResourceStore getStore(ResourceStoreManager manager) {
	if ( store == null ) {
	    store = new SimpleResourceStore() ;
	    store.initialize(manager, repository) ;
	}
	return store ;
    }

    /**
     * Try unloading the space for this entry.
     * Try make all holders unload the store, if some of them don't want to, 
     * too bad.
     */
    
    synchronized void unloadStore() {
	if ( store != null ) {
	    boolean unhandled = true ;
	    for (int i = 0 ; i < holders.length ; i++) {
		ResourceStoreHolder holder = holders[i] ;
		if ( holder == null )
		    continue ;
		if ( holder.notifyStoreUnload(store) ) {
		    holders[i] = null ;
		} else {
		    unhandled = false ;
		}
	    }
	    if ( unhandled ) {
		store.shutdown() ;
		store = null ;
	    }
	}
    }

    /**
     * Shutdown the store.
     */

    synchronized void shutdownStore() {
	if ( store != null ) {
	    for (int i = 0 ; i < holders.length ; i++) {
		ResourceStoreHolder holder = holders[i] ;
		if ( holder == null )
		    continue ;
		else
		    holder.notifyStoreShutdown(store) ;
	    }
	    store.shutdown() ;
	    store = null ;
	}
    }

    /**
     * Try stabilizing the store.
     * Continue until at least one holder succeeds.
     */

    synchronized void notifyStoreStabilize() {
	if ( store != null ) {
	    for (int i = 0 ; i < holders.length ; i++) {
		ResourceStoreHolder holder = holders[i] ;
		if ( holder == null ) {
		    continue ;
		} else if ( holder.notifyStoreStabilize(store) ) {
		    break ;
		}
	    }
	}
    }

    StoreEntry(File repository, ResourceStoreHolder holder) {
	this.store      = null ;
	this.repository = repository ;
	this.holders    = new ResourceStoreHolder[3] ;
	holders[0]      = holder ;
    }
    
}

class StoreManagerSweeper extends Thread {
    ResourceStoreManager manager = null ;
    boolean              killed  = false ;

    protected synchronized void waitEvent() {
	boolean done = false ;

	// Wait for an event to come by:
	while ( ! done ) {
	    try {
		wait() ;
		done = true ;
	    } catch (InterruptedException ex) {
	    }
	}
    }

    protected synchronized void sweep() {
	notify() ;
    }

    protected synchronized void shutdown() {
	killed = true ;
	notify() ;
    }

    public void run() {
	while ( true ) {
	    waitEvent() ;
	    // The whole trick is to run the collect method of the store 
	    // manager, without having the lock on the sweeper itself, so
	    // that clients can still trigger it later
	    if ( killed ) {
		break ;
	    } else {
		try {
		    manager.collect() ;
		} catch (Exception ex) {
		    // We really don't want this thread to die
		    ex.printStackTrace() ;
		}
	    }
	}
    }

    StoreManagerSweeper(ResourceStoreManager manager) {
	this.manager = manager ;
	this.setName("StoreSweeper") ;
	this.start() ;
    }

}

public class ResourceStoreManager {
    /**
     * The loaded resource stores.
     */
    protected Hashtable entries = new Hashtable() ;
    /**
     * Is this store shutdown ?
     */
    protected boolean closed = false ;
    /**
     * The server we are attached to.
     */
    protected httpd server = null ;
    /**
     * The fake head of our LRU list for store entries.
     */
    protected StoreEntry head = null ;
    /**
     * The fake tail of our LRU list fro store entries.
     */
    protected StoreEntry tail = null ;
    /**
     * Our sweeper thread:
     */
    protected StoreManagerSweeper sweeper = null ;

    protected final int getMaxEntries() {
	return 512 ;
    }

    /**
     * Check that this resource store manager isn't closed.
     * @exception RuntimeException If the store manager was closed.
     */

    protected final synchronized void checkClosed() {
	if ( closed )
	    throw new RuntimeException("Invalid store manager access.") ;
    }

    /**
     * Lookup an entry in the store.
     * @param repository The repository of the store to lookup.
     * @param create Create a new entry if it doesn't exist.
     */

    protected synchronized StoreEntry lookupEntry(File repository
						  , ResourceStoreHolder holder
						  , boolean create) {
	StoreEntry entry = (StoreEntry) entries.get(repository) ;
	if (create && (entry == null)) {
	    // Check to see if we have exceeded our quota:
	    if ( entries.size() > getMaxEntries() ) 
		sweeper.sweep() ;
	    entry = new StoreEntry(repository, holder) ;
	    entry.LRUinsert(head) ;
	    entries.put(repository, entry) ;
	}
	return entry ;
    }

    /**
     * Pick the least recently used entry, and remove all links to it.
     * After this method as run, the least recently used entry for some store
     * will be returned. The store manager will have discarded all its link to 
     * it, and the entry shutdown will have to be performed by the caller.
     * @return An StoreEntry instance, to be cleaned up.
     */

    protected synchronized StoreEntry pickLRUEntry() {
	StoreEntry lru = null ;
	synchronized (head) {
	    lru = tail.prev ;
	    lru.LRUremove(head) ;
	}
	entries.remove(lru.repository) ;
	return lru ;
    }

    /**
     * Collect enough entries to go back into fixed limits.
     */

    public void collect() {
	while ( entries.size() > getMaxEntries() ) {
	    StoreEntry lru = pickLRUEntry() ;
	    lru.shutdownStore() ;
	}
    }

    /**
     * Load a resource store.
     * @param holder The holder for the resource store.
     * @param repository Its associated repository.
     */

    public ResourceStore loadResourceStore(ResourceStoreHolder holder
					   , File repository) {
	checkClosed() ;
	StoreEntry entry = lookupEntry(repository, holder, true) ;
	return entry.getStore(this) ;
    }

    /**
     * Unhold the given store.
     * For some reason, the holder for this store has decided to close it.
     * Remove from ths list of this store holder. When this call is made, the
     * caller is expected to have cleaned-up the store (stabilize it, etc).
     * @param holder The holder of the store.
     * @param store The store this holder doesn't want to hold anymore.
     */

    public void unholdResourceStore(ResourceStoreHolder holder
				    , File repository) {
	StoreEntry entry = lookupEntry(repository, null, false) ;
	if ( entry != null )
	    entry.removeHolder(holder) ;
    }

				    
    /**
     * Shutdown this resource store manager.
     * Go through all entries, and shut them down.
     */

    public synchronized void shutdown() {
	// Kill the sweeper thread:
	sweeper.shutdown() ;
	// Clenup all pending resource stores:
	Enumeration enum = entries.elements() ;
	while ( enum.hasMoreElements() ) {
	    StoreEntry entry = (StoreEntry) enum.nextElement() ;
	    entry.shutdownStore() ;
	    entries.remove(entry.repository) ;
	}
	closed = true ;
    }

    /**
     * Mark the given store as having been used recently.
     * @param repository The repository of the store that has been used.
     */

    public void markUsed(File repository) {
	StoreEntry entry = lookupEntry(repository, null, false) ;
	if ( entry != null )
	    entry.LRUfront(head) ;
    }

    /**
     * Create a new resource store for the given server.
     * @param server The server that wants a resource store manager.
     */

    public ResourceStoreManager (httpd server) {
	this.server  = server ;
	this.entries = new Hashtable() ;
	this.sweeper = new StoreManagerSweeper(this) ;
	this.head = new StoreEntry(new File("*head*"), null) ;
	this.tail = new StoreEntry(new File("*tail*"), null) ;
	head.next = tail ;
	head.prev = null ;
	tail.next = null ;
	tail.prev = head ;
    }

}
