// ResourceStore.java
// $Id: ResourceStore.java,v 1.3 1996/05/28 14:37:42 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.resources ;

import java.util.*;
import java.io.* ;

/**
 * A resource store implements persistency for a set of resources.
 * A resource store may implement a number of strategies along different
 * axis, for example: 
 * <ul>
 * <li>It may connect to some database to get the resource attributes. 
 * <li>It may restrict the classes of the resource it handles (for security
 *      reasons), by using a specific class loader.
 * <li>It may implement some caching scheme, to keep memory requirements low.
 * <li>It may be distributed (eg using a ResourceStoreStub in the client, but
 * providing access to a server-side resource store.).
 * <ul>
 */

public interface ResourceStore {
    
    /**
     * Restore the resource whose name is given.
     * This method doesn't assume that the resource will actually be restored,
     * it can be kept in a cache by the ResourceStore object, and the cached 
     * version of the resource may be returned.
     * @param identifier The identifier of the resource to restore.
     * @param defs Default attribute values. If the resource needs to be
     *     restored from its pickled version, this Hashtable provides
     *     a set of default values for some of the attributes.
     * @return A Resource instance, or <strong>null</strong>.
     */

    public Resource loadResource(String identifier, Hashtable defs) ;

    /**
     * Get this resource, but only if already loaded.
     * The resource store may (recommended) maintain a cache of the resource
     * it loads from its store. If the resource having this identifier 
     * has already been loaded, return it, otherwise, return
     * <strong>null</strong>.
     * @param identifier The resource identifier.
     * @return A Resource instance, or <strong>null</strong>.
     */

    public Resource lookupResource(String identifier) ;

    /**
     * Stabilize the resource having the given identifier.
     * @param identifier The resource identifier.
     * @exception UnknownResourceException If the resource is unknown.
     */

    public void saveResource(String identifier) 
	throws UnknownResourceException ;

    /**
     * Lock the given resource in the cache.
     * @param identifier The identifier of the resource to be locked in cache.
     * @return A boolean <strong>true</strong> if locking succeeded.
     */

    public boolean lockResource(ResourceLocker locker, String identifier) ;

    /**
     * Unlock the given resource in the cache.
     * @param identifier The identifier of the resource to be unlocked.
     */

    public void unlockResource(ResourceLocker locker, String identifier) ;

    /**
     * Add this resource to this resource store.
     * @param resource The resource to be added.
     */

    public void addResource(Resource resource) ;

    /**
     * Remove this resource from the repository.
     * @param identifier The identifier of the resource to be removed.
     */

    public void removeResource(String identifier) ;
    
    /**
     * Mark this resource as modified.
     * @param identifier The identifier of the modified resource.
     */

    public void markModified(String identifier) ;

    /**
     * Shutdown this store.
     */

    public void shutdown() ;

    /**
     * Save this store.
     */

    public void save() ;

    /**
     * Enumerate all the resources saved in this store.
     * @return An enumeration of Strings, giving the identifier for all 
     *     the resources that this store knows about.
     */

    public Enumeration enumerateResourceIdentifiers() ;

    /**
     * Check for the existence of a resource in this store.
     * @param identifier The identifier of the resource to check.
     * @return A boolean <strong>true</strong> if the resource exists
     *    in this store, <strong>false</strong> otherwise.
     */

    public boolean hasResource(String identifier) ;

    /**
     * This resource store is being built, initialize it with the given arg.
     * @param repository A file, giving the location of the associated 
     *    repository.
     */

    public void initialize(ResourceStoreManager manager, File repository) ;


}
