// ResourceFilter.java
// $Id: ResourceFilter.java,v 1.1 1996/04/10 13:50:18 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.resources ;

import java.util.*;
import java.io.*;

import w3c.jigsaw.http.* ;

public class ResourceFilter extends Resource implements ResourceShadower {
    /**
     * Return value for ingoingFilter. Don't call the outgoingFilter.
     * When this value is returned by the <code>ingoingFilter</code>, the
     * <code>outgoingFilter</code> won't be called.
     */

    public static final int DontCallOutgoing = 1 ;
    
    /**
     * Return value for ingoingFilter. Call outgoin if no errors.
     * When returned by the <code>ingoingFilter</code>, and no error
     * occured while processing the request, the <code>outgoingFilter</code>
     * will be called.
     */

    public static final int CallOutgoing = 2 ;

    /**
     * Return value for ingoingFilter. Call outgoing even if error occured.
     * When returned byt the <code>ingoingFilter</code>, the 
     * <code>outgoingFilter</code> will be called, even if some error
     * occured.
     */

    public static final int ForceOutgoing = 3 ;

    /**
     * Attribute index - Our target attribute.
     */
    protected static int ATTR_TARGET = -1 ;

    static {
	Attribute a   = null ;
	Class     cls = null ;

	try {
	    cls = Class.forName("w3c.jigsaw.resources.ResourceFilter") ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
	// The class for files having this extension.
	a = new ObjectAttribute("target"
				, "w3c.jigsaw.resources.FilteredResource"
				, null
				, Attribute.DONTSAVE) ;
	ATTR_TARGET = AttributeRegistery.registerAttribute(cls, a) ;
    }

    /**
     * Our target attribute description:
     */
    protected Attribute target_attributes[] = null ;
    /**
     * Our target shadowed values.
     */
    protected Object target_values[] = null ;

    /**
     * Get our target resource.
     */

    public Resource getTargetResource() {
	return (Resource) getValue(ATTR_TARGET, null) ;
    }

    /**
     * Get our target attribute list.
     */

    public Attribute[] getTargetAttributes() {
	return target_attributes;
    }

    /**
     * Get a shadowed attribute value.
     * @param idx The index of the attribute.
     * @param def The default value to return if undef.
     * @return The attribute value or the provided default value.
     */

    public Object getTargetValue(int idx, Object def) {
	if ((idx < 0) || (idx >= target_values.length))
	    throw new IllegalAttributeAccess(this, idx) ;
	Object value = target_values[idx] ;
	if ( value == null )
	    return (def == null) ? target_attributes[idx].getDefault() : def ;
	else
	    return value ;
    }

    /**
     * Get a shadowed attribute value (by name).
     * @param name The name of the attribute.
     * @param def The default value to return if undef.
     * @return The attribute value or the provided default value.
     */

    public Object getTargetValue(String name, Object def) {
	for (int i = 0 ; i < target_attributes.length ; i++) {
	    if ( name.equals(target_attributes[i].getName()) ) {
		return target_values[i] ;
	    }
	}
	return null ;
    }

    /**
     * Set a shadow attribute value.
     * @param idx The index of the attribute to set.
     * @param value Its new value.
     */

    public void setTargetValue(int idx, Object def) {
	if ((idx < 0) || (idx >= target_values.length))
	    throw new IllegalAttributeAccess(this, idx) ;
	target_values[idx] = def ;
	markModified() ;
    }

    /**
     * Set a shadow attribute value.
     * @param name The name of the attribute to set.
     * @param value Its new value.
     */

    public void setTargetValue(String name, Object def) {
	for (int i = 0 ; i < target_attributes.length ; i++) {
	    if ( name.equals(target_attributes[i].getName()) ) {
		setTargetValue(i, def) ;
		markModified() ;
		return ;
	    }
	}
	throw new IllegalAttributeAccess(this, name) ;
    }

    /**
     * Does this filters defines the given target attribute.
     * @param idx The index of the target attribute.
     * @return A boolean <strong>true</strong if attribute is defined.
     */

    public boolean definesTargetAttribute(int idx) {
	if (idx < 0)
	    throw new IllegalAttributeAccess(this, idx) ;
	if (idx >= target_attributes.length)
	    return false ;
	return (target_values[idx] != null) ;
    }

    /**
     * Does this filters defines the given target attribute.
     * @param name The name of the target attribute.
     * @return A boolean <strong>true</strong if attribute is defined.
     */

    public boolean definesTargetAttribute(String name) {
	for (int i = 0 ; i < target_attributes.length ; i++) {
	    if ( name.equals(target_attributes[i].getName()) ) 
		return (target_values[i] != null) ;
	}
	return false ;
    }


    /**
     * Mark this filter as modified.
     * Delegate the dirty bit to our target.
     */

    public void markModified() {
	getTargetResource().markModified() ;
    }

    /**
     * Filter the request before its processing.
     * This method gets called before any actual processing occurs on the
     * given request. Each filter that gets its <strong>ingoingFilter</strong>
     * method called, is guaranteed to have its <strong>outgoingFilter</strong>
     * on the way back.
     * <p>Finally, directory entities should trigger a call to their filters
     * even when the operation is simple a name lookup. 
     * @param client The client issuing the request.
     * @param request The request to be filtered.
     * @return An integer, one of <code>DontCallOutgoing</code>, 
     *    <code>CallOutgoing</code> or <code>ForceOutgoing</code>, as defined
     *    in this class.
     * @see FilteredResource
     */

    public int ingoingFilter (Request request)
	throws HTTPException
    {
	return CallOutgoing ;
    }

    /**
     * Filter the request after its processing.
     * This method gets called after the entity has managed the request. This
     * is the right place to implement such things as on the fly content 
     * conversion.
     * @param client The client issuing the request.
     * @param request The requested being processed.
     * @param reply The actual reply, as emitted by the originial entity.
     * @exception HTTPException If the filter wants to send back an HTTP
     *    error instead of a normal reply
     */

    public Reply outgoingFilter (Request request, Reply reply)
	throws HTTPException
    {
	return reply ;
    }

    /**
     * Pickle a filter.
     * Pickle the filter attributes themselves, and than the set of shadow
     * attribute values we maintain for our target.
     * @param out The data output stream to pickle to.
     */

    public void pickle(DataOutputStream out)
	throws IOException
    {
	// Pickle myself:
	super.pickle(out) ;
	// Pickle my set of shadwo attribute values:
	for (int i = 0 ; i < target_attributes.length ; i++) {
	    Attribute attr = target_attributes[i] ;
	    if ((target_values[i] != null)
		&& ! attr.checkFlag(Attribute.DONTSAVE)) {
		out.writeBoolean(true) ;
		attr.pickle(out, target_values[i]) ;
	    } else {
		out.writeBoolean(false) ;
	    }
	}
    }

    /**
     * Unpickle a resource filter.
     * Unpickle the attribute of the filter, and than the shadow attribute
     * of our target.
     * @param in The input stream to unpickle form.
     * @param defs The default values for our shadow attributes.
     */

    public AttributeHolder unpickleInstance(DataInputStream in, Hashtable defs)
	throws IOException 
    {
	// Unpickle myself:
	super.unpickleInstance(in, defs) ;
	// Unpickle my additional infos:
	FilteredResource target = (FilteredResource) getTargetResource() ;
	Class            cls    = target.getClass() ;
	this.target_attributes = AttributeRegistery.getClassAttributes(cls) ;
	this.target_values     = new Object[target_attributes.length] ;
	for (int i = 0 ; i < target_values.length ; i++) {
	    if ( in.readBoolean() ) {
		target_values[i] = target_attributes[i].unpickle(in) ;
	    } else {
		target_values[i] = null ;
	    }
	}
	return this ;
    }

    /**
     * Initialize the filter.
     * @param values The default attribute values.
     */

    public void initialize(Object values[]) {
	// Super initialize:
	super.initialize(values) ;
	// Check our target attributes:
	FilteredResource target = (FilteredResource) getTargetResource() ;
	if ( target == null )
	    throw new AttributeHolderInitException("No target !") ;
	if ( target_attributes == null ) {
	    Class cls = target.getClass() ;
	    target_attributes = AttributeRegistery.getClassAttributes(cls) ;
	}
	if ( target_values == null )
	    target_values = new Object[target_attributes.length] ;
    }

}
