// PutableDirectory.java
// $Id: PutableDirectory.java,v 1.2 1996/04/19 22:48:51 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.resources;

import java.util.*;
import java.io.*;

import w3c.tools.sorter.*;
import w3c.jigsaw.http.*;

/**
 * A Directoryresource that is able to create documents in reply to PUT.
 */

public class PutableDirectory extends DirectoryResource {
    /**
     * Attribute index - Allow the GNN browse method.
     */
    protected static int ATTR_BROWSABLE = -1 ;

    static {
	Attribute   a = null ;
	Class     cls = null;

	try {
	    cls     = Class.forName("w3c.jigsaw.resources.PutableDirectory") ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
	// The browsable flag:
	a = new BooleanAttribute("browsable"
				 , Boolean.FALSE
				 , Attribute.EDITABLE) ;
	ATTR_BROWSABLE = AttributeRegistery.registerAttribute(cls, a) ;
    }

    /**
     * Get this class browsable flag.
     */

    public boolean getBrowsableFlag() {
	return getBoolean (ATTR_BROWSABLE, false) ;
    }

    /**
     * Create  new resource.
     * @param name The identifier of the new resource.
     * @param request The request that trigered this resource creation.
     */

    protected HTTPResource createResource(String name, Request request)
	throws HTTPException
    {
	// Create an empty file:
	File    file    = new File(getDirectory(), name) ;
	boolean created = false ;

	if ( ! file.exists() ) {
	    try {
		(new RandomAccessFile(file, "rw")).close() ;
		created = true ;
	    } catch (Exception ex) {
		created = false ;
	    }
	}
	if ( ! created ) {
	    // Error, we don't wan't to allow to overide files !
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent("Unable to create the appropriate file: "+file);
	    throw new HTTPException (error) ;
	}
	// Go to the indexer:
	HTTPResource resource = null ;
	try {
	    resource = createDefaultResource(name) ;
	    if ( resource == null ) {
		Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
		error.setContent("Failed to create resource "
				 + name + ". The indexer wasn't able to "
				 + "deduce a resource for it.") ;
		throw new HTTPException (error);
	    } 
	} finally {
	    // If we wasn't able to create the resource, delete empty file.
	    if ( resource == null ) 
		file.delete() ;
	}
	return resource ;
    }

    /**
     * The lookup method creates resources on the fly...
     * The funny thing here, is that you need to create the resource
     * <em>before</em> handling the PUT, since it will determine how to
     * handle it.
     * @param state The current lookup state.
     */

    public HTTPResource lookup(LookupState state)
	throws HTTPException
    {
	String       name   = state.peekNextComponent() ;
	HTTPResource result = null ;
	if ( state.countRemainingComponents() > 1 ) {
	    // We are just crossing the directory:
	    result = super.lookup(state) ;
	} else {
	    // We might well want to create a resource:
	    result = lookup(name) ;
	    if (result == null) {
		Request request = state.getRequest() ;
		if ((request == null) || request.getMethod().equals("PUT"))
		    result = createResource(name, request) ;
	    }
	    state.getNextComponent() ;
	}
	return result ;
    }

    /**
     * Handle the browse method.
     * @param request The request to handle.
     */

    public Reply browse (Request request)
	throws HTTPException
    {
	Enumeration  enum      = enumerateResourceIdentifiers() ;
	Vector       resources = Sorter.sortStringEnumeration(enum) ;
	int          rsize     = ((resources == null) ? 0 : resources.size()) ;
	StringBuffer sb        = new StringBuffer() ;

	// As we have enumerated all resources, just looking the store is ok
	for (int i = 0 ; i < rsize ; i++) {
	    String       rname = (String) resources.elementAt(i) ;
	    HTTPResource r     = lookupStore(rname) ;
	    if ( r instanceof DirectoryResource ) {
		sb.append("application/x-navidir "
			  + rname
			  + "\r\n") ;
	    } else {
		sb.append(r.getContentType().toString()
			  + " "
			  + rname
			  + "\r\n") ;
	    }
	}
	Reply reply = request.makeReply(HTTP.OK) ;
	reply.setContent(sb.toString()) ;
	reply.setContentType("application/x-navibrowse") ;
	return reply ;
    }

    /**
     * We hanlde (in some cases) one extended method.
     * @param request The request to handle.
     */

    public Reply extended (Request request)
	throws HTTPException
    {
	String method = request.getMethod() ;
	if ((method != null) && method.equals("BROWSE") && getBrowsableFlag())
	    return browse(request) ;
	return super.extended(request) ;
    }

}
