// FilteredResource.java
// $Id: FilteredResource.java,v 1.4 1996/05/28 14:37:16 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.resources ;

import java.util.* ;
import java.io.* ;

import w3c.jigsaw.http.* ;

class OutgoingElement {
    ResourceFilter filter = null ;
    int            status = 0 ;
    
    OutgoingElement (ResourceFilter filter, int status) {
	this.filter = filter ;
	this.status = status ;
    }
}


/**
 * Filtered resources.
 * A filtered resource is a resource that can be filtered through any filters.
 * Filters are trigered at three points during request processing:
 * <ul>
 * <li>During the lookup, so that they have a chance to check any access
 * rights,
 * <li>Before the request is actually processed by its target resource,
 * <li>And optionally, after the request has been processed (depending on
 * the return value of the second stage.).
 * </ul>
 * @see w3c.jigsaw.resources.ResourceFilter
 */

public class FilteredResource extends ContainerResource {
    /**
     * The pseudo field of a request we use to memorize what filters is to
     * be called on the way out.
     */
    private static String APPLIED_FILTERS="w3c.jigsaw.resources.filters";
    /**
     * Our list of filters.
     */
    protected ResourceFilter filters[] = null ;

    /**
     * Get our whole list of filters.
     */

    public synchronized ResourceFilter[] getFilters() {
	return filters;
    }
    
    /**
     * Register a new filter to this resource.
     * This method register a new filter to the this filtered resource. The
     * provided filter should be an <em>uninitialized</em> ResourceFilter
     * instance.
     * @param The uninitialized filter.
     */

    public synchronized void registerFilter(ResourceFilter filter
					    , Hashtable defs) {
	// Initialize the filter:
	if ( defs == null )
	    defs = new Hashtable(3) ;
	defs.put("target", this) ;
	filter.initialize(defs) ;
	// Add it to our list of filters:
	if ( filters == null ) {
	    filters    = new ResourceFilter[1] ;
	    filters[0] = filter ;
	} else {
	    // Available slot ?
	    for (int i = 0 ; i < filters.length ; i++) {
		if ( filters[i] == null ) {
		    filters[i] = filter ;
		    markModified() ;
		    return ;
		}
	    }
	    // Extend the filters array
	    ResourceFilter nfilters[] = new ResourceFilter[filters.length+1] ;
	    System.arraycopy(filters, 0, nfilters, 0, filters.length) ;
	    nfilters[filters.length] = filter ;
	    filters = nfilters ;
	}
	markModified() ;
    }

    /**
     * Unregister a filter to this target.
     * @param filter The filter to unregister.
     */

    public synchronized void unregisterFilter(ResourceFilter filter) {
	ResourceFilter filters[] = getFilters() ;
	if ( filters != null ) {
	    for (int i = 0 ; i < filters.length ; i++) {
		if ( filters[i] == filter ) 
		    filters[i] = null ;
	    }
	}
    }

    /**
     * Does this resource or one of its filters define the given attribute.
     * @param idx The attribute index.
     * @return A boolean <strong>true</strong> if attribute is defined.
     */

    public synchronized boolean definesAttribute(int idx)
	throws IllegalAttributeAccess
    {
	ResourceFilter filters[] = getFilters() ;
	if ( filters != null ) {
	    for (int i = 0 ; i < filters.length ; i++) {
		if ( filters[i] == null )
		    continue ;
		if ( filters[i].definesTargetAttribute(i) )
		    return true ;
	    }
	}
	return super.definesAttribute(idx) ;
    }

    /**
     * Filters can supply part of the attributes value for a resource.
     * This method lookup the current resource for an attribute value, 
     * and than examine all its filters, to look for a filter-supplied
     * value.
     * @param idx The index of the queried attribute.
     * @param def The default return value.
     * @return The attribute value, or the provided default value if undef.
     */

    public synchronized Object getValue(int idx, Object def)
	throws IllegalAttributeAccess
    {
	ResourceFilter filters[] = getFilters() ;
	if ( filters != null ) {
	    for (int i = 0 ; i < filters.length ; i++) {
		if ( filters[i] == null )
		    continue ;
		if ( filters[i].definesTargetAttribute(idx) )
		    return filters[i].getTargetValue(idx, null) ;
	    }
	}
	return super.getValue(idx, def) ;
    }

    /**
     * Apply filters on the way in.
     * Any filters that want to be invoked on the way back has to return
     * the appropriate code. These filters are saved into the request, so
     * that on the way back, we know wich of them we should trigger.
     * @param request The request being handled.
     * @exception HTTPException If some error is encoutered during the 
     *    filtering process.
     */

    protected synchronized void applyIngoing (Request request) 
	throws HTTPException
    {
	Vector applied = null ;
	if ( ! request.hasField (APPLIED_FILTERS) ) {
	    applied = new Vector(2) ;
	    request.defineField (APPLIED_FILTERS, applied) ;
	} else {
	    applied = (Vector) request.getFieldObject (APPLIED_FILTERS) ;
	}
	ResourceFilter filters[] = getFilters() ;
	for (int i = 0 ; i < filters.length ; i++) {
	    if ( filters[i] == null )
		continue ;
	    int e = filters[i].ingoingFilter (request) ;
	    switch (e) {
	      case ResourceFilter.CallOutgoing:
	      case ResourceFilter.ForceOutgoing:
		  OutgoingElement item = new OutgoingElement(filters[i], e);
		  applied.insertElementAt (item, 0) ;
	      case ResourceFilter.DontCallOutgoing:
	      default:
		  break ;
	    }
	}
    }

    /**
     * Apply filters on the way back.
     * Get the list of filters to be applied from the pseudo field of the
     * request, and apply the one that needs to be applied (ie the one that
     * return an appropriate code).
     * @param request The request being processed.
     * @param reply The target's resource original reply.
     * @param erred Wether the original reply was an error or not.
     * @exception HTTPException If the original reply was an error, or if 
     *    the outgoing filter failed.
     */

    protected synchronized 
    Reply applyOutgoing (Request request, Reply reply, boolean erred)
	throws HTTPException
    {
	if ( ! request.hasField (APPLIED_FILTERS) ) 
	    return reply ;
	Vector applied = (Vector) request.getFieldObject(APPLIED_FILTERS);
	for (int i = 0 ; i < applied.size() ; i++) {
	    OutgoingElement item = (OutgoingElement) applied.elementAt(i) ;
	    if ( ( ! erred )
		 || (erred && (item.status == ResourceFilter.ForceOutgoing))) {
		reply = item.filter.outgoingFilter (request, reply) ;
	    }
	}
	applied.setSize(0) ;
	return reply ;
    }


    /**
     * Check the access to this filtered resource.
     * Apply each ingoing filters, so that they can check for authorization 
     * access. If the provided state doesn't come with a request, than
     * the filters don't get applied.
     * @param state The current lookup state.
     * @exception HTTPException If access is denied.
     */

    public synchronized void checkAccess(LookupState state) 
	throws HTTPException
    {
	ResourceFilter filters[] = getFilters() ;
	if ((filters != null) && (state.hasRequest()))
	    applyIngoing(state.getRequest()) ;
    }

    /**
     * Perform the request for this filtered target resource.
     * This method applies any found filters on the way in, perform the request
     * and than, applies any filters on the way back. It finally returns
     * the appropriate reply.
     */

    public Reply perform(Request request)
	throws HTTPException
    {
	ResourceFilter filters[] = getFilters() ;
	// Apply filters on the way in:
	if ( filters != null )
	    applyIngoing(request) ;
	// Perform the request:
	Reply   reply = null ;
	boolean erred = false ;
	try {
	    reply = super.perform(request) ;
	} catch (HTTPException ex) {
	    erred = true ;
	    reply = ex.getReply() ;
	}
	// Apply the filters on the way back, or return reply:
	if ( request.hasField(APPLIED_FILTERS) )
	    return applyOutgoing(request, reply, erred) ;
	else
	    return reply ;
    }

    /**
     * Pickle a filtered resource.
     * Pickle the resource itself, along with its associated filters.
     * @param out The data output stream to pickle to.
     */
    
    public synchronized void pickle (DataOutputStream out) 
	throws IOException
    {
	super.pickle(out) ;
	// Pickle the filters:
	ResourceFilter filters[] = getFilters() ;
	if ( filters != null ) {
	    // COunt the number of available filters:
	    int cnt = 0 ;
	    for (int i = 0 ; i < filters.length ; i++) {
		if ( filters[i] != null )
		    cnt++ ;
	    }
	    // Pickle them:
	    out.writeInt(cnt) ;
	    for (int i = 0 ; i < filters.length ; i++) {
		if ( filters[i] == null )
		    continue ;
		filters[i].pickle(out) ;
	    }
	} else {
	    out.writeInt(0);
	}
    }

    /**
     * Unpickle a filtered resource.
     * @param in The input stream to unpickle from.
     * @param defs The default attributes for the unpickled resource.
     */

    public AttributeHolder unpickleInstance(DataInputStream in, Hashtable defs)
	throws IOException
    {
	super.unpickleInstance(in, defs) ;
	// Unpickle the filters:
	int cnt = in.readInt() ;
	if ( cnt > 0 ) {
	    if ( defs == null )
		defs = new Hashtable(3) ;
	    filters = new ResourceFilter[cnt] ;
	    for (int i = 0 ; i < cnt ; i++) {
		defs.put("target", this) ;
		filters[i] = (ResourceFilter)AttributeHolder.unpickle(in,defs);
	    }
	}
	return this ;
    }

    /**
     * Initialize a filtered resource.
     */

    public void initialize(Object values[]) {
	super.initialize(values) ;
    }

}

