// DirectoryResource.java
// $Id: DirectoryResource.java,v 1.6 1996/05/28 14:35:29 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.resources ;

import java.io.* ;
import java.util.* ;

import w3c.tools.sorter.*;
import w3c.jigsaw.http.* ;
import w3c.jigsaw.indexer.* ;
import w3c.jigsaw.html.* ;

/**
 * A simple, and reasonably efficient directory resource.
 * This directory resource embeds its own resource store object to keep
 * track of its next children (wich might themselves be DirectoryResource). It
 * is reasonably efficient in the sense that it won't overload the memory with 
 * unused informations. However, stay tuned for a <em>really</em> efficient
 * file based directory resource (tuned to serve only files).
 */

public class DirectoryResource extends FilteredResource
    implements ResourceStoreHolder 
{
    /**
     * Attribute index - The index for our directory attribute.
     */
    protected static int ATTR_DIRECTORY = -1 ;
    /**
     * Attribute index - The index of our store identifier.
     */
    protected static int ATTR_STOREID = -1 ;
    /**
     * Attribute index - The index for our relocate attribute.
     */
    protected static int ATTR_RELOCATE = -1 ;
    /**
     * Attribute index - The index of wether we are extensible.
     */
    protected static int ATTR_EXTENSIBLE = -1 ;
    /**
     * Attribute index - our index resource name.
     */
    protected static int ATTR_INDEX = -1 ;
    /**
     * Attribute index - The icon directory to use in dir listing.
     */
    protected static int ATTR_ICONDIR = -1 ;
    /**
     * Attribute index - The last we we physically visited the directory.
     */
    protected static int ATTR_DIRSTAMP = -1 ;
    /**
     * Attribute index - Should this directory support content negotiation.
     */
    protected static int ATTR_NEGOTIABLE = -1 ;

    static {
	Attribute a   = null ;
	Class     cls = null ;
	// Get a pointer to our class.
	try {
	    cls = Class.forName("w3c.jigsaw.resources.DirectoryResource") ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
	// The directory attribute.
	a = new FileAttribute("directory"
			      , null
			      , Attribute.COMPUTED|Attribute.DONTSAVE);
	ATTR_DIRECTORY = AttributeRegistery.registerAttribute(cls, a) ;
	// Our store identifier attribute (relative to our directory)
	a = new FilenameAttribute("storeid"
				  , ".jigidx"
				  , Attribute.EDITABLE|Attribute.MANDATORY);
	ATTR_STOREID = AttributeRegistery.registerAttribute(cls, a) ;
	// Should we relocate invalid request to this directory ?
	a = new BooleanAttribute("relocate"
				 , Boolean.TRUE
				 , Attribute.EDITABLE);
	ATTR_RELOCATE = AttributeRegistery.registerAttribute(cls, a) ;
	// Are we extensible (can we create resources on the fly):
	a = new BooleanAttribute("extensible"
				 , Boolean.TRUE
				 , Attribute.EDITABLE) ;
	ATTR_EXTENSIBLE = AttributeRegistery.registerAttribute(cls, a) ;
	// Our index resource name (optional).
	a = new StringAttribute("index"
				, null
				, Attribute.EDITABLE) ;
	ATTR_INDEX = AttributeRegistery.registerAttribute(cls, a) ;
	// Our icon directory.
	a = new StringAttribute("icondir"
				, null
				, Attribute.EDITABLE) ;
	ATTR_ICONDIR = AttributeRegistery.registerAttribute(cls,a);
	// The last time we visited the directory
	a = new DateAttribute("dirstamp"
			      , null
			      , Attribute.COMPUTED) ;
	ATTR_DIRSTAMP = AttributeRegistery.registerAttribute(cls, a) ;
	// The negotiate flag
	a = new BooleanAttribute("negotiable"
				 , Boolean.FALSE
				 , Attribute.EDITABLE) ;
	ATTR_NEGOTIABLE = AttributeRegistery.registerAttribute(cls, a) ;
    }

    /**
     * Our children resource store.
     */
    protected ResourceStore children = null ;
    /**
     * Our current (cached) directory listing.
     */
    protected HtmlGenerator listing = null ;

    /**
     * Get the children resource store repository.
     * @return A File instance, giving the path of the store.
     */

    protected File repository = null ;

    protected File getRepository() {
	if ( repository == null )
	    repository = new File(getDirectory(), getStoreIdentifier());
	return repository ;
    }

    /**
     * Acquire the directory resource store.
     * Should only be called from a synchronized method.
     */

    protected synchronized void acquireChildren() {
	if ( children == null ) {
	    ResourceStoreManager man = getServer().getResourceStoreManager();
	    children  = man.loadResourceStore(this, getRepository());
	}
	return ;
    }

    /**
     * Update a negotiable resource.
     * Given the name of a resource that exists, create or update the 
     * attributes of a resource that allows to negotiate its content.
     * <p>I hate this part here: it has nothing to do within the directory
     * resource itself, and the indexer shouldn't know that much about
     * directory resource, so I am stuck.
     * @param name The name of the newly created resource.
     */

    private String getUnextendedName(String name) {
	int strlen = name.length() ;
	for (int i = 0 ; i < strlen ; i++) {
	    // FIXME: Should use the system props to get the right sep
	    if ( name.charAt(i) == '.' ) {
		if ( i == 0 )
		    return null ;
		return name.substring(0, i) ;
	    }
	}
	return null ;
    }
		
    public synchronized void updateNegotiableResource(String name) {
	// Does the maintainer really wants us to perform this ugly hack ?
	if ( ! getNegotiableFlag() )
	    return ;
	// Check for the corresponding negotiable resource:
	String noext = getUnextendedName(name) ;
	if ( noext == null ) {
	    return ;
	} else {
	    HTTPResource r = lookup(noext) ;
	    if ((r != null) && ! (r instanceof NegotiatedResource) )
		// I know I will pay for this
		return ;
	    // Okay, continue :-(
	    NegotiatedResource negotiated = (NegotiatedResource) r ;
	    if ( negotiated == null ) {
		acquireChildren() ;
		String variants[] = new String[1] ;
		variants[0] = name ;
		negotiated  = new NegotiatedResource() ;
		Hashtable defs = new Hashtable() ;
		defs.put("identifier", noext) ;
		defs.put("server", getServer()) ;
		defs.put("url", getURL()+noext) ;
		defs.put("parent", this) ;
		defs.put("resource-store", children) ;
		defs.put("variants", variants) ;
		try {
		    negotiated.initialize(defs) ;
		} catch (Exception ex) {
		    ex.printStackTrace() ;
		    return ;
		}
		children.addResource(negotiated) ;
	    } else {
		String variants[]  = negotiated.getVariantNames() ;
		String nvariants[] = new String[variants.length+1] ;
		System.arraycopy(variants, 0, nvariants, 0, variants.length);
		nvariants[variants.length] = name ;
		negotiated.setValue(NegotiatedResource.ATTR_VARIANTS
				    , nvariants) ;
	    }
	}
    }

    /**
     * Try creating a default resource having the given name.
     * This method will make its best effort to create a default resource
     * having this name in the directory. If a file with this name exists,
     * it will check the pre-defined admin extensions and look for a match.
     * If a directory with this name exists, and admin allows to do so, it
     * will create a sub-directory resource.
     * @param name The name of the resource to try to create.
     * @return A Resource instance, if possible, <strong>null</strong>
     *    otherwise.
     */

    public synchronized HTTPResource createDefaultResource(String name) {
	// Don't automagically create resources of name '..' or '.'
	if (name.equals("..") || name.equals("."))
	    return null ;
	// Is there a file with such a name ?
	File file = new File(getDirectory(), name) ;
	if ( ! file.exists() )
	    return null ;
	// Try building a default resource for it:
	ResourceIndexer indexer = getServer().getIndexer() ;
	// Prepare a set of default parameters for the resource:
	acquireChildren() ;
	Hashtable defs = new Hashtable(10) ;
	defs.put("parent", this);
	defs.put("resource-store", children) ;
	defs.put("server", getServer()) ;
	defs.put("url", getURL() + name);
	// Try to get the indexer to create the resource:
	HTTPResource resource = indexer.createResource(getDirectory()
						       , name
						       , defs) ;
	if ( resource != null ) {
	    // Register this child in our store:
	    children.addResource(resource) ;
	    // Update or create any relevant negotiable resource:
	    if ( getNegotiableFlag() ) 
		updateNegotiableResource(name) ;
	    markModified() ;
	}
	return resource ;
    }

    /**
     * Register a new resource into this directory.
     * @param resource The uninitialized resource to be added.
     */

    public synchronized void
    registerResource(String identifier, Resource resource, Hashtable defs) {
	// Check that our children repository is still valid:
	acquireChildren() ;
	// Create a default set of attributes:
	if ( defs == null )
	    defs = new Hashtable(11) ;
	defs.put("identifier", identifier);
	defs.put("parent", this);
	defs.put("resource-store", children);
	defs.put("server", getServer()) ;
	defs.put("url", getURL() + identifier) ;
	defs.put("directory", new File(getDirectory(), identifier));
	// Initialize and register theresource to the store:
	resource.initialize(defs) ;
	children.addResource(resource);
	markModified() ;
    }

    /**
     * Unregister the resource.
     * If the resource identifier by the given identifier was a child
     * resource of this directory resource, it wil lbe removed.
     * @param identifier The identifier of the child to remove.
     */

    public synchronized void unregisterResource(String identifier) {
	// Check that our children repository is still valid:
	acquireChildren() ;
	// Unregister the resource from the store (that'all folk)
	children.removeResource(identifier) ;
	markModified() ;
    }

    /**
     * Save the current state of the children resource stroe.
     */

    public synchronized void save() {
	if ( children != null ) 
	    children.save();
    }

    /**
     * Get the physical directory exported by this resource.
     * @return A non-null File object giving the directory of this resource.
     */

    public File getDirectory() {
	return (File) getValue(ATTR_DIRECTORY, null) ;
    }

    /**
     * Get our children resource store identifier.
     * @return A non-null String object, used as our children resource store
     *    identifier.
     */

    public String getStoreIdentifier() {
	return (String) getValue(ATTR_STOREID, null) ;
    }

    /**
     * Should we relocate invalid requests to this directory.
     * @return A boolean <strong>true</strong> if we should relocate.
     */

    public boolean getRelocateFlag() {
	return getBoolean(ATTR_RELOCATE, true) ;
    }

    /**
     * Get the extensible flag value.
     * A DirectoryResource is extensible, if it is allowed to create new
     * resources out of the file system knowledge on the fly.
     * <p>Setting this flag might slow down the server. It unfortunatelly
     * defaults to <strong>true</strong> until I have a decent admin
     * program.
     * @return A boolean <strong>true</strong> if the directory is
     *    extensible.
     */

    public boolean getExtensibleFlag() {
	return getBoolean(ATTR_EXTENSIBLE, true) ;
    }

    /**
     * Get the optinal index name for this directory listing.
     */

    public String getIndex() {
	return (String) getValue(ATTR_INDEX, null) ;
    }

    /**
     * Get the optional icon directory.
     */

    public String getIconDirectory() {
	return getString(ATTR_ICONDIR, "/icons") ;
    }

    /**
     * Get the absolute time at which we examined the physicall directory.
     */

    public long getDirStamp() {
	return getLong(ATTR_DIRSTAMP, -1) ;
    }

    /**
     * Get the negotiable flag for this directory.
     * When turned to <strong>true</strong>, this flag indicates to the
     * directory resource that it should automatically build negotiated 
     * resources ont op of all existing resources.
     * <p>You should know, at least, that truning this flag on has some
     * not so small cost in terms of the size of the index files, and some
     * not so small costs in CPU time when detecting not found documents. 
     * Otherwise, in all other situations its cost is probably negligible.
     */

    public boolean getNegotiableFlag() {
	return getBoolean(ATTR_NEGOTIABLE, false) ;
    }

    /**
     * Initialize this directory resource with the given set of attributes.
     * @param values The attribute values.
     */

    public void initialize(Object values[]) {
	super.initialize(values) ;
	// Get our parent resource and compute our directory:
	File directory = null ;
	if ( ! definesAttribute(ATTR_DIRECTORY) ) {
	    // Get our parent:
	    HTTPResource parent = getParent() ;
	    if ( ! (parent instanceof DirectoryResource) )
		throw new AttributeHolderInitException("implementation bug.");
	    // Compute and set our directory attribute:
	    directory = new File(((DirectoryResource) parent).getDirectory()
				 , getIdentifier()) ;
	    setValue(ATTR_DIRECTORY, directory) ;
	} else {
	    directory = getDirectory() ;
	}
	// If I don't have an associated resource store, create a default one.
	if ( getValue(ATTR_RESOURCE_STORE, null) == null ) {
	    ResourceStoreManager man   = getServer().getResourceStoreManager();
	    ResourceStore store = man.loadResourceStore(this, getRepository());
	    setValue(ATTR_RESOURCE_STORE, store) ;
	}
	// If my URL doesn't end with a slah, correct it:
	String url = getURL() ;
	if ( ! url.endsWith("/") )
	    setValue(ATTR_URL, url+"/") ;
    }

    /**
     * The resource store manager asks us to get rid of our store.
     * The resource store manager has decided that our store hasn't been
     * used enough in the past to be worth keeping around. Shut it down
     * and get rid of any pointers we have to it, so that the garbage
     * collector will really get rid of it.
     * <p>We can still defer this operation by returning <strong>false</strong>
     * in case a user is editing the store for example.
     * @param store The store that has to be freed.
     * @return A boolean <strong>true</strong> if the resource store has been
     * shutdown properly, <strong>false</strong> otherwise.
     */

    public synchronized boolean notifyStoreUnload(ResourceStore store) {
	if ( store != children )
	    throw new RuntimeException("Inconsistency in storage manager.");
	children = null ;
	return true ;
    }

    /**
     * The resource store asks us to shutdown our associated store.
     * @param store The store to shutdown.
     */

    public synchronized void notifyStoreShutdown(ResourceStore store) {
	if ( store != children ) 
	    throw new RuntimeException("Inconsistency in storage manager:");
	children = null ;
    }

    /**
     * The resource store asks us to save our store.
     * Our store has probably been modified recently, save it.
     * @param store The store to save.
     * @return A boolean <strong>true</strong> if success.
     */

    public synchronized boolean notifyStoreStabilize(ResourceStore store) {
	try {
	    store.save() ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    return false ;
	}
	return true ;
    }

    /**
     * Enumerate all available children resource identifiers. 
     * This method <em>requires</em> that we create all our pending resources
     * if we are in the extensible mode...too bad !
     * @return An enumeration of all our resources.
     */
    
    public synchronized Enumeration enumerateResourceIdentifiers() {
	// Check that our children repository is still valid:
	acquireChildren() ;
	// If extensible, update if needed:
	if ( getExtensibleFlag() ) {
	    File directory = getDirectory() ;
	    long dirstamp  = directory.lastModified() ;
	    if ( dirstamp > getDirStamp() ) {
		String list[] = directory.list() ;
		if ( list != null ) {
		    for (int i = 0 ; i < list.length ; i++) {
			if (list[i].equals(".") || list[i].equals(".."))
			    continue ;
			if ( lookupStore(list[i]) == null )
			    createDefaultResource(list[i]) ;
		    }
		}
		setLong(ATTR_DIRSTAMP, dirstamp) ;
	    }
	}
	return children.enumerateResourceIdentifiers() ;
    }

    /**
     * Lookup our store for a child of the given name.
     * This methods may trigger, on its first call, the creation of the
     * children ResourceStore wich is done is a lazy way, so that areas
     * not often visited don't get their store loaded.
     * @param name The name of the child to lookup.
     * @return A Resource instance, or <strong>null</strong> if no
     *     match was found.
     */

    public synchronized HTTPResource lookupStore(String name) {
	// Check that our children repository is still valid:
	acquireChildren() ;
	// Is this child already loaded ?
	HTTPResource child = (HTTPResource) children.lookupResource(name) ;
	if ( child != null )
	    return child ;
	// Restore it:
	Hashtable defs = new Hashtable(5) ;
	defs.put("parent", this) ;
	defs.put("repository", getResourceStore()) ;
	defs.put("server", getServer()) ;
	defs.put("url", getURL() + name);
	// Restore the children:
	return (HTTPResource) children.loadResource(name, defs) ;
    }

    /**
     * Lookup the resource having the given name in this directory.
     * @param name The name of the resource.
     * @return A resource instance, or <strong>null</strong>.
     */

    public HTTPResource lookup(String name) {
	// Try our store:
	HTTPResource resource = lookupStore(name) ;
	if ( resource != null )
	    return resource ;
	// If allowed, than try a default fallback:
	return getExtensibleFlag() ? createDefaultResource(name) : null ;
    }

    /**
     * Lookup the next component of this lookup state in here.
     * @param state The current lookup state.
     * @return A resource instance, or <strong>null</strong> if no match
     *    was found.
     */

    public HTTPResource lookup(LookupState state)
	throws HTTPException
    {
	String       name     = state.getNextComponent() ;
	HTTPResource resource = lookup(name) ;
	// Check that directory lookups end with a slash:
	if ((! state.hasMoreComponents()) 
	    && ( resource instanceof DirectoryResource ) ) {
	    if ( ! state.isDirectory() && ! state.isInternal()) {
		Request request = state.getRequest() ;
		if ( request == null )
		    return null ;
		// Emit an appropriate relocation or error
		String url = resource.getURL() ;
		String msg = "Invalid requested URL: the directory resource "
		    + " you are trying to reach is available only through "
		    + " its full URL: <a href=\""
		    + url + "\">" + url + "</a>.";
		if ( ((DirectoryResource) resource).getRelocateFlag() ) {
		    // Emit an error (with reloc if allowed)
		    Reply reloc = request.makeReply(HTTP.MOVED_TEMPORARILY);
		    reloc.setContent(msg) ;
		    reloc.setLocation(url);
		    throw new HTTPException(reloc) ;
		} else {
		    Reply error = request.makeReply(HTTP.NOT_FOUND) ;
		    error.setContent(msg) ;
		    throw new HTTPException(error) ;
		}
	    }
	}
	return resource ;
    }

    /**
     * The time at which we generated the directory index.
     */
    protected long listing_stamp = -1 ;

    /**
     * Reply with an HTML doc listing the resources of this directory.
     * This function takes special care not to regenerate a directory listing
     * when one is available. It also caches the date of the directory 
     * listing, so that it can win big with NOT_MODIFIED.
     * <p>Using a modem, I know that each place I can reply with an 
     * NOT_MODIFIED, <strong>is</strong> a big win.
     * @param request The request to handle.
     * @exception HTTPException If processsing the request failed.
     */

    public Reply getDirectoryListing(Request request)
	throws HTTPException
    {
	// Have we already an up-to-date computed a listing ?
	if ((listing == null) 
	    || (getDirectory().lastModified() > listing_stamp)
	    || (getLastModified() > listing_stamp)) {
	    Enumeration   enum      = enumerateResourceIdentifiers() ;
	    Vector        resources = Sorter.sortStringEnumeration(enum) ;
	    HtmlGenerator g         = new HtmlGenerator("Directory of "
							+ getIdentifier());
	    g.append("<h1>"+getIdentifier()+"</h1>");
	    g.append("<pre>");
	    for (int i = 0 ; i < resources.size() ; i++) {
		String       name     = (String) resources.elementAt(i);
		HTTPResource resource = lookupStore(name);
		String       icon     = resource.getIcon() ;
		// Icon first, if available
		if ( icon != null ) 
		    g.append("<img src=\""
			     + getIconDirectory() +"/" + icon
			     + "\">");
		// Resource's name with link:
		g.append("<a href=\"" + resource.getURL() + "\">"
			 + name
			 + "</a>\r\n");
	    }
	    g.append("</pre>");
	    g.close() ;
	    listing_stamp = getLastModified() ;
	    listing       = g ;
	}
	// Is it an IMS request ?
	long ims = request.getIfModifiedSince(-1) ;
	if ((ims > 0) && (listing_stamp <= ims)) {
	    return request.makeReply(HTTP.NOT_MODIFIED) ;
	} else {
	    Reply reply = request.makeReply(HTTP.OK) ;
	    reply.setLastModified(listing_stamp) ;
	    reply.setStream(listing) ;
	    return reply ;
	}
    }

    /**
     * GET on a directory, generate a directory listing.
     * @param request The request to handle.
     */

    public Reply get(Request request)
	throws HTTPException
    {
	return getDirectoryListing(request) ;
    }

    /**
     * We are being unloaded.
     * Cleanup-up our attribute values, and make sure our store is closed.
     */

    public synchronized void notifyUnload() {
	// Cleanup up our children store, if needed:
	if ( children != null ) {
	    ResourceStoreManager man = getServer().getResourceStoreManager();
	    man.unholdResourceStore(this, getRepository()) ;
	    children = null ;
	}
	// Super unload:
	super.notifyUnload() ;
    }

    /**
     * Perform requests on a directory.
     * If an index has been given , this method delegates the actual handling
     * of the request to it.
     * @param request The request to handle.
     */

    public Reply perform(Request request) 
	throws HTTPException
    {
	String index = getIndex() ;
	if ( index != null ) {
	    HTTPResource rindex = lookupStore(index) ;
	    if ( rindex == null ) {
		Reply error = request.makeReply(HTTP.NOT_FOUND) ;
		error.setContent("The index ["
				 + index
				 + "] for this directory doesn't exist."
				 + " The server is misconfigured.");
		throw new HTTPException(error);
	    }
	    return rindex.perform(request);
	}
	return super.perform(request) ;
    }

    /**
     * Create an empty resource store, and store a default resource directory
     * in it.
     */

    public static void main(String args[]) 
	throws Exception
    {
	File   repository = new File(args[0]) ;
	File   directory  = new File(args[1]) ;
	String url        = args[2] ;
	String identifier = args[3] ;
	Class  cls        = null ;

	// Get a pointer to the class:
	try {
	    cls = Class.forName("w3c.jigsaw.resources.DirectoryResource") ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
				 
	// Create a fresh resource store:
	ResourceStore store = new SimpleResourceStore() ;
	store.initialize(null, repository) ;

	// Create the Directoryresource itself:
	Attribute         attrs[] = AttributeRegistery.getClassAttributes(cls);
	DirectoryResource resource = new DirectoryResource() ;
	
	// Create a default attribute value set for the directory resource:
	Hashtable defs = new Hashtable(11) ;
	defs.put("directory", directory) ;
	defs.put("identifier", identifier) ;
	defs.put("url", url) ;
	defs.put("resource-store", store) ;

	// Initialize the directory resource:
	resource.initialize(defs) ;

	// Add it to the store:
	store.addResource(resource) ;


	store.shutdown() ;
    }

}
