// ResourceIndexer.java
// $Id: ResourceIndexer.java,v 1.4 1996/04/18 14:02:41 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.indexer ;

import java.util.* ;
import java.io.* ;

import w3c.jigsaw.resources.* ;
import w3c.jigsaw.http.*;
import w3c.mime.* ;

/**
 * Jigsaw indexer.
 * The indexer is an object that given some global configuration informations, 
 * tries to build default resources for files that the server doesn't know
 * about.
 */

public class ResourceIndexer implements ResourceStoreHolder {
    private static boolean inited = false ;

    /**
     * Our extension database repository.
     */
    protected File extrep = null ;
    /**
     * our directory database repository.
     */
    protected File dirrep = null ;
    /**
     * Our extensions database.
     */
    protected ResourceStore extensions = null ;
    /**
     * Our directory templates database.
     */
    protected ResourceStore directories = null ;

    /**
     * The server that's using us.
     */
    protected httpd server = null ;

    /**
     * Acquire the extension store.
     * Should only be called from a synchronized method.
     */

    protected synchronized void acquireExtensionsStore() {
	if ( extensions == null ) {
	    ResourceStoreManager man = server.getResourceStoreManager() ;
	    extensions = man.loadResourceStore(this, extrep);
	}
	return ;
    }

    /**
     * Acquire the directories store.
     * Should only be called from a synchronized method.
     */

    protected synchronized void acquireDirectoriesStore() {
	if ( directories == null ) {
	    ResourceStoreManager man = server.getResourceStoreManager() ;
	    directories = man.loadResourceStore(this, dirrep);
	}
	return;
    }

    /**
     * Load an extension descriptor.
     * @param ext The name of the extension.
     * @return An instance of Extension, or <strong>null</strong>.
     */

    public Extension loadExtension (String name) {
	acquireExtensionsStore() ;
	// Bootstraping only, should be left commented.
//	if ( ! inited ) {
//	    inited = true ;
//	    Class cls = null;
//	    try {
//		cls = Class.forName("w3c.jigsaw.resources.FileResource");
//	    } catch (Exception ex) {
//		ex.printStackTrace();
//		System.exit(1);
//	    }
//	    Extension e = Extension.makeExtension("html", cls);
//	    e.setTargetValue("content-type", MIMEType.TEXT_HTML) ;
//	    extensions.addResource(e);
//	    try {
//		cls =Class.forName("w3c.jigsaw.indexer.ExtensionsEditor");
//	    } catch (Exception ex) {
//		ex.printStackTrace();
//		System.exit(1);
//	    }
//	    e = Extension.makeExtension("fun", cls);
//	    extensions.addResource(e);
//	}
	return (Extension) extensions.loadResource(name, null) ;
    }

    /**
     * Save your state.
     */

    public void save() {
	try {
	    if ( extensions != null )
		extensions.save() ;
	    if ( directories != null )
		directories.save() ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	}
    }

    /**
     * Register a new extension to the store.
     * @param e The extension to register.
     */

    public synchronized void registerExtension(Extension e) {
	acquireExtensionsStore() ;
	extensions.addResource(e) ;
    }

    /**
     * Unregister knowledge about a given extension.
     * @param ext The extension to unregister.
     */

    public synchronized void unregisterExtension(String ext) {
	acquireExtensionsStore() ;
	extensions.removeResource(ext) ;
    }

    /**
     * Get the list of extension names.
     * @return An enumeration of the known extension names.
     */

    public Enumeration enumerateExtensionNames() {
	acquireExtensionsStore() ;
	return extensions.enumerateResourceIdentifiers() ;
    }

    /**
     * Return the class (if any) that our store defines for given extension.
     * @param ext The extension we want a class for.
     * @return A Class instance, or <strong>null</strong>.
     */

    protected Class getClassFor(String ext) {
	Extension e = loadExtension(ext) ;
	return (e != null) ? e.getTargetClass() : null ;
    }

    /**
     * Merge the attributes this extension defines, with the provided ones.
     * @param attrs The attributes we want to fill with default values.
     * @param ext The extension name.
     * @param into The already built set of default values.
     * @return A Hashtable, containing the augmented set of default attribute
     *    values.
     */

    protected Hashtable mergeDefaultAttributes(Attribute attrs[]
					       , String ext
					       , Hashtable into) {
	Extension e = loadExtension(ext) ;
	if ( e != null ) {
	    for (int i = 0 ; i < attrs.length ; i++) {
		Object value = e.getTargetValue(i, null) ;
		if ( value != null )
		    into.put(attrs[i].getName(), value) ;
	    }
	}
	return into ;
    }

     
    /**
     * Get this name's extensions.
     * @param name The file name.
     * @return An array of string, giving ach (parsed) extension, or
     *    <strong>null</strong> if none was found.
     */

    protected String[] getFileExtensions(String name) {
	Vector items = new Vector() ;
	int dpos     = name.indexOf ('.') ;
	
	if ( dpos > 0 ) {
	    int pos = dpos+1 ;
	    while ( (dpos = name.indexOf ('.', pos)) != -1 ) {
		// Skip empty extension:
		if ( dpos == pos+1 ) { 
		    pos++ ;
		    continue ;
		}
		// Else add new extension:
		items.addElement (name.substring(pos, dpos)) ;
		pos = dpos + 1;
	    }
	    if ( pos < name.length() )
		items.addElement (name.substring(pos)) ;
	    String exts[] = new String[items.size()] ;
	    items.copyInto (exts) ;
	    return exts ;
	} else {
	    return null ;
	}
    }

    /**
     * Create a default file resource for this file (that exists).
     * @param directory The directory of the file.
     * @param name The name of the file.
     * @param defs A set of default attribute values.
     * @return An instance of HTTPResource, or <strong>null</strong> if
     *    we were unable to create it.
     */

    protected HTTPResource createFileResource(File directory
					      , String name
					      , Hashtable defs) {
	File  file = new File(directory, name) ;
	Class cls  = null ;      

	// Check that at least one class is defined for all the extensions:
	String exts[] = getFileExtensions(name) ;
	if ( exts == null )
	    return null ;
	for (int i = 0 ; i < exts.length ; i++) {
	    cls = getClassFor(exts[i]) ;
	    if ( cls != null )
		break ;
	}
	if ( cls == null )
	    return null ;
	// Create the runtime-time default values for attributes.
	if ( defs == null )
	    defs = new Hashtable(5) ;
	defs.put("directory", directory) ;
	defs.put("identifier", name) ;
	defs.put("last-modified", new Long(file.lastModified())) ;
	defs.put("content-length", new Integer((int) file.length())) ;
	// Merge with values defined by the extension:
	Attribute attrs[] = AttributeRegistery.getClassAttributes(cls) ;
	for (int i = exts.length ; --i >= 0 ; ) 
	    mergeDefaultAttributes(attrs, exts[i], defs) ;
	// Create, initialize and return the new resouce
	try {
	    HTTPResource resource = (HTTPResource) cls.newInstance() ;
	    resource.initialize(defs) ;
	    return resource;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    return null ;
	}
    }

    /**
     * Register this new directory template into our directory store.
     * @param template The directory template.
     */

    public synchronized void registerDirectory(Directory directory) {
	acquireDirectoriesStore() ;
	directories.addResource(directory) ;
    }

    /**
     * Unregister the given directory from our stor.
     * @param name The name of the directory to unregister.
     */

    public synchronized void unregisterDirectory(String id) {
	acquireDirectoriesStore() ;
	directories.removeResource(id) ;
    }

    /**
     * Enumerate the names of the defined directory templates.
     * @return An enumeration of directory names.
     */

    public synchronized Enumeration enumerateDirectoryNames() {
	acquireDirectoriesStore() ;
	return directories.enumerateResourceIdentifiers() ;
    }

    /**
     * Load a given directory template from the store.
     * @param name The name of the template to load.
     * @return An instance of Directory, or <strong>null</strong>.
     */

    public Directory loadDirectory(String name) {
	acquireDirectoriesStore() ;
	return (Directory) directories.loadResource(name, null) ;
    }
    
    /**
     * Create a default container resource for this directory (that exists).
     * @param directory The parent directory.
     * @param name The name of its sub-directory to index.
     * @param defaults A set of default atribute values.
     * @return An HTTP instance, or <strong>null</strong> if
     *    the indexer was unable to build a default resource for the directory.
     */

    protected HTTPResource createDirectoryResource(File directory
						   , String name
						   , Hashtable defs) {
	// Lookup the directory path, for an existing template.
	File      dir      = new File(directory, name) ;
	Class     cls      = null ;
	String    part     = null ;
	Directory template = loadDirectory(name) ;

	if ((template == null) || ((cls=template.getTargetClass()) == null)) {
	  loop:
	    while ( true ) {
		// Go up one level:
		part = dir.getName() ;
		if ( part != null ) {
		    template = loadDirectory(part) ;
		    if ((template != null)
			&& template.getGenericFlag()
			&& ((cls = template.getTargetClass()) != null) )
			break loop ;
		} else {
		    break loop ;
		}
		String parent = dir.getParent() ;
		if ( parent != null )
		    dir  = new File(dir.getParent()) ;
		else
		    break loop ;
	    }
	}
	// Create a new instance:
	HTTPResource instance = null ;
	if ( cls == null ) {
	    instance = new DirectoryResource() ;
	} else {
	    try {
		instance = (HTTPResource) cls.newInstance() ;
	    } catch (Exception ex) {
		ex.printStackTrace() ;
		return null ;
	    }
	}
	// Compute default attribute values:
	if ( defs == null )
	    defs = new Hashtable(23) ;
	if ( template != null ) {
	    Attribute attrs[] = template.getTargetAttributes() ;
	    if ( attrs != null ) {
		for (int i = 0 ; i < attrs.length ; i++) {
		    if ( template.definesTargetAttribute(i) ) 
			defs.put(attrs[i].getName()
				 , template.getTargetValue(i, null));
		}
	    }
	}
	// End up with the automated attributes:
	File file = new File(directory, name) ;
	defs.put("directory", file) ;
	defs.put("identifier", name) ;
	defs.put("last-modified", new Long(file.lastModified())) ;
	defs.put("icon", "dir.gif");
	instance.initialize(defs) ;
	return instance ;
    }

    /** 
     * Resource store holder implementation - get rid of our store.
     * @param store The store we should get rid of.
     * @return Always <strong>true</strong>.
     */

    public synchronized boolean notifyStoreUnload(ResourceStore store) {
	if ( store == extensions ) {
	    extensions = null ;
	} else if ( store == directories ) {
	    directories = null ;
	}
	return true ;
    }

    /**
     * Resource store holder implementation - Shutdown our store.
     * @param store The store we should shut down.
     */

    public synchronized void notifyStoreShutdown(ResourceStore store) {
	if ( store == extensions ) {
	    extensions = null ;
	} else if ( store == directories ) {
	    directories = null ;
	}
    }

    /**
     * Resource store holder implementation - Save the store.
     * @param store The store to be saved.
     * @return A boolean <strong>true</strong> if success.
     */

    public boolean notifyStoreStabilize(ResourceStore store) {
	try {
	    store.save() ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    return false ;
	}
	return true ;
    }

    /**
     * Try to create a resource for the given file.
     * This method makes its best efforts to try to build a default
     * resource out of a file. 
     * @param directory The directory the file is in.
     * @param name The name of the file.
     * @param defs Any default attribute values that should be provided
     *    to the created resource at initialization time.
     * @return A Resource instance, or <strong>null</strong> if the given
     *    file can't be truned into a resource given our configuration
     *    database.
     */

    public HTTPResource createResource(File directory
				       , String name
				       , Hashtable defs) {
	// Does this file exists ?
	File file = new File(directory, name) ;
	if ( ! file.exists() )
	    return null ;
	// Okay, dispatch on wether it is a file or a directory.
	if ( file.isDirectory() )
	    return createDirectoryResource(directory, name, defs) ;
	else
	    return createFileResource(directory, name, defs) ;
    }

    /**
     * Create a new resource  indexer object.
     */

    public ResourceIndexer(httpd server) {
	this.server     = server ;
	this.extrep = new File(server.getConfigDirectory(), "exts.db");
	this.dirrep = new File(server.getConfigDirectory(), "dirs.db") ;
    }
}
