// httpd.java
// $Id: httpd.java,v 1.8 1996/05/28 14:30:48 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html


package w3c.jigsaw.http ;

import java.io.* ;
import java.util.*;
import java.net.*;

import w3c.tools.timers.* ;

import w3c.jigsaw.resources.* ;
import w3c.jigsaw.indexer.* ;
import w3c.jigsaw.auth.*;

class DummyResourceStoreHolder implements ResourceStoreHolder {

    public boolean notifyStoreUnload(ResourceStore store) {
	return false ;
    }

    public void notifyStoreShutdown(ResourceStore store) {
	return ;
    }

    public boolean notifyStoreStabilize(ResourceStore store) {
	store.save() ;
	return true ;
    }
    

}

/**
 * <p>The server main class. This class can be used either through its
 * main method, to run a full httpd server, or simply by importing it
 * into your app. This latter possibility allows you to export some of
 * your application state through http.
 *
 * <p>The server itself uses this to report about memory consumption,
 * running threads, etc.
 */

public class httpd implements Runnable, PropertyMonitoring {

    /**
     * Name of the server host property.
     * The host property should be set to the name of the host running
     * this server.
     * <p>This property defaults to the local host name, although if you want
     * directory listing to work propertly, you might need to provide the 
     * full host name (including its domain).
     */
    public static final String HOST_P            = "w3c.jigsaw.host" ;
    /**
     * Name of the property giving the server root directory.
     * <p>The server root directory is used to deduce a bunch of defaults
     * properties, when they don't have any specific values.
     * <p>This property has no defaults.
     */
    public static final String ROOT_P = "w3c.jigsaw.root" ;
    /**
     * Name of the property giving the server space directory.
     * The server space directory should contain an index file, built
     * with the indexer.
     * <p>This property defaults to <w3c.jigsaw.root>/WWW.
     */
    public static final String SPACE_P            = "w3c.jigsaw.space" ;
    /**
     * Name of the server port property.
     * At initializatiojn time, the server will bind its accepting socket
     * to the host its runs on, and to the provided port.
     * <p>This property defaults to <code>8888</code>.
     */
    public static final String PORT_P            = "w3c.jigsaw.port" ;
    /**
     * Name of the server's trace property.
     * When set to true, the server will emit some traces indicating 
     * its current state by using the logger <em>trace</em> methods.
     * This property should be set to <string>true</strong> if you want
     * clients to emit traces.
     * <p>This property defaults to <strong>false</strong>.
     */
    public static final String TRACE_P           = "w3c.jigsaw.trace" ;
    /**
     * Name of the server's keep alive flag.
     * This property is used to determine wether this server should keep
     * its connection alive. Keeping connection alive requires this flag
     * to set to <strong>true</strong>, and clients to be compliant to the
     * keep alive feature as described in HTTP/1.1 specification.
     * <p>This property defaults to <strong>true</strong>.
     */
    public static final String KEEP_ALIVE_P      = "w3c.jigsaw.keep_alive" ;
    /**
     * Name of the server's connection time out property.
     * This property gives, in milliseconds, the timeout to use for
     * connections that remains idel, waiting for an incoming request.
     * <p>This property defaults to <code>10000</code> milliseconds.
     */
    public static final String KEEP_TIMEOUT_P="w3c.jigsaw.keep_alive.timeout";
    /**
     * Name of the server's request time out property.
     * The request time out property value indicates, in milliseconds, the
     * allowed duration of a request. Any request whose duration exceeds
     * this time out value will be aborted.
     * <p>This property defaults to <code>60000</code>.
     */
    public static final String REQUEST_TIMEOUT_P="w3c.jigsaw.request.timeout";
    /**
     * Name of the client thread priority property.
     * Every client threads will run at the given priority, which should be
     * in the range of valid threads priority.
     * <p>This property defaults to <code>Thread.NORM_PRIORITY</code>.
     */
    public static final String CLIENT_PRIORITY_P="w3c.jigsaw.client.priority";
    /**
     * Nam eof the property giving the client output buffer size.
     * Each clients, when not using a shuffler, has to allocate its own
     * output buffer, Output buffer size may increase/decrease significantly
     * the Jigsaw performances, so change it with care.
     * <p>This property defaults to <code>8192</code>.
     */
    public static final String CLIENT_BUFSIZE_P="w3c.jigsaw.client.bufsize";
    /**
     * Name of the property indicating wether client should be debuged.
     * When debuged, clients emit some traces, through the server logger
     * about their current state.
     * <p>This property defaults to <strong>false</strong>.
     */
    public static final String CLIENT_DEBUG_P="w3c.jigsaw.client.debug" ;
    /**
     * Name of  property that indicates if some security manager is required.
     * You usually don't want to run a security manager for the server, 
     * except in the unlikely (right now) case that you want the server to
     * be able to host agents.
     * <p>This property defaults to <string>false</strong>.
     */
    public static final String USE_SM_P = "w3c.http.useSecurityManager" ;
    /**
     * Name of property indicating the logger class to use.
     * The Jigsaw server allows you to implement your own logger. The only
     * logger provided with the core server is the 
     * <code>w3c.jigsaw.core.CommonLogger</code>, which implements the
     * common log format.
     * <p>This property defaults to <code>w3c.jigsaw.core.CommonLogger</code>.
     */
    public static final String LOGGER_P = "w3c.jigsaw.logger" ;
    /**
     * Name of the property giving the shuffler path.
     * This property should be set if you are to use the shuffler. The 
     * data shuffler is an external process to whiuch Jigsaw delegates 
     * the task of writing back document content to clients. Use this
     * when you think your server isn't fast enough.
     * <p>This should be an absloute path.
     * <p>This property has no defaults.
     */
    public static final String SHUFFLER_PATH_P   = "w3c.jigsaw.shuffler.path";
    /**
     * Name of the property giving the root store of the server.
     * The root store is the repository for the pickled version of the root 
     * resource. 
     * <p>This property defaults to the <code>.jigidx</code> file under
     * the space directory.
     */
    public static final String ROOT_STORE_P = "w3c.jigsaw.root.store";
    /**
     * Name of the property giving the name of the root resource.
     * Upon startup, or restart, the server will look in its root store
     * a resource whose name is given by this resource, and install it as
     * its root resource.
     * <p>This property defaults to <code>root</code>.
     */
    public static final String ROOT_NAME_P = "w3c.jigsaw.root.name" ;
    /**
     * Name of the property giving the path of the property file.
     * this should be used internally (for restart) only.
     * <p>This property defaults to <code>config/httpd.props</code>.
     */
    public static final String PROPS_P = "w3c.jigsaw.propfile" ;

    /**
     * The list of currently running servers.
     */
    private static Hashtable servers = new Hashtable() ;

    private Thread         thread    = null ;
    private ServerSocket   socket    = null ;
    private Logger         logger    = null ;
    private Shuffler       shuffler  = null ;
    public  EventManager   timer     = null ;
    private ClientPool     pool      = null ;

    /**
     * The server identifier can be any String.
     * This identifier is used by the configuration applets, to show all the 
     * running servers in the process, and to edit their properties, etc.
     */
    private String identifier = null ;

    /**
     * This server statistics object.
     */
    private httpdStatistics statistics = null ;
    /**
     * This server set of properties.
     */
    private httpdProperties props    = null ;
    /** 
     * Should the server run the server in trace mode ?
     */
    private boolean tracep = false ;
    /**
     * Should the server try to keep connections alive ?
     */
    private boolean keep = true ; 
    /**
     * What logger class should the server use to log accesses.
     */
    private String logger_class = null ;
    /**
     * The coordinate of the shuffler, or <strong>null</strong> is none is to 
     * be used. 
     */
    private String shuffler_path = null ;
    /**
     * The server's root directory.
     */
    private File root_dir = null ;
    /**
     * The directory containing the server exported documents.
     */
    private File space_dir = null ;
    /**
     * The server host name.
     */
    private String host = null ;
    /**
     * The server port.
     */
    private int port = 8001 ;
    /**
     * This server client debug flag.
     */
    private boolean client_debug = false ;
    /**
     * This server's request time slice, in milliseconds.
     */
    private int request_time_out = 1200000 ;
    /**
     * This server's connection allowed idle time in milliseconds.
     */
    private int connection_time_out = 1200000 ;
    /**
     * This server's client thread priority.
     */
    private int client_priority = Thread.NORM_PRIORITY ;
    /**
     * This server's clients buffer size.
     */
    private int client_bufsize = 8192 ;
    /**
     * This server root entity.
     */
    public ContainerResource root = null ;
    /**
     * This server URL.
     */
    private String url = null ;
    /**
     * Finishing (killing) the server.
     */
    private boolean finishing = false ;
    /**
     * Finishing, but restart straight up.
     */
    private boolean restarting = false ;
    /**
     * Should we reload the properties on restart ?
     */
    private boolean reinit_props = false ;
    /**
     * The indexer attached to this server.
     */
    private ResourceIndexer indexer = null ;
    /**
     * The realm catalog
     */
    private RealmsCatalog realms = null ;
    /**
     * The resource store manager for this server.
     */
    private ResourceStoreManager manager = null ;
    /**
     * The root store file repository.
     */
    private File root_store = null ;
    /**
     * The root resource's identifier.
     */
    private String root_name = null ;


    /**
     * The property monitoring implementation.
     * @param name The name of the property that has changed.
     * @return A boolean, <strong>true</strong> if the changed was taken into
     *    account, <strong>false</strong> otherwise.
     */

    public boolean propertyChanged (String name) {
	// Is this a property we are interested in ?
	if ( name.equals (TRACE_P) ) {
	    tracep = props.getBoolean(name, tracep) ;
	    errlog (name + " changed to " + tracep) ;
	    return true ;
	} else if ( name.equals (KEEP_ALIVE_P) ) {
	    keep = props.getBoolean (name, keep) ;
	    errlog (name + " changed to " + keep) ;
	    return true ;
	} else if ( name.equals (LOGGER_P) ) {
	    errlog (name + " change failed (server running)") ;
	    return false ;
	} else if (name.equals (ROOT_P)) {
	    errlog (name + " change failed (server running)") ;
	    return false ;
	} else if (name.equals (SPACE_P)) {
	    errlog (name + " change failed (server running)") ;
	    return false ;
	} else if ( name.equals(HOST_P)) {
	    errlog (name + " change failed (server running)") ;
	    return false ;
	} else if ( name.equals(PORT_P)) {
	    errlog (name + " change failed (server running)") ;
	    return false ;
	} else if ( name.equals(CLIENT_DEBUG_P) ) {
	    client_debug = props.getBoolean(name, client_debug) ;
	    errlog (name + " changed to " + client_debug) ;
	    return true ;
	} else if ( name.equals(REQUEST_TIMEOUT_P) ) {
	    request_time_out = props.getInteger(name, request_time_out);
	    errlog (name + " changed to " + request_time_out) ;
	    return true ;
	} else if ( name.equals(KEEP_TIMEOUT_P) ) {
	    connection_time_out = props.getInteger(name,connection_time_out);
	    errlog (name + " changed to " + connection_time_out) ;
	    return true ;
	} else if ( name.equals (CLIENT_PRIORITY_P) ) {
	    client_priority = props.getInteger (name, client_priority) ;
	    errlog (name + " changed to " + client_priority) ;
	    return true ;
	} else if ( name.equals (CLIENT_BUFSIZE_P) ) {
	    client_bufsize = props.getInteger (name, client_bufsize) ;
	    errlog (name + " changed to " + client_bufsize) ;
	    return true ;
	} else {
	    // We  don't care about this one
	    return true ;
	}
    }

    /**
     * Initialize some of the servers instance values from properties.
     */

    private void initializeFromProperties () {
	// Compute some default values (host and port)
	String defhost  = null ;
	String rootstr  = null ;
	String spacestr = null ;
	String storestr = null ;
	try {
	    defhost = InetAddress.getLocalHost().getHostName() ;
	} catch (UnknownHostException e) {
	    defhost = null ;
	}
	// Second stage: get property values:
	tracep         = props.getBoolean(TRACE_P,tracep) ;
	keep           = props.getBoolean(KEEP_ALIVE_P,keep) ;
	logger_class   = props.getString(LOGGER_P, null) ;
	shuffler_path  = props.getString(SHUFFLER_PATH_P, null) ;
	rootstr        = props.getString(ROOT_P, null) ;
	spacestr       = props.getString(SPACE_P, null);
	host           = props.getString(HOST_P, defhost) ;
	port           = props.getInteger(PORT_P, port) ;
	storestr       = props.getString(ROOT_STORE_P, null) ;
	root_name      = props.getString(ROOT_NAME_P, "root") ;
	// Get client properties:
	client_debug        = props.getBoolean (CLIENT_DEBUG_P, client_debug) ;
	request_time_out    = props.getInteger (REQUEST_TIMEOUT_P
						, request_time_out);
	connection_time_out = props.getInteger (KEEP_TIMEOUT_P
						, connection_time_out);
	client_priority     = props.getInteger(CLIENT_PRIORITY_P
					       , client_priority);
	client_bufsize      = props.getInteger(CLIENT_BUFSIZE_P
					       , client_bufsize);
	// Check that a host name has been given:
	if ( host == null )
	    throw new HTTPRuntimeException(this.getClass().getName()
					   , "initializeFromProperties"
					   , "w3c.jigsaw.host undefined.");
	// Default the root directory to the current directory:
	if ( rootstr == null ) {
	    // Try the current directory as root:
	    rootstr = System.getProperties().getProperty("user.dir", null) ;
	    if ( rootstr == null )
		throw new HTTPRuntimeException(this.getClass().getName()
					       , "initializeFromProperties"
					       , "w3c.jigsaw.root undefined.");
	}
	root_dir = new File(rootstr) ;
	// Default the space directory to root/WWW
	if ( spacestr == null ) 
	    space_dir = new File(root_dir, "WWW") ;
	else
	    space_dir = new File(spacestr) ;
	// Default the root store to its value:
	if ( storestr == null ) 
	    root_store = new File(space_dir, ".jigidx") ;
	else
	    root_store = new File(space_dir, storestr) ;
    }

    /**
     * Get this server statistics.
     */

    public httpdStatistics getStatistics() {
	return statistics ;
    }

    /**
     * Get this server properties.
     */

    public httpdProperties getProperties() {
	return props ;
    }

    /**
     * Get the client's debug flags from the properties.
     */

    protected final boolean getClientDebug() {
	return client_debug ;
    }
    
    /**
     * Does this server wants clients to try keeping connections alive ?
     */

    protected final boolean getClientKeepConnection() {
	return keep ;
    }

    /**
     * Get the request allowed time slice from the properties.
     */

    protected final int getRequestTimeOut() {
	return request_time_out ;
    }

    /**
     * Get the connection allowed idle time from the properties.
     */

    protected final int getConnectionTimeOut() {
	return connection_time_out ;
    }

    /**
     * Get the client's threads priority from the properties.
     */

    protected final int getClientThreadPriority() {
	return client_priority ;
    }

    /**
     * Get the client's buffer size.
     */

    protected final int getClientBufferSize() {
	return client_bufsize ;
    }

    /**
     * Get this server host name.
     */

    public String getHost () {
	return host ;
    }

    /**
     * Get this server port number.
     */

    public int getPort () {
	return port ;
    }

    /**
     * Get the server root entity
     */

    public ContainerResource getRoot() {
	return root ;
    }

    /**
     * Get the server URL.
     */

    public String getURL() {
	if ( url == null )
	    url = "http://" + host + ":" + port ;
	return url ;
    }

    /**
     * Get the server software string.
     */

    public String getSoftware () {
	return "Jigsaw/1.0a" ;
    }

    /**
     * Get the server local port
     */

    public int getLocalPort() {
	return socket.getLocalPort() ;
    }

    /**
     * Get this server identifier.
     */

    public String getIdentifier() {
	return identifier ;
    }

    /**
     * Get the server inet address
     */

    public InetAddress getInetAddress() {
	return socket.getInetAddress() ;
    }

    /**
     * Get this server root directory.
     */

    public File getRootDirectory() {
	return root_dir ;
    }

    /**
     * Get this server config directory.
     */

    public File getConfigDirectory() {
	return new File(getRootDirectory(), "config") ;
    }

    /**
     * Get this server resource indexer.
     */

    public ResourceIndexer getIndexer() {
	return indexer ;
    }

    /**
     * Get this server realm catalog.
     */
    
    public RealmsCatalog getRealmsCatalog() {
	return realms ;
    }

    /**
     * Get this server resourcestore manager.
     */

    public ResourceStoreManager getResourceStoreManager() {
	return manager ;
    }

    /**
     * Cleanup the resources associated with this server context.
     * This method should only be called by the server thread itself, when
     * it is requested to perform the cleanup.
     * @param restart If <strong>true</strong> the server is restarted 
     *     (reinitialized) straight away.
     */

    protected synchronized void cleanup(boolean restart) {
	// Close the accepting socket:
	try {
	    socket.close() ;
	    socket = null ;
	} catch (IOException ex) {
	    errlog ("[cleanup]: IOException while closing server socket.");
	}
	// Shutdow all object that need to be shutdown:
	if ( manager != null )
	    manager.shutdown() ;
	manager = null ;
	if ( pool != null )
	    pool.shutdown(true) ;
	pool = null ;
	if ( shuffler != null )
	    shuffler.shutdown() ;
	shuffler = null ;
	// Unregister to property monitoring
	props.unregisterObserver (this) ;
	errlog ("[shutdown]: completed.") ;
	// Finally close the log
	if ( logger != null )
	    logger.shutdown() ;
	logger = null ;
	// Release any other pointers:
	timer.stopEventManager() ;
	System.out.println ("[httpd]: " + getURL() + " done.") ;
	System.out.flush() ;
	// Keep the data neede to reinit (in case needed)
	File            init_propfile   = props.getFile(PROPS_P, null) ;
	httpdProperties init_props      = props ;
	String          init_identifier = identifier ;
	// Release pointed data:
	identifier = null ;
	manager    = null ;
	pool       = null ;
	shuffler   = null ;
	props      = null ;
	indexer    = null ;
	root       = null ;
	realms     = null ;
	logger     = null ;
	socket     = null ;
	timer      = null ;
	thread     = null ;
	url        = null ;
	restarting = false ;
	finishing  = false ;
	if ( restart ) {
	    try {
		if ( reinit_props ) 
		    init_props = loadProperties(init_propfile) ;
		reinit_props = false ;
		initialize(init_identifier, init_props) ;
	    } catch (Exception ex) {
		// We really can't do more than this here:
		System.out.println("*** server restart failed.") ;
		ex.printStackTrace() ;
	    }
	}
    }

    /**
     * Shutdown the server properly.
     * This methods shutdown the server, and clean-up all its associated 
     * resources. If the current thread is not the server thread, it unblocks
     * the server thread from its accept() call, and forces it to perform
     * the rest of the shutdown operation itself.
     * @see httpd#cleanup
     */

    public synchronized void shutdown () {
	errlog ("[shutdown]: inited !") ;
	finishing = true ;
	try {
	    Socket unlock = new Socket(host, port) ;
	    unlock.close() ;
	} catch (IOException ex) {
	    errlog ("[shutdown]: IOException while unblocking server thread.");
	}
    }

    /**
     * Restart the server properly.
     * This methods restarts the server. It cleans-up all its associated 
     * resources, and reinitialize it from scratch. If the current thread is
     * not the server thread, it unblocks
     * the server thread from its accept() call, and forces it to perform
     * the rest of the restart operation itself.
     * @param reload_properties Should we reload the properties from the
     *    property file, or should we just reinitialize from the current set
     *    of properties.
     * @see httpd#cleanup
     */

    public synchronized void restart (boolean reload_properties) {
	errlog ("[restart]: inited !") ;
	finishing    = true ;
	restarting   = true ;
	reinit_props = reload_properties ;
	try {
	    Socket unlock = new Socket(host, port) ;
	    unlock.close() ;
	} catch (IOException ex) {
	    errlog ("[restart]: IOException while unblocking server thread.");
	}
    }

    /**
     * Turn debugging on/off for this instance of httpd server.
     * @param A boolean, true turns debugging on, flase turns it off.
     */

    public void debug (boolean onoff) {
	tracep = onoff ;
    }

    /**
     * Emit a server trace. Traces are used solely for debugging purposes. You
     * should either use <b>log</b> or <b>error</b> to report informations.
     * @param client The client object which wants to report the trace.
     * @param msg The trace message.
     * @see error
     * @see log
     */

    public void trace (Client client, String msg) {
	if ( tracep && (logger != null) )
	    logger.trace (client, msg) ;
    }

    /**
     * Emit a server trace, on behalf of the server itself.
     * @param msg The trace the server wants to emit.
     */
    
    public void trace (String msg) {
	if ( tracep && (logger != null))
	    logger.trace (msg) ;
    }

    /**
     * Emit a log entry.
     * @param client The client whose request is to be logged.
     * @param request The request that has been handled.
     * @param reply The emitted reply.
     * @param nbytes The number of bytes emitted back to the client.
     * @param duration The time it took to process the request.
     */
    
    public void log (Client client
		     , Request request, Reply reply
		     , int nbytes
		     , long duration) {
	if ( logger != null )
	    logger.log (request, reply, nbytes, duration) ;
	statistics.updateStatistics(client, request, reply, nbytes, duration) ;
    }

    /**
     * Emit a server error on behalf of some client object.
     * @param client The client.
     * @param msg The error message.
     */

    public void errlog (Client client, String msg) {
	if ( logger != null )
	    logger.errlog (client, msg) ;
    }

    /**
     * Emit an error on behalf of the server.
     * @param msg The error message.
     */

    public void errlog (String msg) {
	if ( logger != null )
	    logger.errlog ("["+identifier+"]"+msg) ;
    }

    /**
     * Emit a fatal error.
     * @param e Any exception that caused the error.
     * @param msg Any additional message.
     */

    public void fatal (Exception e, String msg) {
	System.out.println ("*** Fatal Error, aborting") ;
	System.out.println (this.getClass().getName() + ": " + msg) ;
	e.printStackTrace() ;
	throw new RuntimeException (msg) ;
    }

    public void fatal(String msg) {
	System.out.println("*** Fatal error, aborting") ;
	System.out.println(this.getClass().getName() + ": " + msg) ;
	throw new RuntimeException(msg) ;
    }
	
    /**
     * Emit a warning.
     * Warnings are emited, typically if the configuration is inconsistent,
     * and the server can continue its work.
     * @param msg The warning message.
     */

    public void warning (String msg) {
	System.out.println ("*** Warning : " + msg) ;
    }
	
    /**
     * Emit a warning.
     * @param e Any exception.
     * @param msg Any message.
     */

    public void warning (Exception e, String msg) {
	System.out.println ("*** Warning: " + msg) ;
	e.printStackTrace() ;
    }

    /**
     * Get a shuffler for this server's client.
     * Whenever possible, we use a shuffler program to speed up communication
     * with the client. This methods return whatever the server deems 
     * appropriate for this client shuffler.
     * @return A Shuffler instance, or <strong>null</strong>.
     * @see w3c.jigsaw.core.Shuffler
     */

    public synchronized Shuffler getShuffler (Client client) {
	return shuffler ;
    }

    public void run () {
	while ( ( ! finishing) && ( socket != null ) ) {
	    Socket ns = null ;
	    try {
		ns = socket.accept() ;
	    } catch (IOException e) {
		e.printStackTrace() ;
		errlog ("failed to accept incoming connection on"+socket) ;
	    }
	    if ( (socket != null) && (ns != null) && (pool != null) ) 
		pool.handleConnection (ns) ;
	}
	// Our socket has been closed, perform associated cleanup.
	cleanup(restarting) ;
    }

    /**
     * Load the default set of properties for the server.
     * Once loaded, the properties are installed system wide.
     * @param propfile The property file to load.
     * @return An instance of httpdProperties, wrapping the system properties 
     *    and the server's specific properties together.
     */

    public httpdProperties loadProperties(File propfile) {
	System.out.println("[httpd] loading props: "+propfile) ;
	// Get the properties for this server:
	httpdProperties props = new httpdProperties(System.getProperties()) ;
	if ( propfile != null ) {
	    System.out.println ("loading properties from: " + propfile) ;
	    try {
		props.load (new FileInputStream(propfile)) ;
	    } catch (FileNotFoundException ex) {
		System.out.println ("Unable to load properties: "+propfile);
		System.out.println ("\t"+ex.getMessage()) ;
		System.exit (1) ;
	    } catch (IOException ex) {
		System.out.println ("Unable to load properties: "+propfile);
		System.out.println ("\t"+ex.getMessage()) ;
		System.exit (1) ;
	    }
	    System.setProperties (props) ;
	}
	return props ;
    }

    /**
     * Initialize a new HTTP server.
     * The server wil first be initialized from the available properties,
     * it will than startup, and finally run in its own thread.
     * @param identifier The string identifying this server's occurence.
     * @param props A set of properties to initialize from.
     * @exception IOException If some IO or network operation failed.
     */

    public void initialize (String identifier, Properties props) 
	throws IOException
    {
	// Initialize from properties:
	if ( ! (props instanceof httpdProperties) ) 
	    this.props = new httpdProperties (props) ;
	else
	    this.props = (httpdProperties) props ;
	this.props.registerObserver ((PropertyMonitoring) this) ;
	initializeFromProperties () ;
	// Create the resource indexer object
	this.indexer = new ResourceIndexer(this) ;
	// Create the resource store manager
	this.manager = new ResourceStoreManager(this) ;
	// Resurect this server root entity:
	ResourceStoreHolder dummy = new DummyResourceStoreHolder() ;
	ResourceStore store = null;
	store = manager.loadResourceStore(dummy, root_store);
	Hashtable defs = new Hashtable(11) ;
	defs.put("resource-store", store) ;
	defs.put("url", getURL());
	defs.put("directory", space_dir) ;
	defs.put("server", this) ;
	try {
	    this.root = (ContainerResource) store.loadResource(root_name,defs);
	} catch (Exception ex) {
	    fatal(ex, "Unable to restore server root resource !");
	}
	if ( root == null ) 
	    fatal("Unable to restore root resource ["
		  + root_name
		  + "] from repository ["
		  + root_store
		  + "]") ;
	// Resurect the realms catalog
	this.realms = new RealmsCatalog(this) ;
	// Initialize the logger object:
	if ( logger_class != null ) {
	    try {
		logger = (Logger) Class.forName(logger_class).newInstance() ;
		logger.initialize (this) ;
	    } catch (Exception e) {
		fatal (e, "unable to create logger " + logger_class ) ;
	    }
	} else {
	    warning ("no logger specified, not logging.");
	}
	// Initialize the statistics object:
	statistics = new httpdStatistics(this) ;
	// Initialize the shuffler object:
	if ( shuffler_path != null ) {
	    try {
		this.shuffler = new Shuffler (shuffler_path) ;
	    } catch (Error e) {
		warning ("unable to launch shuffler to " 
			 + shuffler_path
			 + ": " + e.getMessage()) ;
		this.shuffler = null ;
	    } catch (Exception e) {
		warning (e, "unable to launch shuffler to " 
			 + shuffler_path
			 + ": " + e.getMessage()) ;
		this.shuffler = null ;
	    }
	}
	if ( this.shuffler != null )
	    trace ("using shuffler at: " + shuffler_path) ;
	// Create the socket:
	this.socket  = new ServerSocket (port, 128) ;
	// Create this server event manager 
	this.timer   = new EventManager () ;
	this.timer.setDaemon(true);
	this.timer.start() ;
	// Set the server identifier
	this.identifier = identifier ;
	this.pool     = new ClientPool (this) ;
	this.thread   = new Thread (this) ;
	this.thread.setName (identifier) ;
	this.thread.setPriority (Thread.MAX_PRIORITY) ;
	this.thread.start() ;
	System.out.println ("[httpd]: listening at:" + getURL());
	System.out.flush() ;
	trace("started.");
    }

    /**
     * Create a new server instance in this process.
     * @param identifier The server's identifier.
     * @param props The server properties.
     */

    public httpd(String identifier, Properties props)
	throws IOException 
    {
	initialize(identifier, props) ;
    }

    public static void usage () {
	PrintStream o = System.out ;

	o.println("usage: httpd [OPTIONS]") ;
	o.println("-port <number>    : listen on the given port number.");
	o.println("-host <host>      : full name of host running the server.");
	o.println("-root <directory> : root directory of server.") ;
	o.println("-space <directory>: space directory exported by server") ;
	o.println("-p     <propfile> : property file to read.");
	o.println("-trace            : turns debugging on.") ;
	System.exit (1) ;
    }

    public static void main (String args[]) {
	Integer cmdport  = null ;
	String  cmdhost  = null ;
	String  cmdroot  = null ;
	String  cmdspace = null ;
	String  cmdprop  = null ;
	Boolean cmdtrace = null ;

	// Parse command line options:
	for (int i = 0 ; i < args.length ; i++) {
	    if ( args[i].equals ("-port") ) {
		try {
		    cmdport = new Integer(args[++i]) ;
		} catch (NumberFormatException ex) {
		    System.out.println ("invalid port number ["+args[i]+"]");
		    System.exit (1) ;
		}
	    } else if ( args[i].equals ("-host") ) {
		cmdhost = args[++i] ;
	    } else if ( args[i].equals ("-root") ) {
		cmdroot = args[++i] ;
	    } else if ( args[i].equals ("-space") ) {
		cmdspace = args[++i] ;
	    } else if ( args[i].equals ("-p") ) {
		cmdprop = args[++i] ;
	    } else if ( args[i].equals ("-trace") ) {
		cmdtrace = Boolean.TRUE;
	    } else if ( args[i].equals ("?") || args[i].equals ("-help") ) {
		usage() ;
	    } else {
		System.out.println ("unknown option: ["+args[i]+"]") ;
		System.exit (1) ;
	    }
	}
	// Get the properties for this server:
	Properties props = new httpdProperties(System.getProperties()) ;
	if (cmdprop == null) {
	    if (cmdroot == null)
		cmdroot = props.getProperty("user.dir", null);
	    // Try to guess it, cause it is really required:
	    File guess = new File (new File(cmdroot, "config"),"httpd.props");
	    cmdprop = guess.getAbsolutePath() ;
	}
	if ( cmdprop != null ) {
	    System.out.println ("loading properties from: " + cmdprop) ;
	    try {
		File propfile = new File(cmdprop) ;
		props.load (new FileInputStream(propfile)) ;
		props.put (PROPS_P, propfile.getAbsolutePath()) ;
	    } catch (FileNotFoundException ex) {
		System.out.println ("Unable to load properties: "+cmdprop);
		System.out.println ("\t"+ex.getMessage()) ;
		System.exit (1) ;
	    } catch (IOException ex) {
		System.out.println ("Unable to load properties: "+cmdprop);
		System.out.println ("\t"+ex.getMessage()) ;
		System.exit (1) ;
	    }
	    System.setProperties (props) ;
	}
	// Override properties with our command line options:
	if ( cmdport != null ) 
	    props.put ("w3c.jigsaw.port", cmdport.toString()) ;
	if ( cmdhost != null ) 
	    props.put ("w3c.jigsaw.host", cmdhost) ;
	if ( cmdroot != null )
	    props.put ("w3c.jigsaw.root", cmdroot) ;
	if ( cmdspace != null )
	    props.put ("w3c.jigsaw.space", cmdspace) ;
	if ( cmdtrace != null ) {
	    props.put ("w3c.jigsaw.trace", "true") ;
	    props.put ("w3c.jigsaw.client.debug", "true") ;
	}
	// Install security manager if needed:
	if (Boolean.getBoolean(USE_SM_P)) {
	    SecurityManager sm = new httpdSecurityManager() ;
	    System.setSecurityManager (sm) ;
	}
	// Run the server:
	try {
	    httpd server = new httpd ("httpd", props) ;
	} catch (Exception e) {
	    System.out.println ("*** [httpd]: fatal error, exiting !") ;
	    e.printStackTrace () ;
	}
    }
}
