// Request.java
// $Id: Request.java,v 1.4 1996/05/28 14:03:24 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.http ;

import java.io.*;
import java.util.*;

import w3c.mime.* ;

/**
 * this class extends the normal MIMEHeader object to cope with HTTP request.
 * One subtely here: note how each field acessor <em>never</em> throws an
 * exception, but rather is provided with a default value: this is in the hope
 * that sometime, HTTP will not require all the parsing it requires right now.
 */

public class Request extends MIMEHeaders {
    protected Client     client   = null ;

    /**
     * Unescape a HTTP escaped string
     * @param s The string to be unescaped
     * @return the unescaped string.
     */

    public static String unescape (String s) {
	StringBuffer sbuf = new StringBuffer () ;
	int l  = s.length() ;
	int ch = -1 ;
	for (int i = 0 ; i < l ; i++) {
	    switch (ch = s.charAt(i)) {
	      case '%':
		ch = s.charAt (++i) ;
		int hb = (Character.isDigit ((char) ch) 
			  ? ch - '0'
			  : Character.toLowerCase ((char) ch) - 'a') & 0xF ;
		ch = s.charAt (++i) ;
		int lb = (Character.isDigit ((char) ch)
			  ? ch - '0'
			  : Character.toLowerCase ((char) ch) - 'a') & 0xF ;
		sbuf.append ((char) ((hb << 4) | lb)) ;
		break ;
	      case '+':
		sbuf.append (' ') ;
		break ;
	      default:
		sbuf.append ((char) ch) ;
	    }
	}
	return sbuf.toString() ;
    }
    
    /**
     * Get the request content type.
     * @param def The default return value.
     * @return A MIMEType object giving the request content type, or the 
     *    provided default if content-type isn't properly defined.
     * @see w3c.mime.MIMEType
     */

    public MIMEType getContentType (MIMEType def) {
	MIMEType type = def ;
	try {
	    type = getFieldMIMEType ("content-type") ;
	} catch (MIMEException e) {
	}
	return type;
    }

    public MIMEType getContentType() {
	return getContentType(null) ;
    }

    /**
     * Get the request content length.
     * @param def The default value.
     * @return An integer, giving the content length, or the provided 
     *    default value if not properly defined.
     */

    public int getContentLength (int def) {
	int length = def ;
	try {
	    length = getFieldInt ("content-length") ;
	} catch (MIMEException e) {
	}
	return length ;
    }

    public int getContentLength() {
	return getContentLength(-1) ;
    }

    /**
     * Get the request accept-encodings.
     * @param def The default value.
     * @return An array of String, each string being a content encoding
     *    scheme, or the provided default value.
     */

    public String[] getAcceptEncodings (String def[]) {
	String encodings[] = def ;
	try {
	    encodings = getFieldList ("accept-encoding") ;
	} catch (MIMEException e) {
	}
	return encodings ;
    }

    /**
     * Get the request accepted MIME types.
     * @param def The default value.
     * @return An array of instances of Accept, or the provided default value.
     * @see w3c.mime.Accept
     */

    public Accept[] getAccept(Accept def[]) {
	Accept accept[] = def ;
	try {
	    accept = getFieldAccept ("accept") ;
	} catch (MIMEException e) {
	}
	return accept ;
    }

    /**
     * Get this request if-modified-since header field value.
     * @param def The default value.
     * @return A long, giving the header's value, or the default value.
     * @exception HTTPException If no such header was defined.
     */

    public long getIfModifiedSince (long def) {
	try {
	    return getFieldDate ("if-modified-since") ;
	} catch (MIMEException e) {
	}
	return def ;
    }

    public long getIfModifiedSince() {
	return getIfModifiedSince(-1) ;
    }

    /**
     * Get the request requested URI.
     * @param def The default value.
     * @return A String giving the unparsed requested URI, or the provided 
     *    default value.
     */

    public String getURI(String def) {
	String uri = def ;
	try {
	    uri = getFieldString ("uri") ;
	} catch (MIMEException e) {
	}
	return uri ;
    }

    public String getURI() {
	return getURI(null) ;
    }

    /**
     * Get this request input stream.
     * @return An instance of InputStream, or <strong>null<strong> if the
     *   request doesn't have an input stream.
     */

    public InputStream getInputStream () {
	return (client != null) ? client.getInputStream() : null ;
    }

    /**
     * Get the list of accepted languages for this request.
     * @param def The default value.
     * @return An array of Language objects, or the provided default value.
     * @see w3c.mime.Language
     */

    public Language[] getAcceptLanguage(Language def[]) {
	Language language[] = def ;
	try {
	    language = getFieldLanguage ("accept-language") ;
	} catch (MIMEException e) {
	}
	return language ;
    }

    public Language[] getAcceptLanguage() {
	return getAcceptLanguage(null) ;
    }

    /**
     * Get the name of the method of this request.
     * @return A String giving the name of the method, or the provided
     *     default value.
     */
    
    public String getMethod (String def) {
	String method = def ;
	try {
	    method = getFieldString("method") ;
	} catch (MIMEException e) {
	}
	return method ;
    }

    public String getMethod() {
	return getMethod(null) ;
    }

    /**
     * Get any authorization information associated with the request.
     * @param def The default value.
     * @return A String containing the autorization informations, or the
     *    default value.
     */

    public String getAuthorization (String def) {
	String authorization = def ;
	try {
	    authorization = getFieldString ("authorization") ;
	} catch (MIMEException e) {
	}
	return authorization ;
    }

    public String getAuthorization() {
	return getAuthorization(null);
    }

    /**
     * Get this request connection header.
     * @return The String value of the connectionm header, or the provided 
     *    default value.
     */

    public String getConnection(String def) {
	String connection = def ;
	try {
	    connection = getFieldString ("connection") ;
	} catch (MIMEException e) {
	}
	return connection ;
    }

    /**
     * Get the proxy connection header for this request.
     * @param def The default value.
     * @return A String giving the value of the header, or the provided 
     *    default value.
     */
    
    public String getProxyConnection(String def) {
	String connection = def ;
	try {
	    connection = getFieldString ("proxy-connection") ;
	} catch (MIMEException e) {
	}
	return connection ;
    }

    /**
     * Get this request query string.
     * @param def The default value.
     * @return A String giving the value of the query string for this request
     *    or the provided default value.
     */

    public String getQueryString (String def) {
	String query = def ;
	try {
	    query = getFieldString ("query") ;
	} catch (MIMEException e) {
	}
	return query ;
    }

    public String getQueryString() {
	return getQueryString(null) ;
    }

    /**
     * Get the String version of this request.
     * @param def The default value.
     * @return A String describing the version of the request.
     */

    public String getHTTPVersion(String def) {
	String version = def ;
	try {
	    version = getFieldString ("http-version") ;
	} catch (MIMEException e) {
	}
	return version ;
    }

    public String getHTTPVersion() {
	return getHTTPVersion(null);
    }

    protected Bag parseSingleBag (PushbackInputStream in) 
	throws HTTPException
    {
	int ch = -1 ;
	try {
	    // skip spaces, check for opening bracket
	    while (((ch = in.read()) != -1) && (ch <= 32))
		;
	    if ( ch == -1 )
		return null ;
	    if (ch != '{')
		throw new HTTPException ("Invalid Bag format (no {).") ;
	    // skip spaces, bag list separators, get the bag name
	    while (((ch = in.read()) != -1) && ((ch <= 32) || (ch == ',')))
		;
	    StringBuffer sb = new StringBuffer() ;
	    if ( ch == -1 )
		throw new HTTPException ("Invalid Bag format (no name).") ;
	    sb.append ((char) ch) ;
	    while (((ch = in.read()) != -1) && (ch > 32) && (ch != '}'))
		sb.append ((char) ch) ;
	    Bag bag = new Bag (sb.toString()) ;
	    // get items:
	    while (ch != -1) {
		if ( ch <= 32 ) {
		    ch = in.read() ;
		    continue ;
		} else if ( ch == '}' ) {
		    return bag ;
		} else if ( ch == '{' ) {
		    in.unread (ch) ;
		    Bag subbag = parseSingleBag (in) ;
		    bag.addBag (subbag) ;
		    ch = in.read() ;
		} else if ( ch == '\"') {
		    // get quoted string (FIXME, escape chars !)
		    sb.setLength(0);
		    while (((ch = in.read()) != -1) && (ch != '\"')) 
			sb.append ((char) ch) ;
		    bag.addItem (sb.toString()) ;
		    ch = in.read() ;
		} else {
		    // get token (FIXME, see token spec and tspecials)
		    sb.setLength(0) ;
		    sb.append ((char) ch) ;
		    while (((ch = in.read()) != -1) 
			   && (ch > 32) 
			   && (ch != '}')) 
			sb.append ((char) ch) ;
		    bag.addItem (sb.toString()) ;
		}
	    }
	    return bag ;
	} catch (IOException e) {
	    // I know that I am parsing a StringBuffer here, so this would be 
	    // really funny !
	    throw new HTTPException ("IOException while parsing bag.") ;
	}
    }

    /**
     * Parse a Bag formatted header. 
     * I wonder why PEP specifies that headers are a sequence of bags, 
     * rather than one single bag.
     * @param bagstr The string MIME compliant representation of the bag.
     * @return An instance of Bag, describing the Bag value.
     * @see w3c.jigsaw.core.Bag
     */

    protected Bag parseBag (String bagstr) 
	throws HTTPException 
    {
	PushbackInputStream is = (new PushbackInputStream 
				  (new StringBufferInputStream (bagstr))) ;
	Bag bag  = new Bag ("_root_") ;
	Bag sbag = null ;
	while ((sbag = parseSingleBag (is)) != null ) 
	    bag.addBag (sbag) ;
	return bag ;
    }

    /**
     * HTTP defines a new format for header value, namely <em>bag</em>. 
     * This method returns the value of a named header as a parsed Bag object.
     * @param name The name of the header.
     * @param def The default value to be returned if undefined.
     * @return A bag object.
     * @see w3c.jigsaw.core.Bag
     */

    public Bag getFieldBag (String name, Bag def) {
	Object value = headers.get (name) ;
	if ( value != null ) {
	    if ( value instanceof Bag ) {
		return (Bag) value ;
	    } else {
		return def ;
	    }
	} else {
	    try {
		String body = getField (name) ;
		Bag    bag  = parseBag (body) ;
		headers.put (name, bag) ;
		return bag ;
	    } catch (HTTPException ex) {
		return def ;
	    }
	}
    }

    public Bag getFieldBag(String name) {
	return getFieldBag(name, null) ;
    }

    /**
     * Mark this user as being authorizedf for the request.
     * @param user The name under wich the user has been authentified.
     */

    public void setUser(String user) {
	headers.put("authuser", user) ;
    }

    /**
     * Get the authentifuiied user behind this request.
     * @param def The default value if no user authentified.
     * @return A String giving the user name, or the default value.
     */

    public String getUser(String def) {
	try {
	    return getFieldString("authuser") ;
	} catch (MIMEException ex) {
	}
	return def ;
    }

    /**
     * Make an empty Reply object matching this request version.
     * @param status The status of the reply.
     */

    public Reply makeReply(Integer status) {
	return new Reply(client, getHTTPVersion(), status) ;
    }

    public Reply makeReply(int status) {
	return makeReply(new Integer(status)) ;
    }

    /**
     * Get the client of this request.
     */

    public Client getClient() {
	return client ;
    }

    Request (Client client, Hashtable headers, Hashtable rheaders) {
	super (headers, rheaders) ;
	this.client   = client ;
    }
}
