// FormCardResource.java
// $Id: FormCardResource.java,v 1.2 1996/04/12 21:00:25 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.forms;

import java.util.*;

import w3c.jigsaw.http.* ;
import w3c.jigsaw.resources.*;
import w3c.jigsaw.html.HtmlGenerator ;

public class FormCardResource extends PostableResource {
    FormResource    form     = null ;
    FormCardHandler handler  = null ;
    Vector          vfields  = null ;
    String          name     = null ;
    String          title    = null ;
    String          url      = null ;
    Vector vbuttons  = null ;
    boolean deleted = false ;

    /**
     * Mark this card as removed.
     */

    public synchronized void delete() {
	deleted = true ;
    }

    /**
     * Is this card deleted ?
     */

    public synchronized boolean isDeleted() {
	return deleted ;
    }

    /**
     * Our own lookup, to implement multiple buttons:
     */

    public HTTPResource lookup(LookupState state) 
	throws HTTPException
    {
	String label = state.getNextComponent() ;
	// Is this a valid button ?
	for (int i = 0 ; i < vbuttons.size() ; i++) {
	    String button = (String) vbuttons.elementAt(i) ;
	    if ( button.equals(label) ) {
		// Notify the button click:
		try {
		    handler.notifyButtonClick(label) ;
		} catch (FormProcessingException ex) {
		    return null ;
		}
		return this ;
	    }
	}
	// Unknown button:
	return null ;
    }

    /**
     * Get this card name.
     */

    public String getName() {
	return name ;
    }

    /**
     * Get this card title.
     */

    public String getTitle() {
	return title ;
    }
     
    /**
     * Dump this card into the given HtmlGenerator.
     * @param into The HtmlGenerator to dump ourself into.
     */

    protected void dump (HtmlGenerator g) {
	// Dump the fields:
	g.append ("<form action=\"" + url + "\" method=\"POST\">") ;
	g.append("<table width=\"100%\">");
	if ( vfields.size() > 0 ) {
	    for (int i = 0 ; i < vfields.size() ; i++) 
		((FormField) vfields.elementAt(i)).dump(g) ;
	}
	g.append("</table>");
	g.append ("<p><input type=\"submit\" name=\"submit\" value=\"Ok\">");
	g.append ("<input type=\"reset\" name=\"submit\" value=\"Reset\">");
	g.append ("</form>") ;
	// Dump the buttons:
	if ( vbuttons.size() > 0 ) {
	    g.append("<hr>");
	    for (int i = 0 ; i < vbuttons.size() ; i++) {
		String label = (String) vbuttons.elementAt(i) ;
		g.append("<form method=\"GET\" action=\""
			 + url+"/"+label
			 +"\">");
		g.append("<input type=\"submit\""
			 + " name=\"BUTTONRESERVEDBUTTON\""
			 + " value=\""+label+"\">");
	    }
	}
    }

    /**
     * Add a new field to this form.
     * @param field The field to add.
     */

    public void addField (FormFieldInterface field) {
	vfields.addElement (field) ;
    }

    /**
     * Add a new button.
     * Button click are simulated, the form handler will get notified of
     * them through the notifyButtonClick callback.
     * @param label The button name.
     */
    
    public void addButton(String label) {
	vbuttons.addElement(label) ;
    }

    /**
     * remove the field whose name is given.
     * @param name The name of the field to remove.
     */

    public void removeField(String name) {
	for (int i = 0 ; i < vfields.size() ; i++) {
	    FormField field = (FormField) vfields.elementAt(i) ;
	    if ( field.getName().equals(name) ) {
		vfields.removeElementAt(i) ;
		return ;
	    }
	}
    }

    /**
     * Reset this form card.
     * Remove all the fields and all the buttons defined for this card.
     */

    public void reset() {
	vfields.setSize(0) ;
	vbuttons.setSize(0) ;
    }

    /**
     * Lookup a field in this card.
     * @param name The name of the field to look for.
     * @return An instance of FormFieldInterface or <strong>null</strong>
     *    if none was found.
     */

    public FormFieldInterface lookupField (String name) {
	for (int i = 0 ; i < vfields.size() ; i++) {
	    FormField field = (FormField) vfields.elementAt(i) ;
	    if ( field.getName().equals(name) )
		return field ;
	}
	return null ;
    }
    
		 
    /**
     * Update the field whose name is given, with the provided value.
     * @param name The field name.
     * @param value The new value for this field.
     * @exception HTTPException If the field couldn't be set.
     */

    protected void updateField (String name, String value) 
	throws FormFieldException 
    {
	FormFieldInterface field = lookupField (name) ;
	if ( field == null )
	    return ;
	if ( field.setValue (value) ) {
	    if ( handler != null )
		handler.notifyChange (field, field.getValue()) ;
	}
    }

    /**
     * Get this form card entity.
     * This dumps the whole form (including the card button controlers).
     */

    public Reply get (Request request)
	throws HTTPException
    {
	// If the card has been deleted (for some reason), delegate to the form
	if ( isDeleted() )
	    return form.get(request) ;
	// Is is time to update our HTML for the card, checkit:
	FormCardResource updated = handler.updateFormCard(this) ;
	if ( updated == null )
	    updated = this ;
	// Emit the HTML:
	Reply reply = request.makeReply(HTTP.OK) ;
	HtmlGenerator g = new HtmlGenerator (title) ;
	form.dump (g, updated) ;
	reply.setStream (g) ;
	return reply ;
    }

    /**
     * Handle a post to this card.
     * This method is synchronized so that only one thread of control executes 
     * the sequence of <code>notifyBeginProcessing</code>, 
     * <code>notifyEndProcessing</code> and <code>notifyChange</code>.
     * @param client The clien tthat issued the request.
     * @param request The request to fullfill.
     * @param data The form data.
     * @exception HTTPException If the form processing erred.
     */

    public synchronized Reply handle (Request request, URLDecoder data) 
	throws HTTPException
    {
	// Run the begin processing handler:
	try {
	    handler.notifyBeginProcessing(this, request) ;
	} catch (FormProcessingException ex) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent (ex.getMessage()) ;
	    throw new HTTPException (error) ;
	}
	// Change the fields value apprpriately:
	Enumeration e = data.keys() ;
	while ( e.hasMoreElements() ) {
	    String name  = (String) e.nextElement() ;
	    String value = data.getValue(name) ;
	    try {
		updateField (name, value) ;
	    } catch (FormFieldException ex) {
		Reply         r = request.makeReply(HTTP.OK) ;
		HtmlGenerator g = new HtmlGenerator ("error for " + name) ;
		g.append ("<p>The field " + name
			  + " has an incorrect value:</p><hr>"
			  + "<p>" + ex.getMessage() + "</p><hr>") ;
		g.append("<p>Click <a href=\""+url+"\">here</a> to continue.");
		r.setStream (g) ;
		return r ;
	    }
	}
	// Run the end processing handler:
	String location = null ;
	try {
	    location = handler.notifyEndProcessing(this, request) ;
	} catch (FormProcessingException ex) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent(ex.getMessage()) ;
	    throw new HTTPException (error) ;
	}
	// If a location provided, redirect else get:
	if ( location == null ) {
	    return get (request) ;
	} else {
	    Reply reloc = request.makeReply(HTTP.MOVED_TEMPORARILY) ;
	    reloc.setContentLength(0) ;
	    reloc.setLocation(location) ;
	    throw new HTTPException(reloc) ;
	}
    }

    /**
     * Create a new form card.
     * Users of the package should create new form cards through the form 
     * object itself.
     * @param form The form we are attached to.
     * @param handler This card form handler.
     * @param name This card's name,
     * @param title The card displayed title.
     */

    FormCardResource (FormResource form
		      , FormCardHandler handler
		      , String name, String title) {
	this.form     = form ;
	this.handler  = handler ;
	this.name     = name ;
	this.title    = title ;
	this.vfields  = new Vector() ;
	this.vbuttons = new Vector() ;
	this.url      = form.getURL() + "/" + name ;
    }



}
