// CgiResource.java
// $Id: CgiResource.java,v 1.2 1996/04/18 14:12:57 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.forms ;

import java.io.* ;
import java.util.*;
import java.net.*;

import w3c.mime.* ;
import w3c.jigsaw.http.* ;
import w3c.jigsaw.resources.* ;

/**
 * A simple process feeder class.
 */

class ProcessFeeder extends Thread {
    Process      proc    = null ;
    OutputStream out     = null ;
    InputStream  in      = null ;
    int          count = -1 ;

    public void run () {
	try {
	    byte buffer[] = new byte[4096] ;
	    int  got      = -1 ;
	    
	    // Send the data to the target process:
	    if ( count >= 0 ) {
		while ( (count > 0) && ((got = in.read(buffer)) > 0) ) {
		    out.write (buffer, 0, got) ;
		    count -= got ;
		}
	    } else {
		while ( (got = in.read(buffer)) > 0 ) {
		    out.write (buffer, 0, got) ;
		}
	    }
	    // Clean up the process:
	    out.flush() ;
	    out.close() ;
	    proc.waitFor() ;
	} catch (Exception e) {
	    System.out.println ("ProcessFeeder: caught exception !") ;
	    e.printStackTrace() ;
	}
    }
	
    ProcessFeeder (Process proc, InputStream in) {
	this (proc, in, -1) ;
    }
	
    ProcessFeeder (Process proc, InputStream in, int count) {
	this.proc   = proc ;
	this.out    = proc.getOutputStream() ;
	this.in     = in ;
	this.count  = count ;
    }
}

/**
 * A class to handle the good old cgi scripts. 
 * This should really not be used. I am writing it for fun only, you should
 * rather implement your own resources if you are to handle forms.
 * @see w3c.jigsaw.resources.Resource
 */

public class CgiResource extends FileResource {
    /**
     * Attribute index - The array of string that makes the command to run.
     */
    protected static int ATTR_COMMAND = -1 ;
    /**
     * Attribute index - Does the script takes care of its headers ?
     */
    protected static int ATTR_NOHEADER = -1 ;
    /**
     * Attribute index - Does the script generates the form on GET ?
     */
    protected static int ATTR_GENERATES_FORM = -1 ;

    static {
	Attribute a   = null ;
	Class     cls = null ;
	try {
	    cls = Class.forName("w3c.jigsaw.forms.CgiResource") ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
	// The command attribute:
	a = new StringArrayAttribute("command"
				     , null
				     , Attribute.MANDATORY|Attribute.EDITABLE);
	ATTR_COMMAND = AttributeRegistery.registerAttribute(cls, a) ;
	// The noheader attribute:
	a = new BooleanAttribute("noheader"
				 , Boolean.FALSE
				 , Attribute.EDITABLE) ;
	ATTR_NOHEADER = AttributeRegistery.registerAttribute(cls, a) ;
	// The generates form attribute
	a = new BooleanAttribute("generates-form"
				 , Boolean.TRUE
				 , Attribute.EDITABLE) ;
	ATTR_GENERATES_FORM = AttributeRegistery.registerAttribute(cls, a);
    }

    /**
     * Get the command string array.
     */

    public String[] getCommand() {
	return (String[]) getValue(ATTR_COMMAND, null) ;
    }

    /**
     * Get the noheader flag.
     */

    public boolean getNoheaderFlag() {
	return getBoolean(ATTR_NOHEADER, false) ;
    }

    /**
     * Get the generates form flag.
     */

    public boolean getGeneratesFormFlag() {
	return getBoolean(ATTR_GENERATES_FORM, true) ;
    }

    /**
     * Handle the CGI script output.
     * This methods handles the CGI script output. Depending on the
     * value of the <strong>noheader</strong> attribute it either:
     * <ul>
     * <li>Sends back the script output directly,</li>
     * <li>Parses the script output, looking for a status header or a 
     * location header, or a content-length header, or any combination
     * of those three.
     * </ul>
     * @param process The underlying CGI process.
     * @param request The processed request.
     * @exception HTTPException If an HTTP error should be sent back to the 
     *    client.
     */

    protected Reply handleCGIOutput (Process process, Request request) 
	throws HTTPException
    {
	if ( getNoheaderFlag() ) {
	    Reply reply = request.makeReply(HTTP.NOHEADER) ;
//	    reply.setKeepConnection (false) ;
//	    reply.unsetContentLength() ;
	    reply.setStream (process.getInputStream()) ;
	    return reply ;
	}

	MIMEParser  p = new MIMEParser (process.getInputStream()) ;
	MIMEHeaders h = null ;
	Reply reply   = null ;

	try {
	    h     = p.getMIMEHeaders() ;
	    if ( h.hasField ("status") )
		reply = request.makeReply(h.getFieldInt ("status"));
	    else
		reply = request.makeReply (HTTP.OK) ;
	    if ( h.hasField ("content-type") )
		reply.setContentType (h.getField ("content-type")) ;
	    if ( h.hasField ("location") ) {
		// Emit a redirect:
		reply.setStatus (HTTP.MOVED_TEMPORARILY) ;
		reply.setLocation (h.getField ("location")) ;
		reply.setContentLength (0) ;
	    } else {
		// Emit the CGI output:
//		reply.setKeepConnection (false) ;
		reply.setStream (p.getInputStream()) ;
	    }
	} catch (MIMEException e) {
	    Reply error = request.makeReply(HTTP.INTERNAL_SERVER_ERROR) ;
	    error.setContent("CGI error: unable to parse script headers.") ;
	    throw new HTTPException (error) ;
	}
	return reply ;
    }

    /**
     * Add an enviornment binding to the given vector.
     * @param name The name of the enviornment variable.
     * @param val Its value.
     * @param into The vector to which accumulate bindings.
     */

    private void addEnv (String name, String val, Vector into) {
	into.addElement (name+"="+val) ;
    }

    /**
     * Prepare the command to run for this CGI script, and run it.
     * @param request The request to handle.
     * @return The running CGI process object.
     * @exception HTTPException If we weren't able to build the command or
     *    the environment.
     */

    protected Process makeCgiCommand (Request request) 
	throws HTTPException, IOException
    {
	// Check the command attribute first:
	String      command[] = getCommand() ;
	if ( command == null ) {
	    Reply error = request.makeReply(HTTP.INTERNAL_SERVER_ERROR) ;
	    error.setContent("CgiResource mis-configured: it doesn't have a "
			     + " command attribute");
	    throw new HTTPException(error);
	}
	// Ok:
	Vector      env       = new Vector(32) ;
	httpd       server    = request.getClient().getServer() ;
	InetAddress sadr      = server.getInetAddress() ;
	// Specified environment variables:
	addEnv ("SERVER_SOFTWARE", server.getSoftware(), env) ;
	addEnv ("SERVER_NAME", "jigsaw", env) ;
	addEnv ("GATEWAY_INTERFACE", "CGI/1.1", env) ;
	addEnv ("SERVER_PROTOCOL", request.getHTTPVersion(), env) ;
	addEnv ("SERVER_PORT"
		, new Integer (server.getLocalPort()).toString()
		, env) ;
	addEnv ("REQUEST_METHOD", request.getMethod(), env) ;
	addEnv ("PATH_INFO", request.getURI(), env) ;	     // FIXME
	addEnv ("PATH_TRANSLATED", command[0], env) ; 
	addEnv ("SCRIPT_NAME", getIdentifier(), env) ;
	if ( request.hasField ("query") )
	    addEnv ("QUERY_STRING", request.getQueryString (), env) ;
//	addEnv ("REMOTE_HOST", , env) ;
	addEnv ("REMOTE_ADDR"
		, request.getClient().getInetAddress().toString(), env) ;
	if ( request.hasField ("authtype") ) {
	    try {
		addEnv ("AUTH_TYPE", request.getFieldString ("authtype"), env);
	    } catch (MIMEException e) {
	    }
	}
//	addEnv ("REMOTE_USER", , env) ;
//	addEnv ("REMOTE_IDENT", , env) ;
	if ( request.hasField ("content-type") )
	    addEnv ("CONTENT_TYPE", request.getContentType().toString(), env) ;
	if ( request.hasField ("content-length") )
	    addEnv ("CONTENT_LENGTH"
		    , new Integer(request.getContentLength()).toString()
		    , env) ;
	// Extra HTTP specific fields (we get them with getField, unparsed):
	if ( request.hasField ("user-agent") )
	    addEnv ("HTTP_USER_AGENT", request.getField ("user-agent"), env);
	if ( request.hasField ("accept") )
	    addEnv ("HTTP_ACCEPT", request.getField ("accept"), env) ;
	// Command line:
	if ( request.hasField ("query") ) {
	    String query      = Request.unescape(request.getQueryString()) ;
	    String querycmd[] = new String[command.length+1] ;
	    System.arraycopy(command, 0, querycmd, 0, command.length) ;
	    querycmd[command.length] = query ;
	    command = querycmd ;
	}
	String aenv[] = new String[env.size()] ;
	env.copyInto (aenv) ;
	// Run the process:
	return Runtime.getRuntime().exec (command, aenv) ;
    }

    /**
     * GET method implementation.
     * this method is splitted into two cases:
     * <p>If the resource is able to generates its form, than run the script
     * to emit the form. Otherwsie, use our super class (FileResource) ability
     * to send the file that contains the form.
     * <p>Note that there is no need to feed the underlying process with
     * data in the GET case.
     * @param request The request to handle.
     * @exception HTTPException If processing the request failed.
     */

    public Reply get (Request request)
	throws HTTPException
    {
	if ( ! getGeneratesFormFlag() )
	    return super.get (request) ;
	Process       process = null ;
	ProcessFeeder feeder  = null ;
	try {
	    process = makeCgiCommand (request) ;
	} catch (IOException e) {
	    Reply error = request.makeReply(HTTP.NOT_FOUND) ;
	    error.setContent("The resource's script wasn't found.") ;
	    throw new HTTPException (error) ;
	}
	return handleCGIOutput (process, request) ;
    }

    /**
     * Handle the POST method according to CGI/1.1 specification.
     * The request body is sent back to the launched CGI script, as is, and
     * the script output is handled by the handleCGIOutput method.
     * @param request The request to process.
     * @exception HTTPException If the processing failed.
     */

    public Reply post (Request request) 
	throws HTTPException 
    {
	Process       process = null ;
	ProcessFeeder feeder  = null ;

	// Launch the CGI process:
	try {
	    process = makeCgiCommand (request) ;
	} catch (IOException e) {
	    Reply error = request.makeReply(HTTP.NOT_FOUND) ;
	    error.setContent("The resource's script wasn't found.");
	    throw new HTTPException (error) ;
	}
	// Send it its form input:
	if ( request.hasField ("content-length") ) {
	    int         clen = request.getContentLength(-1) ;
	    InputStream in   = null ;
	    in = new MIMEContentLengthInputStream (request.getInputStream()
						   , clen);
	    feeder = new ProcessFeeder (process, in, clen) ;
	} else {
	    // no content-length provided, we could try to cope with
	    // this, but I just don't like it.
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent("Invalid request: no content-length.");
	    throw new HTTPException (error) ;
	}
	feeder.start() ;
	return handleCGIOutput (process, request) ;
    }
    
}
