// FormPorpertiesEditor.java
// $Id: FormPropertiesEditor.java,v 1.5 1996/05/02 15:46:26 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.config ;

import java.io.* ;
import java.util.* ;

import w3c.jigsaw.http.* ;
import w3c.jigsaw.resources.* ;
import w3c.jigsaw.forms.* ;
import w3c.jigsaw.html.*;

class PropertiesHandler extends FormCardHandler {
    FormPropertiesEditor editor = null ;

    public void notifyChange (FormFieldInterface field, Object value) 
	throws FormFieldException
    {
	editor.changeProperty(field.getName(), value) ;
    }

    PropertiesHandler(FormPropertiesEditor editor) {
	this.editor = editor ;
    }
}


/**
 * Form based editor for the server properties.
 * This is likely to change in future versions of Jigsaw: right now the
 * set of editable properties is hard-coded, there should be a way of 
 * registering new property sets, but it is not that easy since you may want
 * to edit properties <em>before</em> the resources that uses them are loaded.
 * <p>I will stay with this simple model until I found something more suitable.
 * Next versions will probably uses resources instead of properties.
 */

public class FormPropertiesEditor extends FormResource {
    /**
     * Attribute index - The property help page.
     */
    protected static int ATTR_HELPURL = -1 ;

    static {
	Attribute a   = null ;
	Class     cls = null ;
	try {
	    cls = Class.forName("w3c.jigsaw.config.FormPropertiesEditor") ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
	// The help url attribute
	a = new StringAttribute("helpurl"
				, null
				, Attribute.EDITABLE) ;
	ATTR_HELPURL = AttributeRegistery.registerAttribute(cls, a) ;
    }

    /**
     * Our form card handler (shared among all cards).
     */
    protected PropertiesHandler handler = null ;

    /**
     * The set of properties we are editing.
     */
    httpdProperties props = null ;
    /**
     * Does the server needs restart.
     */
    protected boolean needs_restart = false ;
    /**
     * Does the properties needs savings.
     */
    protected boolean needs_save = false ;

    /**
     * Get the help URL for the given property.
     * @return A help URL, or <strong>null</strong>.
     */

    public String getHelpURL() {
	return getString(ATTR_HELPURL, null);
    }

    public String getHelpURL(String prop) {
	String base = getHelpURL() ;
	return (base == null) ? null : (base + "#" + prop) ;
    }

    /**
     * Does the server needs a restart due to some property changes.
     */

    public boolean needsRestart() {
	return needs_restart;
    }

    /**
     * Does the properties need to be saved.
     */

    public boolean needsSave() {
	return needs_save ;
    }

    /**
     * Change the value of a property.
     * The callback for the form card handler.
     * @param name The name of the property.
     * @param value Its new value.
     */

    public void changeProperty(String name, Object value)
	throws FormFieldException
    {
	boolean restartp = false ;

	// Some properties requires a server restart:
	if ( name.equals(Shuffler.SHUFFLER_P)
	     || name.equals(httpd.HOST_P)
	     || name.equals(httpd.ROOT_P)
	     || name.equals(httpd.SPACE_P)
	     || name.equals(httpd.PORT_P)
	     || name.equals(httpd.LOGGER_P)
	     || name.equals(httpd.SHUFFLER_PATH_P) 
	     || name.equals(httpd.ROOT_STORE_P)
	     || name.equals(httpd.ROOT_NAME_P)) {
	    restartp      = true ;
	    needs_restart = true ;
	} 
	// Try setting the value straight:
	if (!props.putValue(name, (value == null) ? null : value.toString())) {
	    if ( restartp ) {
		String msg = "Your change of " + name + " to " 
		    + ((value!=null) ? value.toString() : "null")
		    + " requires a server restart. You can keep editing "
		    + " properties, but they will take effect only after the "
		    + " server is restarted." ;
		needs_save = true ;
		throw new FormFieldException(msg) ;
	    } else {
		String msg = "Your change of " + name + " to " 
		    + ((value!=null) ? value.toString() : "null")
		    + " was rejected, the value is probably incorrect." ;
		throw new FormFieldException(msg) ;
	    }
	} else {
	    needs_save = true ;
	}
    }

    /**
     * Save edited our properties.
     * @param request The request that triggered the save.
     * @exception HTTPException If saving the properties failed.
     */

    protected Reply saveProperties (Request request)
	throws HTTPException
    {
	File propfile   = new File(getServer().getConfigDirectory()
				   , "httpd.props") ;
	// Did we guessed were the place to save the property file ?
	if ( propfile == null ) {
	    Reply         r = request.makeReply(HTTP.OK) ;
	    HtmlGenerator g = new HtmlGenerator ("Error while saving props") ;
	    g.append ("<h1>Error while saving properties</h1>") ;
	    g.append ("<p>Your current setting deson't provide a root "
		      + " directory for the server which is required "
		      + " in order to save properties."
		      + "<p>Set the <code>Root directory</code> or the "
		      + " <code>Config directory</code> property "
		      + " and try saving the properties again.") ;
	    r.setStream (g) ;
	    return r ;
	}
	// We have a suitable file handle, save properties.
	try {
	    FileOutputStream fout = new FileOutputStream(propfile);
	    props.save (fout, "Jigsaw written") ;
	    fout.close() ;
	} catch (IOException ex) {
	    Reply         r = request.makeReply(HTTP.OK) ;
	    HtmlGenerator g = new HtmlGenerator ("Error while saving props");
	    g.append ("<h1>Error while saving properties</h1>") ;
	    g.append ("<p>The following error occured while saving properties"
		      + " to <code>" + propfile + "</code>:</p><pre>"
		      + ex.getMessage()
		      + "</pre>") ;
	    r.setStream (g) ;
	    return r ;
	}
	needs_save = false ;
	getServer().errlog ("properties " + propfile + " have been saved.");
	return null ;
    }

    /**
     * Run the given command.
     * @param request The request to be processed.
     * @param command The command to run.
     * @exception HTTPException If processing the command failed.
     */

    public Reply runCommand(Request request, String command) 
	throws HTTPException
    {
	System.out.println("Run command ["+command+"]") ;
	if ( command.equals("Save")  ) {
	    // Save the properties
	    return saveProperties(request) ;
	} else if ( command.equals("Restart") ) {
	    // Restart the server, without reloading properties
	    getServer().restart(false) ;
	} else {
	    // Unknwon command
	    System.out.println("Unknown command.") ;
	}
	return null ;
    }

    /**
     * Dump the form header:
     */

    public void dumpHeader(HtmlGenerator into) {
	super.dumpHeader(into) ;
	// Does the server needs restart:
	into.append("<form action=\""+getURL()+"\" method=\"GET\">"
		    + "<table>");
	into.append("<tr><th align=\"right\">Restart server"
		    + (needsRestart() ? " (needed)" : "")
		    + "<th align=\"left\"><input type=\"submit\" "
		    + "name=\"Command\" "
		    + "value=\"Restart\">") ;
	into.append("<tr><th align=\"right\">Save properties"
		    + (needsSave() ? " (needed)" : "")
		    + "<th align=\"left\"><input type=\"submit\" "
		    + "name=\"Command\" "
		    + "value=\"Save\">") ;
	into.append("</form></table><hr>");
    }

    /**
     * We overide GET to handle the additional commands.
     * @param request The request to handle.
     * @param HTTPException If the command isn't understood, or if it fails.
     */

    public Reply get(Request request) 
	throws HTTPException
    {
	if ( request.hasField("query") ) {
	    String query = request.getQueryString() ;
	    if ( ! query.startsWith("Command=") ) {
		Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
		error.setContent("Invalid command: "+query) ;
		throw new HTTPException(error) ;
	    } 
	    String command = query.substring("Command=".length()) ;
	    Reply  reply   = runCommand(request, command) ;
	    return (reply == null) ? super.get(request) : reply ;
	} else {
	    return super.get(request) ;
	}
    }

    /**
     * The general card handles the general settings of the server.
     */

    protected void defineGeneralCard() {
	FormFieldInterface f    = null ;
	FormCardResource   card = defineCard(handler
					     , "General"
					     , "General settings") ;
	// The server root directory:
	f = new FileField(httpd.ROOT_P
			  , "root directory"
			  , getHelpURL(httpd.ROOT_P)
			  , props.getFile(httpd.ROOT_P, null)) ;
	card.addField(f) ;
	// The server space directory:
	f = new FileField(httpd.SPACE_P
			  , "space directory"
			  , getHelpURL(httpd.SPACE_P)
			  , props.getFile(httpd.SPACE_P, null));
	card.addField(f) ;
	// The full host name:
	f = new TextField(httpd.HOST_P
			  , "host name"
			  , getHelpURL(httpd.HOST_P)
			  , props.getString(httpd.HOST_P, null));
	card.addField(f) ;
	// The port number
	f = new IntegerField(httpd.PORT_P
			     , "port number"
			     , getHelpURL(httpd.PORT_P)
			     , new Integer(props.getInteger(httpd.PORT_P
							    , 8001)));
	card.addField(f) ;
	// The root store name:
	f = new FileField(httpd.ROOT_STORE_P
			  , "root store repository"
			  , getHelpURL(httpd.ROOT_STORE_P)
			  , props.getFile(httpd.ROOT_STORE_P, null));
	card.addField(f) ;
	// The root resource's name:
	f = new TextField(httpd.ROOT_NAME_P
			  , "root resource's name"
			  , getHelpURL(httpd.ROOT_NAME_P)
			  , props.getString(httpd.ROOT_NAME_P, null));
	card.addField(f) ;
	// The trace flag
	f = new BooleanField(httpd.TRACE_P
			     , "trace flag"
			     , getHelpURL(httpd.TRACE_P)
			     , new Boolean(props.getBoolean(httpd.TRACE_P
							    , false)));
	card.addField(f) ;
    }

    /**
     * Define the client properties card.
     */

    protected void defineClientCard() {
	FormFieldInterface f    = null ;
	FormCardResource   card = defineCard(handler
					     , "Connections"
					     , "Connections properties") ;
	// The keep alive flag
	f = new BooleanField(httpd.KEEP_ALIVE_P
			     ,"keep-alive"
			     ,getHelpURL(httpd.KEEP_ALIVE_P)
			     ,new Boolean(props.getBoolean(httpd.KEEP_ALIVE_P
							   , true)));
	card.addField(f) ;
	// The client's thread priority
	f = new IntegerField(httpd.CLIENT_PRIORITY_P
			     , "thread priority"
			     , getHelpURL(httpd.CLIENT_PRIORITY_P)
			     , new Integer(props.getInteger(httpd.CLIENT_PRIORITY_P
							    , 5)));
	card.addField(f) ;
	// The client's buffer size
	f = new IntegerField(httpd.CLIENT_BUFSIZE_P
			     , "buffer size"
			     , getHelpURL(httpd.CLIENT_BUFSIZE_P)
			     , new Integer(props.getInteger(httpd.CLIENT_BUFSIZE_P
							    , 8192)));
	card.addField(f) ;
	// The client's debug flag
	f = new BooleanField(httpd.CLIENT_DEBUG_P
			     , "trace flag"
			     , getHelpURL(httpd.CLIENT_DEBUG_P)
			     , new Boolean(props.getBoolean(httpd.CLIENT_DEBUG_P
							    , false)));
	card.addField(f) ;
	// The keep timeout
	f = new IntegerField(httpd.KEEP_TIMEOUT_P
			     , "idle timeout"
			     , getHelpURL(httpd.KEEP_TIMEOUT_P)
			     , new Integer(props.getInteger(httpd.KEEP_TIMEOUT_P
							    , 1200000)));
	card.addField(f) ;
	// The request timeout
	f = new IntegerField(httpd.REQUEST_TIMEOUT_P
			     , "request timeout"
			     , getHelpURL(httpd.REQUEST_TIMEOUT_P)
			     , new Integer(props.getInteger(httpd.REQUEST_TIMEOUT_P
							    , 1200000)));
	card.addField(f) ;
	// Maximum number of allowed clients:
	f = new IntegerField(ClientPool.MAX_CLIENT_P
			     , "max number of clients"
			     , getHelpURL(ClientPool.MAX_CLIENT_P)
			     , new Integer(props.getInteger(ClientPool.MAX_CLIENT_P
							    , 100)));
	card.addField(f) ;
	// Number of clients to keep 
	f = new IntegerField(ClientPool.MIN_CLIENT_P
			     , "spare clients"
			     , getHelpURL(ClientPool.MIN_CLIENT_P)
			     , new Integer(props.getInteger(ClientPool.MIN_CLIENT_P
							    , 32)));
	card.addField(f) ;
    }

    /**
     * The logging card.
     */

    protected void defineLoggingCard() {
	FormFieldInterface f    = null ;
	FormCardResource   card = defineCard(handler
					     , "Logging"
					     , "Logging properties") ;
	// The class of the logger
	f = new ClassnameField(httpd.LOGGER_P
			       , "logger class"
			       , getHelpURL(httpd.LOGGER_P)
			       , props.getString(httpd.LOGGER_P, null)) ;
	card.addField(f) ;
	// The error log:
	f = new FilenameField(CommonLogger.ERRLOGNAME_P
			      , "errors file"
			      , getHelpURL(CommonLogger.ERRLOGNAME_P)
			      , props.getString(CommonLogger.ERRLOGNAME_P
						, "errors"));
	card.addField(f) ;
	// The normal log:
	f = new FilenameField(CommonLogger.LOGNAME_P
			      , "log file"
			      , getHelpURL(CommonLogger.LOGNAME_P)
			      , props.getString(CommonLogger.LOGNAME_P
						, "log"));
	card.addField(f) ;
	// The trace log:
	f = new FilenameField(CommonLogger.TRACELOGNAME_P
			      , "traces file"
			      , getHelpURL(CommonLogger.TRACELOGNAME_P)
			      , props.getString(CommonLogger.TRACELOGNAME_P
						, "traces"));
	card.addField(f) ;
    }
    
    /**
     * The shuffler properties.
     */

    protected void defineShufflerCard() {
	FormFieldInterface f    = null ;
	FormCardResource   card = defineCard(handler
					     , "Shuffler"
					     , "Shuffler properties");
	// The shuffler server
	f = new FileField(Shuffler.SHUFFLER_P
			  , "server's path"
			  , getHelpURL(Shuffler.SHUFFLER_P)
			  , props.getFile(Shuffler.SHUFFLER_P, null));
	card.addField(f) ;
	// The shuffler socket path
	f = new FileField(httpd.SHUFFLER_PATH_P
			  , "socket path"
			  , getHelpURL(httpd.SHUFFLER_PATH_P)
			  , props.getFile(httpd.SHUFFLER_PATH_P, null));
	card.addField(f) ;
    }

    /**
     * Define the form card to edit server properties.
     */

    protected void defineCards() {
	defineGeneralCard() ;
	defineClientCard() ;
	defineLoggingCard() ;
	defineShufflerCard() ;
    }

    /**
     * Initialize the form based properties editor.
     */

    public void initialize(Object values[]) {
	super.initialize(values) ;
	// Get the set of props to edit, and initialize:
	this.props   = getServer().getProperties() ;
	this.handler = new PropertiesHandler(this) ;
	defineCards() ;
    }
	
    public FormPropertiesEditor() {
    }

}
