// RealmsEditor.java
// $Id: RealmsEditor.java,v 1.1 1996/04/10 13:38:21 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.auth;

import java.util.* ;
import java.io.*;

import w3c.tools.sorter.*;
import w3c.mime.*;
import w3c.jigsaw.http.* ;
import w3c.jigsaw.resources.* ;
import w3c.jigsaw.forms.* ;
import w3c.jigsaw.formedit.*;

class UserEditorHandler extends FormCardHandler {
    AuthUser target = null ;

    public void notifyChange(FormFieldInterface field, Object value) 
	throws FormFieldException
    {
	target.setValue(field.getName(), value) ;
    }

    public String notifyEndProcessing(FormCardResource card, Request request)
	throws FormProcessingException
    {
	return "..";
    }

    UserEditorHandler(AuthUser target) {
	this.target = target ;
    }

}

class UserEditor extends FormResource {

    UserEditor(String url, AuthUser target) {
	super(url) ;
	// Define the card for user's attribute edition:
	FormCardResource c = defineCard(new UserEditorHandler(target) 
					, "Attributes"
					, "Attributes of " + target.getName());
	Attribute attrs[] = target.getAttributes() ;
	for (int i = 0 ; i < attrs.length ; i++) {
	    Attribute a = attrs[i] ;
	    if ( ! a.checkFlag(Attribute.EDITABLE) )
		continue ;
	    FormFieldInterface field = FormFieldRegistery.getField(target, a) ;
	    if ( field != null )
		c.addField(field) ;
	}
    }
}

class RealmAddHandler extends FormCardHandler {
    String    url    = null ;
    AuthRealm target = null ;
    
    public void notifyButtonClick(String label) 
	throws FormProcessingException
    {
	target.save() ;
    }

    public String notifyEndProcessing(FormCardResource card, Request request)
	throws FormProcessingException
    {
	// Check input values:
	String name = (String) card.lookupField("name").getValue() ;
	if ( name == null )
	    throw new FormProcessingException("Fill in the user's name.");
	// Create the user:
	AuthUser user = AuthUser.makeUser(name) ;
	target.registerUser(user) ;
	// Relocate to its editor:
	return url+"/"+name ;
    }

    RealmAddHandler (String url, AuthRealm target) {
	this.url    = url ;
	this.target = target ;
    }

}

class RealmListHandler extends FormCardHandler {
    String    url    = null ;
    AuthRealm target = null ;
    Vector    marked = null ;

    /**
     * Update the form card.
     */

    public FormCardResource updateFormCard(FormCardResource card) {
	// Reset the whole card (What a waste of time :-(
	card.reset() ;
	// Rebuild the fields that are now available
	Enumeration enum  = target.enumerateUserNames() ;
	Vector      users = Sorter.sortStringEnumeration(enum) ;
	for (int i = 0 ; i < users.size() ; i++) {
	    String name = (String) users.elementAt(i) ;
	    card.addField(new CheckBoxField(name, name, url+"/"+name, false));
	}
	return card ;
    }
    
    /**
     * On notify change, mark the user for deletion.
     */

    public void notifyChange(FormFieldInterface field, Object value) 
	throws FormFieldException
    {
	if ( marked == null )
	    marked = new Vector() ;
	marked.addElement(field.getName()) ;
    }

    /**
     * On notifyEndProcessing, delete all marked users
     */

    public String notifyEndProcessing(FormCardResource card, Request request)
	throws FormProcessingException
    {
	if ( marked == null )
	    return null ;
	// Remove marked users
	int vsize = marked.size() ;
	for (int i = 0 ; i < vsize ; i++) {
	    String id = (String) marked.elementAt(i) ;
	    target.unregisterUser(id) ;
	    card.removeField(id) ;
	}
	marked.setSize(0) ;
	return null ;
    }
    
    RealmListHandler(String url, AuthRealm target) {
	this.url    = url ;
	this.target = target ;
    }

}

class RealmEditor extends FormResource {
    AuthRealm target = null ;
    Hashtable loaded = null ;

    /**
     * Define the form cards for the editor.
     */

    protected void defineFormCards() {
	FormFieldInterface field = null ;
	String             url   = getURL() ;

	// The card to remove users:
	FormCardHandler list_handler = new RealmListHandler(url, target) ;
	FormCardResource c = defineCard(list_handler, "Users", "Edit users");
	list_handler.updateFormCard(c) ;
	// The card to add user (is the same as the card 
	FormCardHandler add_handler = new RealmAddHandler(url, target);
	c = defineCard(add_handler, "AddUser", "Add users");
	// The name of the user to create:
	field = new TextField("name", "name", null, null) ;
	c.addField(field) ;
	// The button to save the realm:
	c.addButton("Save") ;
    }

    /**
     * Lookup in the realm for an editor targeted to some given user.
     * @param state The current lookup state.
     * @exception HTTPException If the user wasn't found.
     */

    public HTTPResource lookup(LookupState state) 
	throws HTTPException
    {
	// Look for an already existing editor
	String     name   = state.peekNextComponent() ;
	UserEditor editor = (UserEditor) loaded.get(name) ;
	if ( editor != null ) {
	    state.getNextComponent() ;
	    return editor ;
	}
	// Install a new one (if user's exist)
	AuthUser user = target.loadUser(name);
	if ( user == null )
	    return super.lookup(state) ;
	state.getNextComponent() ;
	editor = new UserEditor(getURL()+"/"+name, user) ;
	loaded.put(name, editor) ;
	return editor ;
    }

    RealmEditor(String url, AuthRealm target) {
	super(url) ;
	this.target = target ;
	this.loaded = new Hashtable() ;
	defineFormCards() ;
    }
}

class RealmsAddHandler extends FormCardHandler {
    RealmsCatalog catalog = null ;
    String        url     = null ;

    /**
     * The only button is the save button.
     */

    public void notifyButtonClick(String label) 
	throws FormProcessingException
    {
	catalog.save() ;
    }

    /**
     * Given a realm name and a realm repository, create a new realm.
     */

    public String notifyEndProcessing(FormCardResource card, Request request)
	throws FormProcessingException
    {
	// Get and check input values:
	String name = (String) card.lookupField("name").getValue() ;
	File   file = (File) card.lookupField("repository").getValue() ;
	if ((name == null) || (file == null)) {
	    String msg = "Fill in both the realm name (mandatory), and the "
		+ " repository (were the realm's database should be saved).";
	    throw new FormProcessingException(msg) ;
	}
	// Create the realm :
	AuthRealm realm = AuthRealm.makeRealm(name, file) ;
	catalog.registerRealm(realm) ;
	// Relocate to the realm editor.
	return url+"/"+name;
    }

    RealmsAddHandler (String url, RealmsCatalog catalog) {
	this.url = url ;
	this.catalog = catalog ;
    }

}

class RealmsListHandler extends FormCardHandler {
    RealmsCatalog catalog = null ;
    String        url     = null ;
    Vector        marked  = null ;

    /**
     * Update the catalog list of realms.
     */

    public FormCardResource updateFormCard(FormCardResource card) {
	// FIXME (check for last-modified)
	card.reset() ;
	// Enumerate the available realms:
	Enumeration enum   = catalog.enumerateRealmNames() ;
	Vector      realms = Sorter.sortStringEnumeration(enum) ;
	for (int i = 0 ; i < realms.size() ; i++) {
	    String name = (String) realms.elementAt(i) ;
	    card.addField(new CheckBoxField(name, name, url+"/"+name, false)) ;
	}
	return card ;
    }

    /**
     * On notifyChange, mark the appropriate realm.
     */

    public void notifyChange(FormFieldInterface field, Request request)
	throws FormFieldException
    {
	if ( marked == null ) 
	    marked = new Vector() ;
	marked.addElement(field.getName()) ;
    }

    /**
     * On notifyEndProcessing, delete all marked realms.
     * @param card The card wose processing is to be finished.
     * @param request The request that trigered the processing.
     * @exception FormProcessingException If the processing failed.
     */

    public String notifyEndProcessing(FormCardResource card, Request request) 
	throws FormProcessingException
    {
	// No realms marked ?
	if ( marked == null )
	    return null ;
	// Remove all marked realms.
	int vsize = marked.size() ;
	for (int i = 0 ; i < vsize ; i++) {
	    String id = (String) marked.elementAt(i);
	    catalog.unregisterRealm(id) ;
	    card.removeField(id) ;
	}
	marked.setSize(0) ;
	return null ;
    }

    RealmsListHandler (String url, RealmsCatalog catalog) {
	this.url     = url ;
	this.catalog = catalog ;
    }
}

public class RealmsEditor extends FormResource {
    Hashtable     loaded  = null ;
    RealmsCatalog catalog = null ;

    /**
     * Define the forms card.
     */

    protected void defineFormCards() {
	FormFieldInterface field = null ;
	String             url   = getURL() ;
	
	// The card to delete realms:
	FormCardHandler list_handler = new RealmsListHandler(url, catalog);
	FormCardResource c = defineCard(list_handler
					, "Realms"
					, "Edit authentication realms");
	list_handler.updateFormCard(c) ;
	// The card to define new realms:
	FormCardHandler add_handler = new RealmsAddHandler(url, catalog);
	c = defineCard(add_handler, "AddRealm", "Define a new realm.");
	// The name of the realm.
	field = new TextField("name", "name", null, null) ;
	c.addField(field) ;
	// The repository for the realm
	field = new FileField("repository", "repository", null, null) ;
	c.addField(field);
	// The save button:
	c.addButton("Save") ;
    }

    /**
     * Lookup in the realms editor for a defined realm.
     * @param state The current lookup state.
     * @return A Resource to edit the realm.
     */

    public HTTPResource lookup(LookupState state) 
	throws HTTPException
    {
	String      name   = state.peekNextComponent() ;
	RealmEditor editor = (RealmEditor) loaded.get(name) ;
	// Is an editor available for the realm ?
	if ( editor != null ) {
	    state.getNextComponent() ;
	    return editor ;
	}
	// Install a new editor (if the realm exists) 
	AuthRealm realm = catalog.loadRealm(name) ;
	if ( realm == null )
	    return super.lookup(state) ;
	state.getNextComponent() ;
	editor = new RealmEditor(getURL()+"/"+name, realm) ;
	loaded.put(name, editor) ;
	return editor ;
    }

    /**
     * Initialize the realm editor
     */

    public void initialize(Object values[]) {
	super.initialize(values) ;
	this.catalog = getServer().getRealmsCatalog() ;
	this.loaded  = new Hashtable() ;
	defineFormCards() ;
    }

}
