/**
 * Author: Ted Guild <ted@w3.org> 
 * (c) COPYRIGHT W3C http://www.w3.org/Consortium/Legal/copyright-software
 * $Id: BaseAuthURIResolver.java,v 1.4 2007/08/01 14:39:55 ted Exp $
 */

package org.w3c.app.xsl;

import net.sf.saxon.functions.URIQueryParameters;
import net.sf.saxon.trans.DynamicError;
import net.sf.saxon.trans.XPathException;
import net.sf.saxon.event.Stripper;
import net.sf.saxon.event.IDFilter;
import net.sf.saxon.om.AllElementStripper;
import net.sf.saxon.AugmentedSource;
import net.sf.saxon.Configuration;
import net.sf.saxon.functions.EscapeURI;
import net.sf.saxon.functions.ResolveURI;
import net.sf.saxon.Platform;
import net.sf.saxon.value.Whitespace;
import org.xml.sax.InputSource;
import org.xml.sax.XMLReader;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.sax.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;
import javax.xml.transform.*;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.w3c.www.mime.MimeType;
import org.w3c.www.mime.MimeTypeFormatException;

public class BaseAuthURIResolver extends net.sf.saxon.StandardURIResolver {
    private ServletAuthReader sar;
    private Configuration config;

    public BaseAuthURIResolver() {

    }

    public BaseAuthURIResolver(Configuration config) {
        this.config = config;
    }

    public BaseAuthURIResolver(Configuration config, ServletAuthReader sar) {
	this.sar=sar;
        this.config = config;
    }

    protected Platform getPlatform() {
        return Configuration.getPlatform();
    }


    public Source resolve(String href, String base)
    throws XPathException {

        // System.err.println("StandardURIResolver, href=" + href + ", base=" + base);

        Platform platform = getPlatform();

        String relativeURI = href;
        String id = null;

        // Extract any fragment identifier. Note, this code is no longer used to
        // resolve fragment identifiers in URI references passed to the document()
        // function: the code of the document() function handles these itself.

        int hash = href.indexOf('#');
        if (hash>=0) {
            relativeURI = href.substring(0, hash);
            id = href.substring(hash+1);
            // System.err.println("StandardURIResolver, href=" + href + ", id=" + id);
        }

        URIQueryParameters params = null;
        URI uri;
        URI relative;
        try {
            relativeURI = ResolveURI.escapeSpaces(relativeURI);
            relative = new URI(relativeURI);
        } catch (URISyntaxException err) {
            throw new DynamicError("Invalid relative URI " + relativeURI, err);
        }

        String query = relative.getQuery();
        if (query != null && recognizeQueryParameters) {
            params = new URIQueryParameters(query, config);
            int q = relativeURI.indexOf('?');
            relativeURI = relativeURI.substring(0, q);
        }

        Source source = null;

	try {
	    uri = platform.makeAbsolute(relativeURI, base);
	} catch (URISyntaxException err) {
	    // System.err.println("Recovering from " + err);
	    // last resort: if the base URI is null, or is itself a relative URI, we
	    // try to expand it relative to the current working directory
	    String expandedBase = ResolveURI.tryToExpand(base);
	    if (!expandedBase.equals(base)) { // prevent infinite recursion
		return resolve(href, expandedBase);
	    }
	    //err.printStackTrace();
	    throw new DynamicError("Invalid URI " + relativeURI + " - base " + base, err);
	}
	
	// Check that any "%" sign in the URI is part of a well-formed percent-encoded UTF-8 character.
	// Without this check, dereferencing the resulting URL can fail with arbitrary unchecked exceptions
	
	final String uriString = uri.toString();
	EscapeURI.checkPercentEncoding(uriString);

	source = new StreamSource(uri.toString());
	try {
	    ((StreamSource)source).setReader(sar.initAuthInputStreamReader(uri.toString()));
	}
	catch (Exception err) {
	    throw new DynamicError("Auth Reader " + uri.toString(), err);
	}

	source.setSystemId(uri.toString());
	
	if (params != null) {
	    XMLReader parser = params.getXMLReader();
	    if (parser != null) {
		((SAXSource)source).setXMLReader(parser);
	    }
	}
	
        if (params != null) {
            int stripSpace = params.getStripSpace();
            switch (stripSpace) {
                case Whitespace.ALL: {
                    Stripper stripper = AllElementStripper.getInstance();
                    stripper.setStripAll();
                    source = AugmentedSource.makeAugmentedSource(source);
                    ((AugmentedSource)source).addFilter(stripper);
                    break;
                }
                case Whitespace.IGNORABLE:
                case Whitespace.NONE:
                    source = AugmentedSource.makeAugmentedSource(source);
                    ((AugmentedSource)source).setStripSpace(stripSpace);
            }
        }

        if (id != null) {
            IDFilter filter = new IDFilter(id);
            source = AugmentedSource.makeAugmentedSource(source);
            ((AugmentedSource)source).addFilter(filter);
        }

        if (params != null) {
            Integer validation = params.getValidationMode();
            if (validation != null) {
                source = AugmentedSource.makeAugmentedSource(source);
                ((AugmentedSource)source).setSchemaValidationMode(validation.intValue());
            }
        }

        if (params != null) {
            Boolean xinclude = params.getXInclude();
            if (xinclude != null) {
                source = AugmentedSource.makeAugmentedSource(source);
                ((AugmentedSource)source).setXIncludeAware(xinclude.booleanValue());
            }
        }

        return source;
    }

}
