#!/usr/bin/env python
#
# SKOS Thesaurus Web Client Demo
#
# $Id: get.py,v 1.17 2004/09/08 09:54:47 cmdjb Exp $
#
# Copyright 2004 Dave Beckett, ILRT, University of Bristol
# SWAD-Europe Thesaurus Activity
# http://www.w3.org/2001/sw/Europe/reports/thes/
# See LICENSE for license information.
#


# Python standard libraries
import sys
import cgi
import tempfile

# HTML templating engine - htmltmpl 1.22 at http://htmltmpl.sourceforge.net/
from htmltmpl import TemplateManager, TemplateProcessor

# Types derived from WSDL - ZSI 1.5.0 at http://pywebsvcs.sourceforge.net/
from SKOSThesaurusService_services import *


portAddress = 'http://thes.ilrt.bris.ac.uk/SKOSThesaurusService'
title = "SKOS Thesaurus Web Service Demonstrations"


class Concept:
  __keys__=("URI", "Definition", "Example", "External ID", "In Scheme",
            "Preferred Label", "Non-Preferred Labels", "Scope Note")
  
  def __init__(self, concept):
    self.content= {
      "URI" : concept._uri,
      "Definition" : concept._definition,
      "Example" : concept._example,
      "External ID" : concept._externalID,
      "In Scheme" : concept._inScheme,
      "Preferred Label" : concept._preferredLabel,
      "Non-Preferred Labels" : concept._nonPreferredLabels,
      "Scope Note" : concept._scopeNote
    }
    
  def __str__(self):
    """Get a string representation of a Concept."""
    report = '<dl>\n'
    for key in Concept.__keys__:
      value = self.content[key]
      if not value:
        continue
          
      if isinstance(value, ns1.URI_Def):
        value = value._uri
        value = """<a href="%s">%s</a>""" % (value,value)
        
      if isinstance(value, list):
        value = "<ul>\n<li>" + "</li>\n<li>".join(value) + "</li>\n</ul>\n"
          
      report += "<dt>%s<br /></dt>\n<dd>%s</dd>\n\n" %(key,value)
                  
      report += '</dl>\n'
                  
    return report.encode("utf-8")


class ConceptSequence:
  def __init__(self, concepts):
    self.content=[]
    for concept in concepts:
      self.content.append(Concept(concept))
                  
  def __str__(self):
    """Get a string representation of a ConceptSequence."""
    report = '<p>Concept list</p>\n<ol>\n'
    for concept in self.content:
      report += "<li>Concept<br />"+str(concept)+"</li>\n"
    report += '</ol>\n'

    return report


class Relation:
  __keys__=("URI", "Label")

  def __init__(self, relation):
    self.content= { "URI" : relation._uri, "Label" : relation._label }
    
  def __str__(self):
    """Get a string representation of a Relation."""
    report = '<dl>\n'
    for key in Relation.__keys__:
      value = self.content[key]
      if not value:
        continue
          
      if isinstance(value, ns1.URI_Def):
        value = value._uri
        value = """<a href="%s">%s</a>""" % (value,value)
        
      if isinstance(value, list):
        value = "<ul>\n<li>" + "</li>\n<li>".join(value) + "</li>\n</ul>\n"
          
      report += "<dt>%s<br /></dt>\n<dd>%s</dd>\n\n" %(key,value)
                  
      report += '</dl>\n'
                  
    return report.encode("utf-8")


class RelationSequence:
  def __init__(self, relations):
    self.content=[]
    for relation in relations:
      self.content.append(Relation(relation))
                  
  def __str__(self):
    """Get a string representation of a RelationSequence."""
    report = '<p>Relation list</p>\n<ol>\n'
    for relation in self.content:
      report += "<li>Relation<br />"+str(relation)+"</li>\n"
    report += '</ol>\n'

    return report


class SKOSService:
  def __init__(self, tracefile = None):
    # portAddress defaults to the WSDL-defined default of
    #   http://thes.ilrt.bris.ac.uk/SKOSThesaurusService
    # the alternative is to use:
    #  loc = SKOSThesaurusServiceLocator(portAddress=..., host=...,
    #                                    port=..., url=...)
    loc = SKOSThesaurusServiceLocator()

    # prints messages sent and received if tracefile is set
    if tracefile is not None:
      kw = { 'tracefile' : tracefile }
    else:
      kw = {}
      
    self.portType = loc.getSKOSThesaurus(**kw)
    self.response = None


  def getConcept(self, uri_string):
    if not uri_string:
      raise TypeError, "No concept URI given"
    
    uri = ns1.URI_Def()
    uri._uri = uri_string

    request = getConceptRequestWrapper()
    request._in0 = uri

    self.response = self.portType.getConcept(request)

    return Concept(self.response._getConceptReturn)


  def getConceptByExternalID(self, id):
    if not id:
      raise TypeError, "No concept ID given"
    
    request = getConceptByExternalIDRequestWrapper()
    request._in0 = id

    self.response = self.portType.getConceptByExternalID(request)

    return Concept(self.response._getConceptByExternalIDReturn)


  def getAllConceptRelatives(self, uri_string):
    if not uri_string:
      raise TypeError, "No concept URI given"

    uri = ns1.URI_Def()
    uri._uri = uri_string

    # First request - get the concept for the URI
    request1 = getConceptRequestWrapper()
    request1._in0 = uri
    response1 = self.portType.getConcept(request1)
    concept = response1._getConceptReturn
    
    concept = ns1.Concept_Def()
    #concept._uri = uri

    # Second request - get the relatives of the concept
    request2 = getAllConceptRelativesRequestWrapper()
    request2._in0 = concept

    self.response = self.portType.getAllConceptRelatives(request2)
    x=self.response._getAllConceptRelativesReturn
    print "response is",x
    return ConceptSequence(x)


  def getTopmostConcepts(self, uri_string):
    if not uri_string:
      raise TypeError, "No thesaurus URI given"

    uri=ns1.URI_Def()
    uri._uri=uri_string

    request=getTopmostConceptsRequestWrapper()
    request._in0 = uri

    self.response = self.portType.getTopmostConcepts(request)

    return ConceptSequence(self.response._getTopmostConceptsReturn)

  def getConceptByPreferredLabel(self, string):
    if not string:
      raise TypeError, "No label string given"
    
    request = getConceptByPreferredLabelRequestWrapper()
    request._in0 = string

    self.response = self.portType.getConceptByPreferredLabel(request)

    return Concept(self.response._getConceptByPreferredLabelReturn)


  def getConceptsMatchingKeyword(self, string):
    if not string:
      raise TypeError, "No string given"

    request=getConceptsMatchingKeywordRequestWrapper()
    request._in0 = string

    self.response = self.portType.getConceptsMatchingKeyword(request)

    return ConceptSequence(self.response._getConceptsMatchingKeywordReturn)


  def getSupportedSemanticRelations(self):
    request=getSupportedSemanticRelationsRequestWrapper()

    self.response = self.portType.getSupportedSemanticRelations(request)

    return RelationSequence(self.response._getSupportedSemanticRelationsReturn)


# The values of all fields in the form
fieldStorage = cgi.FieldStorage()

# Turn cgi fieldStorage into a regular dict
params = {}
for key in fieldStorage.keys():
  params[key] = fieldStorage[key].value


# Argument processing

method = ''

# Form parameters
uri = ''
thesaurus = ''
string = ''
id = ''

results = ''
error = ''
trace = ''

if params.has_key('method'):
  method=params['method']

  tracefile = tempfile.TemporaryFile(dir='/tmp')
  service = SKOSService(tracefile)
  
  if method == 'getconcept' and params.has_key('uri'):
    uri=params['uri']
    try:
      concept = service.getConcept(uri)
      results += str(concept)
    except Exception, e:
      error = str(e)

  if method == 'getconceptbyexternalid' and params.has_key('id'):
    id=params['id']
    try:
      concept = service.getConceptByExternalID(id)
      results += str(concept)
    except Exception, e:
      error = str(e)

  if method == 'getallconceptrelatives' and params.has_key('uri'):
    uri=params['uri']
    try:
      concepts = service.getAllConceptRelatives(uri)
      results += str(concepts)
    except Exception, e:
      error = str(e)

  if method == 'gettopmostconcepts' and params.has_key('thesaurus'):
    thesaurus=params['thesaurus']
    try:
      concepts = service.getTopmostConcepts(thesaurus)
      results += str(concepts)
    except Exception, e:
      error = str(e)

  if method == 'getconceptbypreferredlabel' and params.has_key('string'):
    string=params['string']
    try:
      concepts = service.getConceptByPreferredLabel(string)
      results += str(concepts)
    except Exception, e:
      error = str(e)

  if method == 'getconceptsmatchingkeyword' and params.has_key('string'):
    string=params['string']
    try:
      concepts = service.getConceptsMatchingKeyword(string)
      results += str(concepts)
    except Exception, e:
      error = str(e)

  if method == 'getsupportedsemanticrelations':
    try:
      concepts = service.getSupportedSemanticRelations()
      results += str(concepts)
    except Exception, e:
      error = str(e)

  tracefile.seek(0)
  trace = tracefile.read()


# Compile or load already precompiled template.
template = TemplateManager().prepare("get.tmpl")
tproc = TemplateProcessor()

tproc.set("title", title)
tproc.set("header", title)

# form parameters
tproc.set("uri", uri)
tproc.set("thesaurus", thesaurus)
tproc.set("string", string)

tproc.set("results", results)
tproc.set("error", error)
tproc.set("trace", trace)

print "Content-Type: text/html\r\n\r"

print tproc.process(template)
