// DateField.java
// $Id: DateField.java,v 1.1 1996/04/10 13:41:33 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.forms ;

import java.util.Date ;

import w3c.jigsaw.html.HtmlGenerator ;

/**
 * The date field.
 * Returns a Long value, not a Date object, since this is how date should be 
 * manipulated.
 * The returned long object will give the current time in milliseconds since
 * the epoch (00:00:00 UTC, January 1, 1970).
 * @see java.lang.Date
 */

public class DateField extends FormField {
    Long   value  = null ;
    String strval = null ;

    /**
     * Get this field value.
     */

    public Object getValue() {
	return value ;
    }

    /**
     * Set the date value, without notifying the form handler.
     * @param d The new date value, as a long.
     */

    public void setValue (long d) {
	value = new Long(d) ;
    }

    /**
     * Set the date value, without notifying the form handler.
     * @param d The new date value, as a Long.
     */

    public void setValue (Long d) {
	value = d ;
    }

    /**
     * Set this date field value to the given String value.
     * If the value is changed, than the handler for this field is notified
     * of the change through an appropriate method call.
     * @param nval The new string value for this field.
     * @return A boolean, <strong>true</strong> indicates that the field value
     *    has effectively changed.
     * @exception FormFieldException If the provided value didn't match
     *    the expected type for this field.
     */

    public boolean setValue (String nval) 
	throws FormFieldException
    {
	// Unset the date ?
	if ((nval == null) || nval.equals("")) {
	    boolean result = (value != null) ;
	    value = null;
	    return result ;
	}
	// Is this a new value ?
	if ((value != null) && nval.equals(strval))
	    return false ;
	// Parse the new value, and update state:
	Date dval = null ;
	try {
	    dval = new Date(nval) ;
	} catch (IllegalArgumentException ex) {
	    throw new FormFieldException ("invalid date format in " + nval);
	}
	if ( dval.equals (value) )
	    return false ;
	value  = new Long(dval.getTime()) ;
	strval = dval.toString() ;
	return true ;
    }

    /**
     * Dump this field into the HTML form.
     * @param into The HTML generator to dump to.
     */

    public void dump (HtmlGenerator into) {
	dumpTitle(into) ;
	into.append ("<th align=left><input type=\"text\""
		     + " name=\""+name+"\""
		     + " size=\"32\""
		     + " value=\""
		     + strval
		     + "\"></th>") ;
    }

    /*
     * Initialize a DateField.
     * @param name The name of the field.
     * @param title Its title.
     * @param url Its url (or <strong>null</strong>).
     * @param value Its default value.
     * @exception FormFieldException If the provided value doesn't match 
     *    the expected type.
     */
    
    public void initialize(String name, String title, String url, Object val) 
	throws FormFieldException
    {
	super.initialize(name, title, url, null) ;
	try {
	    this.value = (Long) val ;
	} catch (ClassCastException ex) {
	    String msg = "Invalid type: "+val.getClass().getName() ;
	    throw new FormFieldException (msg) ;
	}
	strval = (value == null) ?"":(new Date(value.longValue())).toString();
    }

    public DateField (String name, String title, String url, Long dval) {
	super (name, title, url) ;
	value  = dval ;
	strval = (dval == null) ? "" : (new Date(dval.longValue())).toString();
    }

    public DateField() {
    }

}
