//*****************************************************************************
/*
** FILE:   xs_ChildrenElement.java
**
** (c) 1998 Steve Withall.
**
** HISTORY:
**    19Jun98  stevew  Created.
*/
package xs;

import xg.xg_Element;
import xg.xg_Node;

import eh.eh_Debug;

import java.io.IOException;
import java.io.Writer;

//*****************************************************************************
/** An XML element representing a 'children' action within an XSL rule.
*/
public class xs_ChildrenElement extends xs_ActionElement
{
    /** The element type name normally used in XML for this sort of element. */
    public final static String RegisteredName = "children";

    //*****************************************************************************
    /** Construct a children element with no name.
     */
    public xs_ChildrenElement()
    {
    }

    //*****************************************************************************
    /** Construct a children element with a name.
     *
     *  @param  InputNodeName  The name of the element
     */
    public xs_ChildrenElement(String InputNodeName)
    {
        super(InputNodeName);
    }

    //*****************************************************************************
    /** Invoke this action to the InputSourceElement, and write the results to the
     *  InputResultsWriter.
     *
     *  @param      InputSourceElement   The pre-parsed source element currently being
     *                                    processed
     *  @param      InputResultsWriter   The writer to which to write the results of
     *                                    applying a stylesheet to the source element
     *  @exception  xs_StyleException    Error applying style
     *  @exception  IOException          Error writing to results writer
     */
    public void invoke(xg_Element  InputSourceElement,
                       Writer      InputResultsWriter)
                          throws xs_StyleException, IOException
    {
        eh_Debug.add(8, "xs_ChildrenElement.invoke: apply style to children of source "
                             + InputSourceElement);

        xg_Node  CurrentChild = null;
        for (int ChildIndex = 0; ChildIndex < InputSourceElement.getChildCount(); ChildIndex++)
        {
            CurrentChild = InputSourceElement.getChild(ChildIndex);
            invokeChild(CurrentChild, InputResultsWriter);
        }
    }

    //*****************************************************************************
    /** Apply style to the InputChildNode, which is a child of the source element
     *  currently being processed. The way in which this is done depends upon the
     *  type of node: if it is an xg_Element, we create an xs_Stylist specially to
     *  apply style to it; otherwise we just write out the InputChildNode's content
     *  (eg. text or CDATA).
     *
     *  @param      InputChildNode      A child of the source element to which style
     *                                   is currently being applied
     *  @param      InputResultsWriter  The writer to which to write the results of
     *                                   applying a stylesheet to the source element
     *  @exception  xs_StyleException   Error applying style
     *  @exception  IOException         Error writing to results writer
     */
    protected void invokeChild(xg_Node  InputChildNode,
                               Writer   InputResultsWriter)
                                           throws xs_StyleException, IOException
    {
        eh_Debug.add(8, "xs_ChildrenElement.invokeChild: apply style to child "
                               + InputChildNode);

        if (InputChildNode instanceof xg_Element)
        {
            // Apply style to this child.
            xs_Stylist  ChildStylist = new xs_Stylist(TheStylesheetDefn);
            ChildStylist.applyStyle((xg_Element)InputChildNode,
                                    InputResultsWriter,
                                    StyleProgressBar);
        }
        else
            // It must be text or CDATA - or something (such as a comment) that
            // writes no content.
            InputChildNode.writeContent(InputResultsWriter);
    }
}

//*****************************************************************************
