//*****************************************************************************
/*
** FILE:   xm_XmlEngineFactory.java
**
** (c) 1998 Steve Withall.
**
** HISTORY:
**    05Jul98  stevew  Created, based on xm_NodeFactory.
*/
package xm;

import xg.xg_AttList;

import eh.eh_Debug;

import java.lang.Class;

//*****************************************************************************
/** Factory for generating XML engines (ie. xm_XmlEngine objects).
 */
public class xm_XmlEngineFactory extends xm_Factory
{
    /** The name of the attribute in an attribute list whose value is the name
     *  of an XML engine's class. */
    public final static String EngineAttName = "Engine";

    //*****************************************************************************
    /** Create an XML engine of a type defined by the InputEngineAttributes. These
     *  must include an 'Engine' attribute which is the name of an XML engine
     *  which derives from xm.xm_XmlEngine. All the InputEngineAttributes are
     *  passed to the newly-created engine, as configuration parameters. Different
     *  types of engine may use different parameters as they see fit.
     *
     *  @param      InputEngineAttributes  The class name of the engine we need to
     *                                     create
     *  @return                            The newly-created engine
     *  @exception  xm_ParseException      The InputEngineAttributes do not include
     *                                      an 'Engine' attribute, or this attribute's
     *                                      value is not the name of a valid class
     *                                      which derives from xm.xm_XmlEngine
     */
    static public xm_XmlEngine createXmlEngine(xg_AttList  InputEngineAttributes)
                                                       throws xm_ParseException
    {
        // Find name of engine class from 'Engine' attribute in attribute list.
        String  EngineClassName = InputEngineAttributes.getAttributeValue(EngineAttName);
        if (EngineClassName == null)
            throw new xm_ParseException("Engine definition does not contain an '"
                                           + EngineAttName + " attribute");

        // Create an instance of this engine class.
        xm_XmlEngine  NewEngine = createXmlEngine(EngineClassName);

        // Pass the new engine its attribute list.
        NewEngine.setAttList(InputEngineAttributes);

        return NewEngine;
    }

    //*****************************************************************************
    /** Create an XML engine as an instance of the class whose name is
     *  InputEngineClassName. This class must derive from xm.xm_XmlEngine.
     *
     *  @param      InputEngineClassName  The class name of the engine we need to
     *                                     create
     *  @return                           The newly-created engine
     *  @exception  xm_ParseException     The given class could not be found, or
     *                                     is not derived from xm.xm_XmlEngine
     */
    static public xm_XmlEngine createXmlEngine(String  InputEngineClassName)
                                                          throws xm_ParseException
    {
        eh_Debug.add(5, "xm_XmlEngineFactory: Create engine of type '"
                                                 + InputEngineClassName + "'");

        Class  EngineClass = null;
        try
        {
            EngineClass = Class.forName(InputEngineClassName);
        }
        catch (Exception  InputException)
        {
            throw new xm_ParseException("Engine class " + InputEngineClassName
                                           + " not found (" + InputException + ")");
        }

        checkClassType(EngineClass, InputEngineClassName, "xm.xm_XmlEngine");

        // Create an engine.
        xm_XmlEngine  NewEngine = null;
        try
        {
            NewEngine = (xm_XmlEngine)EngineClass.newInstance();
        }
        catch (Exception  InputException)
        {
            throw new xm_ParseException("Error instantiating " + InputEngineClassName
                                           + " (" + InputException + ")");
        }

        return NewEngine;
    }
}

//*****************************************************************************
