//*****************************************************************************
/*
** FILE:   xm_ParseException.java
**
** (c) 1997, 1998 Steve Withall.
**
** HISTORY:
**    13Oct97  stevew  Created.
**    03Jul98  stevew  Moved from xe to xm.
*/
package xm;

import java.io.IOException;

//*****************************************************************************
/** Exception when an error occurs during parsing which is sufficiently serious
 *  that the parse cannot continue.
 *
 *  This exception is thrown when a terminal error occurs during parsing. This
 *  allows a trap to be set at the top level of parsing without intervening
 *  parsers having to bother with it.
 */
public class xm_ParseException extends IOException
{
    /** The line in the source we were parsing when the exception occurred. */
    private int  LineNum = 0;

    /** The column in the source line we were parsing when the exception occurred. */
    private int  ColumnNum = 0;

    /** The position in the source at which the last significant item occurred
     *  (which is typically the start of the node currently being parsed) when the
      * exception occurred. */
    private int  LastSignificantPosition = 0;

    /** The position in the source (number of characters) when the exception
     *  occurred. */
    private int  ParsePosition           = 0;

    //*****************************************************************************
    /** Constructor.
     *
     *  @param  InputDescription  Description of exception
     *  @param  InputLineNum      The source line at which the exception occurred
     *  @param  InputColumnNum    The source column at which the exception occurred
     *  @param  InputLastSignificantPosition  The position of the last significant
     *                                         event
     *  @param  InputParsePosition  The current character position in the source
     */
    public xm_ParseException(String InputDescription,
                             int    InputLineNum,
                             int    InputColumnNum,
                             int    InputLastSignificantPosition,
                             int    InputParsePosition)
    {
        super(InputDescription);
        LineNum                 = InputLineNum;
        ColumnNum               = InputColumnNum;
        LastSignificantPosition = InputLastSignificantPosition;
        ParsePosition           = InputParsePosition;
    }

    //*****************************************************************************
    /** Constructor.
     *
     *  @param  InputDescription  Description of exception
     *  @param  InputLineNum      The source line at which the exception occurred
     *  @param  InputColumnNum    The source column at which the exception occurred
     */
    public xm_ParseException(String InputDescription,
                             int    InputLineNum,
                             int    InputColumnNum)
    {
        super(InputDescription);
        LineNum   = InputLineNum;
        ColumnNum = InputColumnNum;
    }

    //*****************************************************************************
    /** Constructor.
     *
     *  @param  InputDescription  Description of exception
     */
    public xm_ParseException(String InputDescription)
    {
        super(InputDescription);
    }

    //*****************************************************************************
    /** Get the number of the source line at which the exception occurred.
     *
     *  @return  The source line at which the exception occurred
     */
    public int getLineNum()
    {
        return LineNum;
    }

    //*****************************************************************************
    /** Get the number of the source column at which the exception occurred.
     *
     *  @return  The source column at which the exception occurred
     */
    public int getColumnNum()
    {
        return ColumnNum;
    }

    //*****************************************************************************
    /** Get the position in the source at which the last significant item occurred.
     *
     *  @return  The position in the source of the last significant event
     */
    public int getLastSignificantPosition()
    {
        return LastSignificantPosition;
    }

    //*****************************************************************************
    /** Get the position in the source (number of characters) when the exception
     *  occurred.
     *
     *  @return  The source position at which the exception occurred
     */
    public int getParsePosition()
    {
        return ParsePosition;
    }

    //*****************************************************************************
    /** Get a string representation of this exception.
     *
     *  @return  A string representation of this exception
     */
/*    public String toString()
    {
        String  ExceptionString = new String("xm.xm_ParseException at line "
                                               + LineNum + ", column "
                                               + ColumnNum + ", positions "
                                               + LastSignificantPosition + "-"
                                               + ParsePosition);
        return ExceptionString;
    }
*/

    //*****************************************************************************
    /** Get a string representation of this exception.
     *
     *  @return  A string representation of this exception
     */
    public String getMessage2()
    {
        String  ExceptionString = new String("xm.xm_ParseException at line "
                                               + LineNum + ", column "
                                               + ColumnNum + ", positions "
                                               + LastSignificantPosition + "-"
                                               + ParsePosition);
        return ExceptionString;
    }
}

//*****************************************************************************
