//*****************************************************************************
/*
** FILE:   xm_NodeTypeDefaultsElement.java
**
** (c) 1998 Steve Withall.
**
** HISTORY:
**    04Aug98  stevew  Created.
*/
package xm;

import xg.xg_Element;
import xg.xg_VerificationException;

import java.sql.Driver;
import java.sql.DriverManager;

import eh.eh_Debug;

//*****************************************************************************
/** An XML element which sets the default values for class names when
 *  subsequently registering entries in the node type registry. This element
 *  exists purely for convenience: to save having to repeat the same class
 *  names over and over again.
 */
public class xm_NodeTypeDefaultsElement extends xg_Element
{
    /** The element type name normally used in XML for this sort of element. */
    public final static String RegisteredName         = "NodeTypeDefaults";

    /** The name of the attribute which holds the name of the default node class. */
//    public final static String NodeClassAttName       = "NodeClass";

    /** The name of the attribute which holds the name of the default parser
     *  class. */
    public final static String ParserClassAttName     = "ParserClass";

    /** The name of the attribute which holds the name of the default customizer
     *  class. */
    public final static String CustomizerClassAttName = "CustomizerClass";

    /** The name of the attribute which holds the name of the default view class. */
    public final static String ViewClassAttName       = "ViewClass";

    //*****************************************************************************
    /** Construct a node type defaults element with no name.
     */
    public xm_NodeTypeDefaultsElement()
    {
    }

    //*****************************************************************************
    /** Construct a node type defaults element with a name.
     *
     *  @param  InputNodeName  The name of the element
     */
    public xm_NodeTypeDefaultsElement(String InputNodeName)
    {
        super(InputNodeName);
    }

    //*****************************************************************************
    /** <p>Verify that this node is correct (ie. internally correct and/or consistent
     *  with other nodes - such as its parent). The node is OK if it possesses a
     *  'Name' attribute which names a valid node type registry entry class.</p>
     *
     *  @exception  xg_VerificationException  Description of verification problem
     */
    public void verify() throws xg_VerificationException
    {
        eh_Debug.add(8, "xm_NodeTypeDefaultsElement.verify: Verify '" + NodeName + "'");

        // Obtain the names of the classes for which to set defaults. All of them
        // are optional, but if present, the class must exist.
//        String  NodeClassName       = checkClassExistence(NodeClassAttName,
//                                                          "node");
        String  ParserClassName     = checkClassExistence(ParserClassAttName,
                                                          "parser");
        if (ParserClassName != null)
            xm_NodeTypeRegistry.setDefaultParserClassName(ParserClassName);

        String  CustomizerClassName = checkClassExistence(CustomizerClassAttName,
                                                          "customizer");
        if (CustomizerClassName != null)
            xm_NodeTypeRegistry.setDefaultCustomizerClassName(CustomizerClassName);

        String  ViewClassName       = checkClassExistence(ViewClassAttName,
                                                          "view");
        if (ViewClassName != null)
            xm_NodeTypeRegistry.setDefaultViewClassName(ViewClassName);
    }

    //*****************************************************************************
    /** Check that the class exists.
     *
     *  @param      InputClassName             The name of the class to be checked
     *  @param      InputClassTypeDescription  A snippet of text to place in any
     *                                          exception to explain the type of
     *                                          class which could not be found
     *  @return     The name of the class from the attribute named InputAttName
     *  @exception  xg_VerificationException  The named class does not exist
     */
    public String checkClassExistence(String  InputAttName,
                                      String  InputClassTypeDescription)
                                                throws xg_VerificationException
    {
        // Obtain name of the class from the attribute named in InputAttName.
        String  ClassName = getAttributeValue(InputAttName);

        if (ClassName != null)
        {
            try
            {
                Class.forName(ClassName);  // Load the named class
            }
            catch (Exception InputException)
            {
                throw new xg_VerificationException("Error loading "
                                                      + InputClassTypeDescription
                                                      + " class '"
                                                      + ClassName + "': "
                                                      + InputException);
            }
        }

        return ClassName;
    }

    //*****************************************************************************
    /** Get a summary descriptive string suitable for display in the tree view or
     *  elsewhere.
     *
     *  @return  A description suitable for display in the tree view
     */
    public String toString()
    {
        return("Node type defaults");
    }
}

//*****************************************************************************
