//*****************************************************************************
/*
** FILE:   xg_NamespaceDecl.java
**
** (c) 1998 Steve Withall.
**
** HISTORY:
**    31May98  stevew  Created.
*/
package xg;

import xa.xa_Keyword;
import xa.xa_NodeTypeChoiceList;

import eh.eh_Debug;

import java.io.IOException;
import java.io.Writer;

//*****************************************************************************
/** An XML namespace declaration.
*/
public class xg_NamespaceDecl extends xg_Node
{
    /** List of attributes defining this namespace */
    protected xg_AttList    NamespaceAttList       = new xg_AttList();

    /** Specification of the namespace name, in the form "ns = '(Literal)'".
     *  This is mandatory*/
    protected xg_Attribute  NamespaceNameAttribute = null;

    /** Specification of the namespace source, in the form "src = '(Literal)'".
     *  This is optional*/
    protected xg_Attribute  SourceAttribute        = null;

    /** Specification of the namespace prefix, in the form "prefix = '(Name)'".
     *  This is mandatory*/
    protected xg_Attribute  PrefixAttribute        = null;

    /** The whitespace preceding the '?>' at the end. */
    protected String        WhitespacePrecedingEnd = null;

    // Constants.
    public final static String NAMESPACE_STRING        = "namespace";
    public final static String NAMESPACE_NAME_STRING   = "ns";
    public final static String NAMESPACE_PREFIX_STRING = "prefix";
    public final static String NAMESPACE_SOURCE_STRING = "src";

    //*****************************************************************************
    /** Default constructor.
     */
    public xg_NamespaceDecl()
    {
    }

    //*****************************************************************************
    /** Save this namespace declaration in XML source form in InputWriter.
     *
     *  @param  InputWriter   The writer to which the XML declaration will be written
     */
    public void save(Writer  InputWriter) throws IOException
    {
        eh_Debug.add(7, "xg_NamespaceDecl.save: Save namespace declaration");  //TEMP
        InputWriter.write(PrecedingWhitespace);
        InputWriter.write(xa_Keyword.PI_START_STRING);   // <?
        InputWriter.write(xa_Keyword.XML_STRING);        //   xml
        InputWriter.write(xa_Keyword.COLON_CHAR);        //      :
        InputWriter.write(NAMESPACE_STRING);             //       namespace

        // Save the attributes.
        NamespaceAttList.save(InputWriter);

        // Save whitespace preceding the end tag.
        if (WhitespacePrecedingEnd != null)
            InputWriter.write(WhitespacePrecedingEnd);

        InputWriter.write(xa_Keyword.PI_END_STRING);     // ?>
    }

    //*****************************************************************************
    /** Add the given attribute.
     *
     *  @param   InputAttribute   Attribute to be added
     *  @return  true if the attribute was added; false if an attribute with this
     *            name already exists
     */
    public boolean addAttribute(xg_Attribute  InputAttribute)
    {
        boolean  SuccessFlag = NamespaceAttList.addAttribute(InputAttribute);
        if (SuccessFlag)
        {
            String  NewAttName = InputAttribute.getName();
            if (NewAttName.equals(NAMESPACE_NAME_STRING) )
                NamespaceNameAttribute = InputAttribute;
            else if (NewAttName.equals(NAMESPACE_SOURCE_STRING) )
                SourceAttribute        = InputAttribute;
            else if (NewAttName.equals(NAMESPACE_PREFIX_STRING) )
                PrefixAttribute        = InputAttribute;
        }
        return SuccessFlag;
    }

    //*****************************************************************************
    /** Set the name of this namespace.
     *
     *  @param  InputNamespaceNameAttribute   Attribute containing full details of
     *                                         the 'ns' of this namespace declaration
     */
/*    public void setNamespaceName(xg_Attribute  InputNamespaceNameAttribute)
    {
        NamespaceNameAttribute = InputNamespaceNameAttribute;
    }
*/
    //*****************************************************************************
    /** Set the source of this namespace.
     *
     *  @param  InputSourceAttribute   Attribute containing full details of the
     *                                  'src' of the namespace
     */
/*    public void setSource(xg_Attribute  InputSourceAttribute)
    {
        SourceAttribute = InputSourceAttribute;
    }
*/
    //*****************************************************************************
    /** Set the prefix of this namespace.
     *
     *  @param  InputPrefixAttribute   Attribute containing full details of the
     *                                  'prefix' of this namespace
     */
/*    public void setPrefix(xg_Attribute  InputPrefixAttribute)
    {
        PrefixAttribute = InputPrefixAttribute;
    }
*/
    //*****************************************************************************
    /** Set the whitespace preceding the end tag of this namespace declaration.
     *
     *  @param  InputWhitespacePrecedingEnd   Whitespace before the end
     */
    public void setWhitespacePrecedingEnd(String  InputWhitespacePrecedingEnd)
    {
        WhitespacePrecedingEnd = InputWhitespacePrecedingEnd;
    }

    //*****************************************************************************
    /** Get the type of this node.
     *
     *  @return  The type of this node
     */
    public int getType()
    {
        return xa_NodeTypeChoiceList.NAMESPACE_DECL_TYPE;
    }

    //*****************************************************************************
    /** Get the type of this node, using only type values defined for the DOM.
     *  (Implements org.w3c.dom.Node.getNodeType.)
     *
     *  @return  The DOM type of this node
     */
    public int getNodeType()
    {
        return xa_NodeTypeChoiceList.PI_TYPE;
    }

    //*****************************************************************************
    /** Get the list of attributes.
     *
     *  @return  The list of attributes
     */
    public xg_AttList getAttList()
    {
        return NamespaceAttList;
    }

    //*****************************************************************************
    /** Get the named attribute.
     *
     *  @param   InputAttName  The name of the attribute to retrieve
     *  @return                The named attribute (or null if it doesn't exist)
     */
    public xg_Attribute getAttribute(String  InputAttName)
    {
        return NamespaceAttList.getAtt(InputAttName);
    }

    //*****************************************************************************
    /** Get the name of this namespace.
     *
     *  @return  The name of this namespace (the value of the 'ns' attribute)
     */
    public String getNamespaceName()
    {
        String  NameSpaceName = null;
        if (NamespaceNameAttribute != null)
            NameSpaceName = NamespaceNameAttribute.getValue();
        return NameSpaceName;
    }

    //*****************************************************************************
    /** Get the attribute containing the name of this namespace.
     *
     *  @return  Attribute containing full details of the 'ns' of this namespace
     */
    public xg_Attribute getNamespaceNameAtt()
    {
        return NamespaceNameAttribute;
    }

    //*****************************************************************************
    /** Get the name of the source of this namespace.
     *
     *  @return  The name of this source (the value of the 'src' attribute)
     */
    public String getSource()
    {
        String  SourceName = null;
        if (SourceAttribute != null)
            SourceName = SourceAttribute.getValue();
        return SourceName;
    }

    //*****************************************************************************
    /** Get the attribute containing the source of this namespace.
     *
     *  @return  Attribute containing full details of the 'src' of the namespace
     */
    public xg_Attribute getSourceAtt()
    {
        return SourceAttribute;
    }

    //*****************************************************************************
    /** Get the prefix name of this namespace.
     *
     *  @return  The name of this prefix (the value of the 'prefix' attribute)
     */
    public String getPrefix()
    {
        String  PrefixName = null;
        if (PrefixAttribute != null)
            PrefixName = PrefixAttribute.getValue();
        return PrefixName;
    }

    //*****************************************************************************
    /** Get the attribute containing the prefix of this namespace.
     *
     *  @return  Attribute containing full details of the 'prefix' of this namespace
     */
    public xg_Attribute getPrefixAtt()
    {
        return PrefixAttribute;
    }

    //*****************************************************************************
    /** Return a string representation of the namespace declaration (intended for
     *  use as debug output).
     *
     *  @return  String representation of this namespace declaration.
     */
    public String toString()
    {
        return ("Namespace declaration '" + getPrefix() + "'");
    }
}

//*****************************************************************************
