//*****************************************************************************
/*
** FILE:   xg_Attribute.java
**
** (c) 1997, 1998 Steve Withall.
**
** HISTORY:
**    13Oct97  stevew  Created.
**    31May98  stevew  Implemented org.w3c.dom.Attribute interface.
*/
package xg;

import xa.xa_Keyword;
import eh.eh_Debug;

//TBD See comments below.
//import org.w3c.dom.Attribute;

import java.io.IOException;
import java.io.Writer;

//*****************************************************************************
/** An attribute of an entity, which is basically a name and a value. It also
 *  has several whitespace values, to allow it to be rewritten exactly as read,
 *  as follows:
 *
 *    [PrecedingWhitespace][Name][WhitespaceBeforeEquals]=[WhitespaceAfterEquals]"[Value]"
 *
 *  <p>Note that this class implements all the methods of the DOM
 *  org.w3c.dom.AttributeList interface, but since the rest of xg does not,
 *  references to the DOM interfaces is commented out to allow xg to run
 *  without the DOM interfaces being present.</p>
*/
public class xg_Attribute
//                          implements org.w3c.dom.Attribute
{
    /** The name of this attribute. */
    private String          AttName  = null;

    /** The value of this attribute. */
    private String          AttValue = null;

    /** Flag to indicate whether the value of this attribute has been specified
     *  explicitly (in which case it is 'true'), or whether it has been created
     *  with its default ('false'). */
    private boolean         AttSpecifiedFlag = true;

    /** Whitespace before the name. */
    private String          PrecedingWhitespace    = null;

    /** Whitespace before the '=' sign. */
    private String          WhitespaceBeforeEquals = null;

    /** Whitespace after the '=' sign. */
    private String          WhitespaceAfterEquals  = null;

    //*****************************************************************************
    /**
     * Constructor: create an attribute based on three values, which represent the
     * name, equals and value parts. Each 'value' can contain preceding whitespace.
     *
     * @param  InputNameValue    Value for the name of the attribute
     * @param  InputEqualsValue  Value for the '=' of the attribute
     * @param  InputValueValue   Value for the value of the attribute
     */
    public xg_Attribute(xg_Value  InputNameValue,
                        xg_Value  InputEqualsValue,
                        xg_Value  InputValueValue)
    {
        setPrecedingWhitespace(InputNameValue.getPrecedingWhitespace() );
        setName(InputNameValue.getValue() );
        setWhitespaceBeforeEquals(InputEqualsValue.getPrecedingWhitespace() );
        setWhitespaceAfterEquals(InputValueValue.getPrecedingWhitespace() );
        setValue(InputValueValue.getValue() );
    }

    //*****************************************************************************
    /**
     * Constructor: create an attribute with a name and value and the simplest
     * possible whitespace to make it well-formed: one preceding space, and no
     * whitespace around the '='.
     *
     * @param  InputAttName   The name of this attribute
     * @param  InputAttValue  The value of this attribute
     */
    public xg_Attribute(String  InputAttName, String  InputAttValue)
    {
        PrecedingWhitespace = new String(" ");
        AttName  = InputAttName;
        AttValue = InputAttValue;
    }

    //*****************************************************************************
    /**
     * Default constructor.
     */
    public xg_Attribute()
    {
    }

    //*****************************************************************************
    /**
     * Save this attribute in InputWriter in XML source form:
     *
     *   Name = "Value"
     *
     * @param  InputWriter   The writer to which the XML will be written
     */
    public void save(Writer  InputWriter) throws IOException
    {
//        eh_Debug.add(9, "XE_Attribute.save: Save attribute '" + AttName + "'");  //TEMP

        if (PrecedingWhitespace != null)
            InputWriter.write(PrecedingWhitespace);
        InputWriter.write(AttName);
        if (WhitespaceBeforeEquals != null)
            InputWriter.write(WhitespaceBeforeEquals);
        InputWriter.write(xa_Keyword.EQUALS_CHAR);
        if (WhitespaceAfterEquals != null)
            InputWriter.write(WhitespaceAfterEquals);

        // Choose the type of quote to use. (Use doublequotes unless the value
        // contains a doublequote.)
        char  QuoteValue = '"';
        if (AttValue.indexOf('"') >= 0)
            QuoteValue = '\'';

        InputWriter.write(QuoteValue);
        InputWriter.write(AttValue);
        InputWriter.write(QuoteValue);
    }

    //*****************************************************************************
    /**
     * Create a new xg_Attribute which is exactly the same as this one.
     *
     * @return  A duplicate attribute
     */
    public xg_Attribute duplicate()
    {
        xg_Attribute  NewAtt = new xg_Attribute();
        NewAtt.setName(AttName);
        NewAtt.setValue(AttValue);
        NewAtt.setSpecified(AttSpecifiedFlag);
        NewAtt.setPrecedingWhitespace(PrecedingWhitespace);
        NewAtt.setWhitespaceBeforeEquals(WhitespaceBeforeEquals);
        NewAtt.setWhitespaceAfterEquals(WhitespaceAfterEquals);
        return NewAtt;
    }

    //*****************************************************************************
    /**
     * Set the name of this attribute.
     *
     * @param  InputName   Name of the attribute
     */
    public void setName(String  InputAttName)
    {
        AttName = InputAttName;
    }

    //*****************************************************************************
    /**
     * Set the value of this attribute.
     *
     * @param  InputAttValue   Value of the attribute
     */
    public void setValue(String  InputAttValue)
    {
        AttValue = InputAttValue;
    }

    //*****************************************************************************
    /**
     * Set the value of the preceding whitespace.
     *
     * @param  InputPrecedingWhitespace   Value of the preceding whitespace
     */
    public void setPrecedingWhitespace(String  InputPrecedingWhitespace)
    {
        PrecedingWhitespace = InputPrecedingWhitespace;
    }

    //*****************************************************************************
    /**
     * Set the value of the whitespace before the equals sign.
     *
     * @param  InputWhitespaceBeforeEquals   Value of the whitespace before the
     *                                       equals sign
     */
    public void setWhitespaceBeforeEquals(String  InputWhitespaceBeforeEquals)
    {
        WhitespaceBeforeEquals = InputWhitespaceBeforeEquals;
    }

    //*****************************************************************************
    /**
     * Set the value of the whitespace after the equals sign.
     *
     * @param  InputWhitespaceAfterEquals   Value of the whitespace after the
     *                                      equals sign
     */
    public void setWhitespaceAfterEquals(String  InputWhitespaceAfterEquals)
    {
        WhitespaceAfterEquals = InputWhitespaceAfterEquals;
    }

    //*****************************************************************************
    /**
     * Set the flag which indicates whether this attribute's value has been set
     * explicitly. (Implements org.w3c.dom.Attribute.setSpecified().)
     *
     * @param  InputAttSpecifiedFlag - if true it indicates that the value of this
     *          attribute has been set explicitly; if false then if it is deemed to
     *          have been assigned its default value.
     */
    public void setSpecified(boolean  InputAttSpecifiedFlag)
    {
        AttSpecifiedFlag = InputAttSpecifiedFlag;
    }

    //*****************************************************************************
    /**
     * Get the name of this attribute. (Implements org.w3c.dom.Attribute.getName().)
     *
     * @return  The name of the attribute.
     */
    public String getName()
    {
        return AttName;
    }

    //*****************************************************************************
    /**
     * Get the value of this attribute. (Implements org.w3c.dom.Attribute.getValue().)
     *
     * @return  The value of the attribute.
     */
    public String getValue()
    {
        return AttValue;
    }

    //*****************************************************************************
    /**
     * Get the flag which indicates whether this attribute's value has been set
     * explicitly. (Implements org.w3c.dom.Attribute.getSpecified().)
     *
     * @return  true if the value of this attribute has been set explicitly; false
     *           if it has been assigned its default value.
     */
    public boolean getSpecified()
    {
        return AttSpecifiedFlag;
    }

    //*****************************************************************************
    /**
     * Get the value of the preceding whitespace.
     *
     * @return  String value or null
     */
    public String getPrecedingWhitespace()
    {
        return PrecedingWhitespace;
    }

    //*****************************************************************************
    /**
     * Return a string representation of the attribute (intended for use as debug
     * output). (Implements org.w3c.dom.Attribute.getValue().)
     *
     * @return  String representing the attribute name and value.
     */
    public String toString()
    {
        return "Name = " + AttName + "; Value = " + AttValue;
    }
}

//*****************************************************************************
