//*****************************************************************************
/*
** FILE:   xe_TokenTypeDefn.java
**
** (c) 1997, 1998 Steve Withall.
**
** HISTORY:
**    19Oct97  stevew  Created.
**    08Jun98  stevew  Renamed from xe_KeywordDefn.
**                     Removed information about node-related classes (which
**                      is now handled by xe_NodeTypeDefn).
*/
package xe;

//*****************************************************************************
/**
*  Class xe_TokenTypeDefn - The full definition of a particular XML keyword.
*  Token types define the low-level tokens (reserved words, sequences of symbols
*  and other constructs) which make up XML, and the definition of each token type
*  specifies a little about how it can be used, and whether it is a subset of
*  some other token type.
*
*  The attributes of an xe_TokenTypeDefn are used to control certain aspects of
*  the low-level parsing.
*/
public class xe_TokenTypeDefn
{
    /** The string value of this token type. */
    protected String   TokenValue;

    /** The type of this token. */
    protected int      TokenType;

    /** A string representation of the type of this token. */
    protected String   TokenTypeString;

    /** If this token represents the beginning of a node, this is the type of node. */
    protected int      NodeType;

    /** If we parsed this value, could it be considered a valid token? True means
     *  it can; false means it cannot. */
    protected boolean  ValidTokenFlag;

    /** Are there other token types which begin with this string value? This value was
     *  added to drive autoparsing (see class xe_AutoParseManager), although that is
     *  now considered more as an interesting experiment than anything else. The
     *  allowed values are:
     *
     *   xe_TokenTypeRegistry.NO_MORE      - no other token types begin with these
     *                                        characters
     *   xe_TokenTypeRegistry.MORE_SPECIAL - by following this with one non-alphanumeric
     *                                        character (and maybe other characters),
     *                                        we may get another valid token type
     *   xe_TokenTypeRegistry.MORE_NAME    - by following this with a name, we may
     *                                        get another valid token type
     *   xe_TokenTypeRegistry.MORE_NUMBER  - by following this with a name, we may
     *                                        get another valid token type
     *   xe_TokenTypeRegistry.MORE_ANY     - by following this with either special
     *                                        characters or a name, we may get
     *                                        another valid token type. */
    protected int      MoreControl;

    //*****************************************************************************
    /**
     * Construct a token type definition.
     *
     * @param  InputTokenValue       The string value of this token type
     * @param  InputTokenType        Type of the token
     * @param  InputTokenTypeString  String representation of the token type
     * @param  InputNodeType         Type of the node for which this is the opening
     *                                token (or 0 if it is not the beginning of a
     *                                node)
     * @param  InputValidTokenFlag   If we parsed this value, could it be
     *                                considered a valid token?
     * @param  InputMoreControl      Are there other token types which begin with this
     *                                string value? The allowed values are:
     *                                xe_TokenTypeRegistry.NO_MORE,
     *                                xe_TokenTypeRegistry.MORE_SPECIAL,
     *                                xe_TokenTypeRegistry.MORE_NAME,
     *                                xe_TokenTypeRegistry.MORE_NUMBER,
     *                                xe_TokenTypeRegistry.MORE_ANY.
     */
    public xe_TokenTypeDefn(String   InputTokenValue,
                            int      InputTokenType,
                            String   InputTokenTypeString,
                            int      InputNodeType,
                            boolean  InputValidTokenFlag,
                            int      InputMoreControl)
    {
        TokenValue        = InputTokenValue;
        TokenType         = InputTokenType;
        TokenTypeString   = InputTokenTypeString;
        NodeType          = InputNodeType;
        ValidTokenFlag    = InputValidTokenFlag;
        MoreControl       = InputMoreControl;
    }

    //*****************************************************************************
    /**
     * Get the string value of the token type.
     *
     * @return  The string value of the token type
     */
    public String getValue()
    {
        return TokenValue;
    }

    //*****************************************************************************
    /**
     * Get the type of the token.
     *
     * @return  The type of the token
     */
    public int getType()
    {
        return TokenType;
    }

    //*****************************************************************************
    /**
     * Get the string representation of the type of the token.
     *
     * @return  String representation of the token type
     */
    public String getTypeString()
    {
        return TokenTypeString;
    }

    //*****************************************************************************
    /**
     * Get the type of the node for which this is the opening token.
     *
     * @return  The type of the node for which this is the opening token. 0 means
     *           this is not the opening token of a node.
     */
    public int getNodeType()
    {
        return NodeType;
    }

    //*****************************************************************************
    /**
     * Get the value specifying whether this is a valid token type.
     *
     * @return  If we parsed this value, could it be considered a valid token?
     */
    public boolean getValidTokenFlag()
    {
        return ValidTokenFlag;
    }

    //*****************************************************************************
    /**
     * Get the value specifying whether there are others beginning with these
     * characters.
     *
     * @return  Are there other token types which begin with this string value?
     */
    public int getMoreControl()
    {
        return MoreControl;
    }

    //*****************************************************************************
    /**
     * Return a string representation of this token type definition - mainly to be
     * used for debug purposes.
     *
     * @return  String representation of this token type definition
     */
    public String toString()
    {
        return (new String("'" + TokenValue
                               + "' is token type " + TokenTypeString
                               + "(" + TokenType + ")") );
    }

    //*****************************************************************************
    /**
     * Set the string value of this token type.
     *
     * @param  InputTokenValue   String value of this token type
     */
    public void setValue(String  InputTokenValue)
    {
        TokenValue = InputTokenValue;
    }

    //*****************************************************************************
    /**
     * Set the type of this token.
     *
     * @param  InputTokenType   Type of this token
     */
    public void setType(int  InputTokenType)
    {
        TokenType = InputTokenType;
    }

    //*****************************************************************************
    /**
     * Set the string representation of the type of this token.
     *
     * @param  InputTokenTypeString   String representation of this token type
     */
    public void setTypeString(String  InputTokenTypeString)
    {
        TokenTypeString = InputTokenTypeString;
    }
}

//*****************************************************************************
