//*****************************************************************************
/*
** FILE:   xa_ChoiceList.java
**
** (c) 1998 Steve Withall.
**
** HISTORY:
**    21Apr98  stevew  Created.
*/
package xa;

import java.util.Hashtable;
import java.util.Vector;

//*****************************************************************************
/** <p>A container for a list of String values which define the values allowed
 *  for whatever use it is being put to. The list is held internally in two
 *  forms: a Hashtable (to allow validation by key), and as a Vector (to allow
 *  retrieval of the values in the order they were supplied).</p>
 *
 *  <p>xa_ChoiceList is intended to be restricted in its use to definitions of
 *  lists of allowed value strings. Each allowed value (choice) string has an
 *  integer associated with it, which may be a constant used programmatically.
 *  No checking is performed on these integer values (so they do not have to be
 *  unique) - but it is inadvisable to use -1, since -1 is used to indicate an
 *  error when failing to find a choice.</p>
 */
public class xa_ChoiceList
{
    /** Allows this choice list to be identified. By qualifying choice names with
     *  the name of the choice list to which they belong, we can derive globally
     *  unique names for all choices. Such a global list may be useful for, say,
     *  documentation purposes - to allow automatic reference to the relevant
     *  section of the XML specification. */
    protected String     ChoiceListName       = null;

    /** Hashtable used to retrieve an allowed value, given the value. */
    protected Hashtable  ChoiceHashtable      = new Hashtable();

    /** Hashtable of allowed values, to allow retrieval of a choice name given its
     *  integer choice value. */
    protected Hashtable  ChoiceValueHashtable = new Hashtable();

    /** Array of allowed values, to allow retrieval in the order they were added. */
    protected Vector     ChoiceVector         = new Vector();

    //*****************************************************************************
    /** Constructor, giving this choice list a name.
     */
    public xa_ChoiceList(String  InputChoiceListName)
    {
        ChoiceListName = InputChoiceListName;
    }

    //*****************************************************************************
    /** Add an allowed value.
     *
     *  @param  InputChoiceName   The value which is allowed
     *  @param  InputChoiceValue  A number associated with this value
     */
    public void addChoice(String  InputChoiceName, int  InputChoiceValue)
    {
        Integer  ChoiceValueObject = new Integer(InputChoiceValue);
        ChoiceHashtable.put(InputChoiceName, ChoiceValueObject);
        ChoiceValueHashtable.put(ChoiceValueObject, InputChoiceName);
        ChoiceVector.addElement(InputChoiceName);
    }

    //*****************************************************************************
    /** Return true of InputChoiceName is a valid allowed value; false if not.
     *
     *  @param  InputChoiceName   The name of the choice to be checked
     */
    public boolean hasChoice(String  InputChoiceName)
    {
        Integer  ChoiceValue = (Integer)ChoiceHashtable.get(InputChoiceName);
        return (ChoiceValue != null);
    }

    //*****************************************************************************
    /** Retrieve the choice whose name is InputChoiceName.
     *
     *  @param  InputChoiceName   The name of the choice to be retrieved
     */
    public int getChoice(String  InputChoiceName)
    {
         int      ChoiceInt   = -1;
        Integer  ChoiceValue = (Integer)ChoiceHashtable.get(InputChoiceName);
        if (ChoiceValue != null)
            ChoiceInt = ChoiceValue.intValue();
        return (ChoiceInt);
    }

    //*****************************************************************************
    /** Retrieve the choice whose integer value is InputChoiceIndex.
     *
     *  @param  InputChoiceValue   The integer value of the choice to be retrieved
     */
    public String getChoice(int  InputChoiceValue)
    {
         Integer  ChoiceValueObject = new Integer(InputChoiceValue);
        return( (String)ChoiceValueHashtable.get(ChoiceValueObject) );
    }

    //*****************************************************************************
    /** Retrieve the choice whose index is InputChoiceIndex in the ChoiceVector.
     *
     *  @param  InputChoiceIndex   The index of the choice to be retrieved
     */
    public String getChoiceAt(int  InputChoiceIndex)
    {
         return( (String)ChoiceVector.elementAt(InputChoiceIndex) );
    }

    //*****************************************************************************
    /** Find out how many choices there are.
     *
     *  @return  The number of choices in the list
     */
    public int getNumChoices()
    {
        return(ChoiceVector.size() );
    }

    //*****************************************************************************
    /** Retrieve the name of this choice list.
     *
     *  @return  The name of this choice list
     */
    public String getName()
    {
        return ChoiceListName;
    }

    //*****************************************************************************
    /** Return a string representation of the choice list, in the form of a comma-
     *  separated list ("A, B, C").
     *
     *  @return  String represention of the list.
     */
    public String toString()
    {
        StringBuffer  ListBuffer        = new StringBuffer();
        String        CurrentChoiceName = null;
        for (int ListIndex = 0;
             ListIndex < ChoiceVector.size();
             ListIndex++)
        {
            if (ListIndex > 0)
                ListBuffer.append(", ");
            CurrentChoiceName = (String)ChoiceVector.elementAt(ListIndex);
            ListBuffer.append(CurrentChoiceName);
        }
        return (ListBuffer.toString() );
    }
}

//*****************************************************************************
