/*
 * Copyright (c) 2003 World Wide Web Consortium,
 *
 * (Massachusetts Institute of Technology, European Research Consortium for
 * Informatics and Mathematics, Keio University). All Rights Reserved. This
 * work is distributed under the W3C(r) Software License [1] in the hope that
 * it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * [1] http://www.w3.org/Consortium/Legal/2002/copyright-software-20021231
 */

package org.w3c.dom.validation;

import org.w3c.dom.Node;
import org.w3c.dom.DOMStringList;

/**
 * This interface is similar to the [<a href='http://www.w3.org/TR/2002/WD-DOM-Level-3-Core-20021022'>DOM Level 3 Core</a>] <code>Node</code> interfaces, with 
 * methods for guided document editing.
 * <p>See also the <a href='http://www.w3.org/TR/2003/WD-DOM-Level-3-Val-20030205'>Document Object Model (DOM) Level 3 Validation Specification</a>.
 */
public interface NodeEditVAL {
    // CheckTypeVAL
    /**
     * Check for well-formedness of this node.
     */
    public static final short WF_CHECK                  = 1;
    /**
     * Check for namespace well-formedness includes <code>WF_CHECK</code>.
     */
    public static final short NS_WF_CHECK               = 2;
    /**
     * Checks for whether this node is partially valid. It includes 
     * <code>NS_WF_CHECK</code>. 
     */
    public static final short PARTIAL_VALIDITY_CHECK    = 3;
    /**
     * Checks for strict validity of the node with respect to the grammar 
     * which by definition includes <code>NS_WF_CHECK</code>.
     */
    public static final short STRICT_VALIDITY_CHECK     = 4;

    /**
     *  The default value specified in an attribute or an element declaration. 
     */
    public String getDefaultValue();

    /**
     *  A <code>DOMStringList</code> of distinct values for an attribute or an 
     * element declaration. 
     */
    public DOMStringList getEnumeratedValues();

    /**
     * Determines whether the <code>Node.insertBefore</code> operation would 
     * make this document not partially valid with respect to the grammar. 
     * @param newChild <code>Node</code> to be inserted.
     * @param refChild Reference <code>Node</code>.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canInsertBefore(Node newChild, 
                                   Node refChild);

    /**
     * Has the same arguments as <code>Node.removeChild</code>. Determines 
     * whether the <code>Node.removeChild</code> operation would make this 
     * document not partially valid with respect to the grammar. 
     * @param oldChild <code>Node</code> to be removed.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canRemoveChild(Node oldChild);

    /**
     * Has the same arguments as <code>Node.replaceChild</code>. Determines 
     * whether the <code>Node.replaceChild</code> operation would make this 
     * document not partially valid with respect to the grammar. 
     * @param newChild New <code>Node</code>.
     * @param oldChild <code>Node</code> to be replaced.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canReplaceChild(Node newChild, 
                                   Node oldChild);

    /**
     * Has the same arguments as <code>Node.appendChild</code>. Determines 
     * whether the <code>Node.replaceChild</code> operation would make this 
     * document not partially valid with respect to the grammar. 
     * @param newChild <code>Node</code> to be appended.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canAppendChild(Node newChild);

    /**
     * Determines if the Node is valid relative to the grammar. It doesn't 
     * normalize before checking if the document is valid. To do so, one 
     * would need to explicitly call a normalize method. 
     * @param deep Setting the <code>deep</code> flag on causes the 
     *   <code>isNodeValid</code> method to check for the whole subtree of 
     *   the current node for validity. Setting it to <code>false</code> 
     *   only checks the current node and its immediate child nodes. The 
     *   <code>validateDocument</code> method on the <code>DocumentVAL</code>
     *    interface, however, checks to determine whether the entire 
     *   document is valid.
     * @param wFValidityCheckLevel Flag to tell at what level validity and 
     *   well-formedness checking is done. 
     * @return <code>true</code> if the node is valid/well-formed in the 
     *   grammar and check level defined by <code>wfValidityCheckLevel</code>
     *   , <code>false</code> if not.
     * @exception ExceptionVAL
     *   NO_GRAMMAR_AVAILABLE_ERR: Exception is raised if the DocumentEditVAL 
     *   related to this node does not have any grammar associated with it 
     *   and <code>wfValidityCheckLevel</code> is set to <code>PARTIAL</code>
     *    or <code>STRICT_VALIDITY_CHECK</code>.
     */
    public boolean isNodeValid(boolean deep, 
                               short wFValidityCheckLevel)
                               throws ExceptionVAL;

}
