/*
 * Copyright (c) 2003 World Wide Web Consortium,
 *
 * (Massachusetts Institute of Technology, European Research Consortium for
 * Informatics and Mathematics, Keio University). All Rights Reserved. This
 * work is distributed under the W3C(r) Software License [1] in the hope that
 * it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * [1] http://www.w3.org/Consortium/Legal/2002/copyright-software-20021231
 */

package org.w3c.dom.validation;

import org.w3c.dom.Node;
import org.w3c.dom.Attr;
import org.w3c.dom.NameList;

/**
 * This interface extends the <code>NodeEditVAL</code> interface with 
 * additional methods for guided document editing. An object implementing 
 * this interface must also implement <code>Element</code> interface.
 * <p>See also the <a href='http://www.w3.org/TR/2003/WD-DOM-Level-3-Val-20030205'>Document Object Model (DOM) Level 3 Validation Specification</a>.
 */
public interface ElementEditVAL extends NodeEditVAL {
    /**
     * A <code>NameList</code> of possible <code>Element</code> nodes that can 
     * appear as children of this type of element. Note that if no context 
     * of this element exists, then this is <code>NULL</code>; it is an 
     * empty list if the element is not in the document tree. Given the 
     * names, nodes can be created from them; note that these are not nodes 
     * from the instance document, but rather are new nodes that could be 
     * inserted in the document. 
     */
    public NameList getAllowedChildren();

    /**
     * A <code>NameList</code> of possible <code>Element</code> nodes that can 
     * appear as a parent of this type of element. Note that if no context 
     * of this element exists, for example, the parent element of this 
     * element, then this is <code>NULL</code>; it is an empty list if the 
     * element is not in the document tree. Given the names, nodes can be 
     * created from them; note that these are not nodes from the instance 
     * document, but rather are new nodes that could be inserted in the 
     * document. 
     */
    public NameList getAllowedParents();

    /**
     * A <code>NameList</code> of possible sibling <code>Element</code> nodes 
     * that can appear after this element. 
     */
    public NameList getAllowedNextSiblings();

    /**
     * A <code>NameList</code> of possible sibling <code>Element</code> nodes 
     * that can appear before this element. 
     */
    public NameList getAllowedPreviousSiblings();

    /**
     * A <code>NameList</code> of possible <code>Attr</code> nodes that can 
     * appear with this type of element. Given the names, nodes can be 
     * created from them; note that these are not nodes from the instance 
     * document, but rather are new nodes that could be inserted in the 
     * document. 
     */
    public NameList getAllowedAttributes();

    /**
     * A <code>NameList</code> of required <code>Attr</code> nodes that must 
     * appear with this type of element. Given the names, nodes can be 
     * created from them; note that these are not nodes from the instance 
     * document, but rather are new nodes that could be inserted in the 
     * document. 
     */
    public NameList getRequiredAttributes();

    /**
     * Determines element content type.
     * @return Constant for one of <code>EMPTY_CONTENTTYPE</code> - content 
     *   type of an element which has neither child elements nor character 
     *   data, <code>ANY_CONTENTTYPE</code> - content type of an element 
     *   which can contain zero or more child elements as well as character 
     *   data, <code>MIXED_CONTENTTYPE</code> - content type of an element 
     *   which contains character data optionally interspersed with child 
     *   elements, <code>ELEMENTS_CONTENTTYPE</code> - content type of an 
     *   element which contains only child elements optionally separated by 
     *   whitespace, <code>SIMPLE_CONTENTTYPE</code> - content type of an 
     *   element which contains character data with attributes.
     */
    public short contentType();

    /**
     * Determines if the value for specified attribute can be set.
     * @param attrname Name of attribute.
     * @param attrval Value to be assigned to the attribute.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canSetAttribute(String attrname, 
                                   String attrval);

    /**
     * Determines if an attribute node can be added with respect to the 
     * validity check level.
     * @param attrNode <code>Node</code> in which the attribute can possibly 
     *   be set.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canSetAttributeNode(Attr attrNode);

    /**
     * Determines if the attribute with given namespace and qualified name can 
     * be created if not already present in the attribute list of the 
     * element. If the attribute with the same qualified name and 
     * namespaceURI is already present in the element's attribute list, it 
     * tests whether the value of the attribute and its prefix can be set to 
     * the new value. See DOM core <code>setAttributeNS</code>.
     * @param namespaceURI <code>namespaceURI</code> of namespace.
     * @param qualifiedName Qualified name of attribute.
     * @param value Value to be assigned to the attribute.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canSetAttributeNS(String namespaceURI, 
                                     String qualifiedName, 
                                     String value);

    /**
     * Verifies if an attribute by the given name can be removed.
     * @param attrname Name of attribute.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canRemoveAttribute(String attrname);

    /**
     * Verifies if an attribute by the given local name and namespace can be 
     * removed.
     * @param namespaceURI The namespace URI of the attribute to remove.
     * @param localName Local name of the attribute to be removed.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canRemoveAttributeNS(String namespaceURI, 
                                        String localName);

    /**
     * Determines if an attribute node can be removed.
     * @param attrNode The <code>Attr</code> node to remove from the 
     *   attribute list.
     * @return <code>true</code> if no reason it can't be done; 
     *   <code>false</code> if it can't be done.
     */
    public boolean canRemoveAttributeNode(Node attrNode);

    /**
     * Determines if <code>name</code> is defined in the grammar.
     * @param name Name of element.
     * @return A boolean that is <code>true</code> if the element is defined, 
     *   false otherwise.
     */
    public boolean isElementDefined(String name);

    /**
     * Determines if <code>name</code> in this namespace is defined in the 
     * current context.
     * @param namespaceURI <code>namespaceURI</code> of namespace.
     * @param name Name of element.
     * @return A boolean that is <code>true</code> if the element is defined, 
     *   false otherwise.
     */
    public boolean isElementDefinedNS(String namespaceURI, 
                                      String name);

}
