/*
 * Copyright (c) 2003 World Wide Web Consortium,
 *
 * (Massachusetts Institute of Technology, European Research Consortium for
 * Informatics and Mathematics, Keio University). All Rights Reserved. This
 * work is distributed under the W3C(r) Software License [1] in the hope that
 * it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * [1] http://www.w3.org/Consortium/Legal/2002/copyright-software-20021231
 */

package org.w3c.dom.events;

import org.w3c.dom.views.AbstractView;

/**
 * The <code>MouseEvent</code> interface provides specific contextual 
 * information associated with Mouse events.
 * <p> In the case of nested elements mouse events are always targeted at the 
 * most deeply nested element. Ancestors of the targeted element may use 
 * bubbling to obtain notification of mouse events which occur within its 
 * descendent elements.  To create an instance of the <code>MouseEvent</code>
 *  interface, use the feature string "MouseEvent" as the value of the input 
 * parameter used with the <code>DocumentEvent.createEvent</code> method. 
 * <p>See also the <a href='http://www.w3.org/TR/2003/WD-DOM-Level-3-Events-20030221'>Document Object Model (DOM) Level 3 Events Specification</a>.
 * @since DOM Level 2
 */
public interface MouseEvent extends UIEvent {
    /**
     * The horizontal coordinate at which the event occurred relative to the 
     * origin of the screen coordinate system.
     */
    public int getScreenX();

    /**
     * The vertical coordinate at which the event occurred relative to the 
     * origin of the screen coordinate system.
     */
    public int getScreenY();

    /**
     * The horizontal coordinate at which the event occurred relative to the 
     * DOM implementation's client area.
     */
    public int getClientX();

    /**
     * The vertical coordinate at which the event occurred relative to the DOM 
     * implementation's client area.
     */
    public int getClientY();

    /**
     *  <code>true</code> if the control (Ctrl) key modifier is activated. 
     */
    public boolean getCtrlKey();

    /**
     *  <code>true</code> if the shift (Shift) key modifier is activated. 
     */
    public boolean getShiftKey();

    /**
     *  <code>true</code> if the alt (alternative) key modifier is activated. 
     */
    public boolean getAltKey();

    /**
     *  <code>true</code> if the meta (Meta) key modifier is activated.  The 
     * Command key modifier on Macintosh system must be represented using 
     * this key modifier. 
     */
    public boolean getMetaKey();

    /**
     *  During mouse events caused by the depression or release of a mouse 
     * button, <code>button</code> is used to indicate which mouse button 
     * changed state. <code>0</code> indicates the normal (in general on the 
     * left or the one button on Macintosh mice, used to activate a button 
     * or select text) button of the mouse. <code>2</code> indicates the 
     * contextual property (in general on the right, used to display a 
     * context menu) button of the mouse if present. <code>1</code> 
     * indicates the extra (in general in the middle and often combined with 
     * the mouse wheel) button. Some mice may provide or simulate more 
     * buttons and values higher than <code>2</code> could be used to 
     * represent such buttons. 
     */
    public short getButton();

    /**
     *  Used to identify a secondary <code>EventTarget</code> related to a UI 
     * event. Currently this attribute is used with the mouseover event to 
     * indicate the <code>EventTarget</code> which the pointing device 
     * exited and with the mouseout event to indicate the 
     * <code>EventTarget</code> which the pointing device entered.
     */
    public EventTarget getRelatedTarget();

    /**
     * The <code>initMouseEvent</code> method is used to initialize the value 
     * of a <code>MouseEvent</code> created using the 
     * <code>DocumentEvent.createEvent</code> method. This method may only 
     * be called before the <code>MouseEvent</code> has been dispatched via 
     * the <code>EventTarget.dispatchEvent</code> method, though it may be 
     * called multiple times before being dispatched. If called multiple 
     * times, the final invocation takes precedence.
     * @param typeArg Specifies the event type.
     * @param canBubbleArg Specifies whether or not the event can bubble.
     * @param cancelableArg Specifies whether or not the event's default 
     *   action can be prevented.
     * @param viewArg Specifies the <code>Event</code>'s view.
     * @param detailArg Specifies the <code>Event</code>'s mouse click count.
     * @param screenXArg Specifies the <code>Event</code>'s screen x 
     *   coordinate
     * @param screenYArg Specifies the <code>Event</code>'s screen y 
     *   coordinate
     * @param clientXArg Specifies the <code>Event</code>'s client x 
     *   coordinate
     * @param clientYArg Specifies the <code>Event</code>'s client y 
     *   coordinate
     * @param ctrlKeyArg Specifies whether or not control key was depressed 
     *   during the <code>Event</code>.
     * @param altKeyArg Specifies whether or not alt key was depressed during 
     *   the <code>Event</code>.
     * @param shiftKeyArg Specifies whether or not shift key was depressed 
     *   during the <code>Event</code>.
     * @param metaKeyArg Specifies whether or not meta key was depressed 
     *   during the <code>Event</code>.
     * @param buttonArg Specifies the <code>Event</code>'s mouse button.
     * @param relatedTargetArg Specifies the <code>Event</code>'s related 
     *   <code>EventTarget</code>.
     */
    public void initMouseEvent(String typeArg, 
                               boolean canBubbleArg, 
                               boolean cancelableArg, 
                               AbstractView viewArg, 
                               int detailArg, 
                               int screenXArg, 
                               int screenYArg, 
                               int clientXArg, 
                               int clientYArg, 
                               boolean ctrlKeyArg, 
                               boolean altKeyArg, 
                               boolean shiftKeyArg, 
                               boolean metaKeyArg, 
                               short buttonArg, 
                               EventTarget relatedTargetArg);

    /**
     * The <code>initMouseEventNS</code> method is used to initialize the 
     * value of a <code>MouseEvent</code> created using the 
     * <code>DocumentEvent.createEvent</code> interface. This method may 
     * only be called before the <code>MouseEvent</code> has been dispatched 
     * via the <code>EventTarget.dispatchEvent</code> method, though it may 
     * be called multiple times during that phase if necessary. If called 
     * multiple times, the final invocation takes precedence.
     * @param namespaceURI  Specifies the namespace URI associated with this 
     *   event, or <code>null</code> if the application wish to have no 
     *   namespace. 
     * @param typeArg  Specifies the event type. 
     * @param canBubbleArg Specifies whether or not the event can bubble.
     * @param cancelableArg Specifies whether or not the event's default 
     *   action can be prevented.
     * @param viewArg Specifies the <code>Event</code>'s 
     *   <code>AbstractView</code>.
     * @param detailArg Specifies the <code>Event</code>'s mouse click count.
     * @param screenXArg Specifies the <code>Event</code>'s screen x 
     *   coordinate
     * @param screenYArg Specifies the <code>Event</code>'s screen y 
     *   coordinate
     * @param clientXArg Specifies the <code>Event</code>'s client x 
     *   coordinate
     * @param clientYArg Specifies the <code>Event</code>'s client y 
     *   coordinate
     * @param ctrlKeyArg Specifies whether or not control key was depressed 
     *   during the <code>Event</code>.
     * @param altKeyArg Specifies whether or not alt key was depressed during 
     *   the <code>Event</code>.
     * @param shiftKeyArg Specifies whether or not shift key was depressed 
     *   during the <code>Event</code>.
     * @param metaKeyArg Specifies whether or not meta key was depressed 
     *   during the <code>Event</code>.
     * @param buttonArg Specifies the <code>Event</code>'s mouse button.
     * @param relatedTargetArg Specifies the <code>Event</code>'s related 
     *   <code>EventTarget</code>.
     * @param altGraphKeyArg Specifies whether or not alt graph key was 
     *   depressed during the <code>Event</code>.
     * @since DOM Level 3
     */
    public void initMouseEventNS(String namespaceURI, 
                                 String typeArg, 
                                 boolean canBubbleArg, 
                                 boolean cancelableArg, 
                                 AbstractView viewArg, 
                                 int detailArg, 
                                 int screenXArg, 
                                 int screenYArg, 
                                 int clientXArg, 
                                 int clientYArg, 
                                 boolean ctrlKeyArg, 
                                 boolean altKeyArg, 
                                 boolean shiftKeyArg, 
                                 boolean metaKeyArg, 
                                 short buttonArg, 
                                 EventTarget relatedTargetArg, 
                                 boolean altGraphKeyArg);

    /**
     *  <code>true</code> if the alt-graph (Alt Gr) key modifier is activated. 
     *  Some operating systems simulate the alt-graph key modifier with the 
     * combinaison of alt and ctrl key modifiers. Implementations are 
     * encouraged to use this modifier instead. 
     * @since DOM Level 3
     */
    public boolean getAltGraphKey();

}
